package io.dyte.core.controllers.polls

import io.dyte.core.events.EventEmitter
import io.dyte.core.feat.DytePollMessage
import io.dyte.core.feat.DytePollOption
import io.dyte.core.listeners.DytePollEventsListener
import io.dyte.core.network.info.PollPermissions
import io.dyte.core.observability.DyteLogger

internal abstract class BasePollsController(private val permissions: PollPermissions) :
  EventEmitter<DytePollEventsListener>(), IPollsController {

  protected val logger = DyteLogger

  override var polls: ArrayList<DytePollMessage> = arrayListOf()

  abstract suspend fun sendVote(pollMessage: DytePollMessage, pollOption: DytePollOption)

  abstract suspend fun sendPoll(
    question: String,
    options: List<String>,
    anonymous: Boolean,
    hideVotes: Boolean,
  )

  override fun init() {
    setupEvents()
  }

  override suspend fun createPoll(
    question: String,
    options: List<String>,
    anonymous: Boolean,
    hideVotes: Boolean,
  ) {
    if (permissions.canCreate.not()) {
      logger.error("PollsController::createPoll::Not permitted")
      return
    }

    /*
     * Question cannot be empty or just whitespaces.
     * */
    if (question.isBlank()) {
      logger.error("PollsController::createPoll::Question is blank")
      return
    }

    /*
     * Must provide at least 2 options.
     * */
    if (options.size < 2) {
      logger.error("PollsController::createPoll::Number of options are less than 2")
      return
    }

    /*
     * Return if any of the option isBlank.
     * TODO: When introducing proper error, discuss with team about whether
     *  we should switch to a for-loop so that we can inform client about which option number is blank or invalid.
     * */
    if (options.count { it.isBlank() } > 0) {
      logger.error("PollsController::createPoll::One or more options provided are blank")
      return
    }

    sendPoll(question, options, anonymous, hideVotes)
  }

  fun onNewPoll(poll: DytePollMessage, emit: Boolean) {
    if (permissions.canView.not()) {
      logger.warn("PollsController::onNewPoll::not permitted to view poll")
      return
    }

    val existingPoll = polls.find { it.id == poll.id }
    if (existingPoll == null) {
      polls.add(poll)
      if (emit) {
        emitEvent { it.onNewPoll(poll) }
        emitEvent { it.onPollUpdates(polls) }
      }
    } else {
      val updatedPoll = existingPoll.copy(options = poll.options, voted = poll.voted)
      val existingIndex = polls.indexOf(existingPoll)
      polls.removeAt(existingIndex)
      polls.add(existingIndex, updatedPoll)
      if (emit) {
        emitEvent { it.onPollUpdates(polls) }
      }
    }
  }

  override suspend fun vote(pollMessage: DytePollMessage, pollOption: DytePollOption) {
    if (permissions.canVote.not()) {
      logger.error("PollsController::vote::not permitted")
      return
    }
    sendVote(pollMessage, pollOption)
  }
}

interface IPollsController {
  val polls: List<DytePollMessage>

  suspend fun loadPolls()

  suspend fun vote(pollMessage: DytePollMessage, pollOption: DytePollOption)

  suspend fun createPoll(
    question: String,
    options: List<String>,
    anonymous: Boolean,
    hideVotes: Boolean,
  )

  fun init()

  fun setupEvents()
}
