package io.dyte.core

import android.content.Context
import android.util.AttributeSet
import android.widget.LinearLayout
import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.models.DyteJoinedMeetingParticipant
import io.dyte.core.models.VideoScaleType
import io.dyte.core.observability.DyteLogger
import io.dyte.webrtc.VideoStreamTrack
import io.dyte.webrtc.WebRtc
import io.webrtc.EglBase
import io.webrtc.RendererCommon.ScalingType as WebRtcScalingType
import io.webrtc.SurfaceViewRenderer
import io.webrtc.VideoSink

class VideoView : LinearLayout {
  private var participant: DyteJoinedMeetingParticipant? = null
  private var videoRenderer: SurfaceViewRenderer? = null
  private var controllerContainer: IControllerContainer? = null

  private var isScreenShare: Boolean = false

  constructor(context: Context) : super(context) {
    init(context)
  }

  constructor(context: Context, attrs: AttributeSet?) : super(context, attrs) {
    init(context)
  }

  constructor(
    context: Context,
    attrs: AttributeSet?,
    defStyleAttr: Int,
  ) : super(context, attrs, defStyleAttr) {
    init(context)
  }

  private fun init(context: Context) {
    inflate(context, R.layout.view_video, this)
    videoRenderer = findViewById(R.id.video_renderer)
    addOnLayoutChangeListener { _, _, _, _, _, _, _, _, _ ->
      stopVideoRender()
      renderVideo()
    }
  }

  internal fun setup(
    controllerContainer: IControllerContainer,
    dyteParticipant: DyteJoinedMeetingParticipant,
    isScreenShare: Boolean,
    videoScaleType: VideoScaleType,
  ) {
    this.participant = dyteParticipant
    this.controllerContainer = controllerContainer
    this.isScreenShare = isScreenShare
    activate(videoScaleType)
  }

  private fun activate(videoScaleType: VideoScaleType) {
    videoRenderer?.setScalingType(videoScaleType.toWebRtcScalingType())
    try {
      videoRenderer?.init(WebRtc.rootEglBase.eglBaseContext as EglBase.Context, null)
    } catch (e: Exception) {
      DyteLogger.error("VideoView::activate::error", e)
    }
  }

  fun renderVideo() {
    try {
      val track = getVideoTrack()

      if (track != null) {
        videoRenderer?.visibility = VISIBLE
        track.addSinkCatching(videoRenderer!!)
      } else {
        videoRenderer?.visibility = GONE
      }
    } catch (e: Exception) {
      DyteLogger.error("VideoView::renderVideo::error", e)
    }
  }

  fun stopVideoRender() {
    try {
      val track: VideoStreamTrack? = getVideoTrack()

      if (track != null) {
        videoRenderer?.visibility = VISIBLE
        track.removeSinkCatching(videoRenderer!!)
      } else {
        videoRenderer?.visibility = GONE
      }
    } catch (e: Exception) {
      e.printStackTrace()
    }
  }

  fun release() {
    val track = getVideoTrack()
    track?.removeSinkCatching(videoRenderer!!)
    videoRenderer!!.release()
  }

  fun setZOrderMediaOverlay(isMediaOverlay: Boolean) {
    videoRenderer?.setZOrderMediaOverlay(isMediaOverlay)
  }

  private fun getVideoTrack(): VideoStreamTrack? {
    var track: VideoStreamTrack? = null
    if (isScreenShare && participant?.screenShareTrack != null) {
      track = participant?.screenShareTrack!!
    } else if (participant?.videoEnabled == true && participant?.videoTrack != null) {
      track = participant?.videoTrack
    }
    return track
  }

  private fun VideoStreamTrack.addSinkCatching(sink: VideoSink) {
    // runCatching as track may be disposed while activity was in pause mode
    runCatching { addSink(sink) }
  }

  private fun VideoStreamTrack.removeSinkCatching(sink: VideoSink) {
    // runCatching as track may be disposed while activity was in pause mode
    runCatching { removeSink(sink) }
  }

  private companion object {
    private fun VideoScaleType.toWebRtcScalingType(): WebRtcScalingType {
      return when (this) {
        VideoScaleType.SCALE_ASPECT_FIT -> WebRtcScalingType.SCALE_ASPECT_FIT
        VideoScaleType.SCALE_ASPECT_FILL -> WebRtcScalingType.SCALE_ASPECT_FILL
      }
    }
  }
}
