package io.dyte.core.events

import io.dyte.core.listeners.EventListener
import io.dyte.core.listeners.ExternalEventListener
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.DelicateCoroutinesApi
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.launch
import kotlinx.coroutines.newSingleThreadContext
import kotlinx.coroutines.runBlocking

/**
 * This class is used to handle events in the controller Extend the Controller class with this class
 */
internal open class EventEmitter<T : EventListener> {

  private val listeners = mutableSetOf<T>()
  private val listenersMainThread = mutableSetOf<T>()

  @OptIn(DelicateCoroutinesApi::class, ExperimentalCoroutinesApi::class)
  private val serialScope = CoroutineScope(newSingleThreadContext("GenericEmitter"))

  internal fun copy(e: EventEmitter<T>) {
    serialScope.launch {
      listeners.addAll(e.listeners)
      listenersMainThread.addAll(e.listenersMainThread)
    }
  }

  fun addListener(listener: T) {
    serialScope.launch { listeners.add(listener) }
  }

  fun removeListener(listener: T) {
    serialScope.launch { listeners.remove(listener) }
  }

  fun dispose() {
    listeners.clear()
    listenersMainThread.clear()
  }

  fun addMainThreadListener(listener: T) {
    serialScope.launch { listenersMainThread.add(listener) }
  }

  fun removeMainThreadListener(listener: T) {
    serialScope.launch { listenersMainThread.remove(listener) }
  }

  open fun emitEvent(event: (listener: T) -> Unit) {
    runBlocking(serialScope.coroutineContext) {
      listeners.iterator().forEach { event(it) }
      val mainScope = CoroutineScope(Dispatchers.Main)
      listenersMainThread.iterator().forEach { mainScope.launch { event(it) } }
    }
  }
}

/**
 * This class is used to handle events in the public API But since the source of truth is in the
 * controller, this will internally proxy to the controller emitter Set `emitterSuperClass` to the
 * controller emitter
 */
abstract class PublicAPIEmitter<T : ExternalEventListener> {

  internal abstract val emitterSuperClass: EventEmitter<T>

  fun addListener(listener: T) {
    this.emitterSuperClass.addListener(listener)
  }

  fun removeListener(listener: T) {
    this.emitterSuperClass.removeListener(listener)
  }

  fun addMainThreadListener(listener: T) {
    this.emitterSuperClass.addMainThreadListener(listener)
  }

  fun removeMainThreadListener(listener: T) {
    this.emitterSuperClass.removeMainThreadListener(listener)
  }
}
