package io.dyte.core.hive

import io.dyte.core.controllers.MetaController
import io.dyte.core.network.IApiClient
import io.dyte.core.observability.DyteLogger
import io.dyte.core.socket.socketservice.ISockratesSocketService
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import io.dyte.core.socket.socketservice.SocketServiceUtils
import io.dyte.core.utils.JsonUtils.encodeToByteString
import kotlinx.datetime.Clock
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonObject
import media.edge.GeoLocation
import media.edge.PeerJoinCompleteRequest
import media.edge.PeerJoinCompleteResponse
import media.edge.PeerJoinRequest
import socket.room.BroadcastMessage

internal interface HiveNodeSocketHandler {
  fun subscribeToBroadcasts(
    onPeerJoinedBroadcast: SocketServiceEventListener,
    onPeerProducerCreateBroadcast: SocketServiceEventListener,
    onPeerProducerToggleBroadcast: SocketServiceEventListener,
    onSelectedPeer: SocketServiceEventListener,
    onSelectedPeerDiff: SocketServiceEventListener,
    onPeerLeaveBroadcast: SocketServiceEventListener,
    onPeerProducerCloseBroadcast: SocketServiceEventListener,
    onGlobalPeerPinBroadcast: SocketServiceEventListener,
    onBroadcastMessage: SocketServiceEventListener,
    onMediaRoomTerminationBroadcast: SocketServiceEventListener,
  )

  suspend fun joinRoom(roomUuid: String = "", displayName: String, preJoined: Boolean = false)

  suspend fun completeJoinRoom(): PeerJoinCompleteResponse?

  suspend fun broadcastMessage(type: String, payload: JsonObject)

  suspend fun broadcastMessageToPeers(type: String, peerIds: List<String>, payload: JsonObject)
}

internal class DefaultHiveNodeSocketHandler(
  private val socketService: ISockratesSocketService,
  private val json: Json,
  private val metaController: MetaController,
  private val apiClient: IApiClient,
) : HiveNodeSocketHandler {
  override fun subscribeToBroadcasts(
    onPeerJoinedBroadcast: SocketServiceEventListener,
    onPeerProducerCreateBroadcast: SocketServiceEventListener,
    onPeerProducerToggleBroadcast: SocketServiceEventListener,
    onSelectedPeer: SocketServiceEventListener,
    onSelectedPeerDiff: SocketServiceEventListener,
    onPeerLeaveBroadcast: SocketServiceEventListener,
    onPeerProducerCloseBroadcast: SocketServiceEventListener,
    onGlobalPeerPinBroadcast: SocketServiceEventListener,
    onBroadcastMessage: SocketServiceEventListener,
    onMediaRoomTerminationBroadcast: SocketServiceEventListener,
  ) {
    socketService.subscribe(
      SocketServiceUtils.MediaEvent.PEER_JOINED_BROADCAST.id,
      onPeerJoinedBroadcast,
    )

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.PEER_PRODUCER_CREATE_BROADCAST.id,
      onPeerProducerCreateBroadcast,
    )

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.PEER_PRODUCER_TOGGLE_BROADCAST.id,
      onPeerProducerToggleBroadcast,
    )

    socketService.subscribe(SocketServiceUtils.MediaEvent.SELECTED_PEER.id, onSelectedPeer)

    socketService.subscribe(SocketServiceUtils.MediaEvent.SELECTED_PEER_DIFF.id, onSelectedPeerDiff)

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.PEER_LEAVE_BROADCAST.id,
      onPeerLeaveBroadcast,
    )

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.PEER_PRODUCER_CLOSE_BROADCAST.id,
      onPeerProducerCloseBroadcast,
    )

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.GLOBAL_PEER_PIN_BROADCAST.id,
      onGlobalPeerPinBroadcast,
    )

    socketService.subscribe(SocketServiceUtils.RoomEvent.BROADCAST_MESSAGE.id, onBroadcastMessage)

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.MEDIA_ROOM_TERMINATION_BROADCAST.id,
      onMediaRoomTerminationBroadcast,
    )
  }

  override suspend fun joinRoom(roomUuid: String, displayName: String, preJoined: Boolean) {
    val ipDetails = apiClient.getIpDetails(metaController.getPeerId())
    val location =
      if (ipDetails != null) {
        val coordinates = ipDetails.location.split(",")
        if (coordinates.size != 2) {
          null
        } else {
          GeoLocation(latitude = coordinates[0].toFloat(), longitude = coordinates[1].toFloat())
        }
      } else {
        null
      }
    val req =
      PeerJoinRequest(
        room_uuid = roomUuid,
        display_name = displayName,
        prejoined = preJoined,
        location = location,
      )

    try {
      // note : keep this as request response only.
      val response =
        socketService.requestResponse(
          event = SocketServiceUtils.MediaEvent.JOIN_ROOM.id,
          payload = PeerJoinRequest.ADAPTER.encode(req),
        )
      DyteLogger.debug("HiveSocketHandler: joined to socket room")
    } catch (e: Exception) {
      DyteLogger.error("HiveSocketHandler: Error on join room: ${e.message}")
    }
  }

  override suspend fun completeJoinRoom(): PeerJoinCompleteResponse? {
    return try {
      val req = PeerJoinCompleteRequest()

      PeerJoinCompleteResponse.ADAPTER.decode(
        socketService.requestResponse(
          event = SocketServiceUtils.MediaEvent.SELF_JOIN_COMPLETE.id,
          payload = PeerJoinCompleteRequest.ADAPTER.encode(req),
        )!!
      )
    } catch (e: Exception) {
      DyteLogger.error("HiveSocketHandler: Error on complete join room: ${e.message}")
      null
    }
  }

  override suspend fun broadcastMessage(type: String, payload: JsonObject) {
    try {
      val message =
        BroadcastMessage(
          type = type,
          payload = json.encodeToByteString(payload),
          timestamp = Clock.System.now().toEpochMilliseconds(),
        )
      socketService.send(
        event = SocketServiceUtils.RoomEvent.BROADCAST_MESSAGE.id,
        payload = BroadcastMessage.ADAPTER.encode(message),
      )
    } catch (e: Exception) {
      DyteLogger.error("HiveSocketHandler: Error on broadcast message: ${e.message}")
    }
  }

  override suspend fun broadcastMessageToPeers(
    type: String,
    peerIds: List<String>,
    payload: JsonObject,
  ) {
    try {
      val message =
        BroadcastMessage(
          type = type,
          payload = json.encodeToByteString(payload),
          timestamp = Clock.System.now().toEpochMilliseconds(),
          peer_ids = peerIds,
        )
      socketService.send(
        event = SocketServiceUtils.RoomEvent.BROADCAST_TO_PEERS.id,
        payload = BroadcastMessage.ADAPTER.encode(message),
      )
    } catch (e: Exception) {
      DyteLogger.error("HiveSocketHandler: Error on broadcast message to peers: ${e.message}")
    }
  }
}
