package io.dyte.core.controllers

import io.dyte.core.Utils.decodeAuthToken
import io.dyte.core.models.ActiveTabType
import io.dyte.core.models.DyteMeetingState
import io.dyte.core.models.DyteMeetingType
import io.dyte.core.network.info.DyteDesignToken
import io.dyte.core.network.info.ParticipantInfo
import io.dyte.core.observability.DyteLogger
import io.dyte.core.spotlight.ActiveTab
import io.dyte.core.spotlight.SpotlightController
import io.dyte.core.utils.PeerIdGenerator
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.launch

/**
 * Meta controller
 *
 * Contains meta details about the meeting
 *
 * @constructor Create empty Meta controller
 */
internal class MetaController(
  private val meetingInfo: MeetingInfo,
  private val controllerContainer: IControllerContainer,
  private val peerIdGenerator: PeerIdGenerator,
  private val scope: CoroutineScope,
) : IMetaController {

  internal var _meetingState: DyteMeetingState = DyteMeetingState.NotInitialised
  override val meetingState: DyteMeetingState
    get() = _meetingState

  private lateinit var participantInfo: ParticipantInfo
  private var meetingUserPeerId: String = peerIdGenerator.generate()
  private lateinit var roomUuid: String
  private var useHive: Boolean? = null
  private lateinit var designToken: DyteDesignToken

  internal val selfActiveTab: ActiveTab?
    get() = spotlightController?.spotlitActiveTab?.value

  private var spotlightController: SpotlightController? = null

  fun init() {
    meetingInfo.meetingId = decodeAuthToken(meetingInfo.authToken)
    meetingInfo.isV2Meeting = meetingInfo.meetingId != null
    //    setupRoomJoinedListeners()
  }

  fun setSpotlightController(spotlightController: SpotlightController) {
    this.spotlightController = spotlightController
    this.spotlightController?.let {
      it.init()
      it.spotlitActiveTab.onEach { tabUpdate -> handleActiveTabUpdate(tabUpdate) }.launchIn(scope)
    }
  }

  override fun getDisplayName(): String {
    return participantInfo.name
  }

  override fun getRoomName(): String {
    return meetingInfo.getRoomName()
  }

  override fun getAuthToken(): String {
    return meetingInfo.authToken
  }

  override fun getPeerId(): String {
    return meetingUserPeerId
  }

  override fun getOrgId(): String {
    return if (::participantInfo.isInitialized) participantInfo.organizationId else ""
  }

  override fun getMeetingTitle(): String {
    return meetingInfo.meetingTitle ?: ""
  }

  override fun getMeetingStatedTimestamp(): String {
    return meetingInfo.meetingStartedAt ?: ""
  }

  override fun setMeetingTitle(title: String) {
    meetingInfo.meetingTitle = title
  }

  override fun setMeetingStartedTimestamp(timestamp: String) {
    meetingInfo.meetingStartedAt = timestamp
  }

  override fun getMeetingId(): String {
    return meetingInfo.getRoomName()
  }

  override fun isV2AuthToken(): Boolean {
    return meetingInfo.isV2Meeting
  }

  override fun setParticipantInfo(participantInfo: ParticipantInfo) {
    this.participantInfo = participantInfo
  }

  override fun isWebinar(): Boolean {
    return participantInfo.presetInfo.config.viewType == "WEBINAR"
  }

  override fun isGroupCall(): Boolean {
    return participantInfo.presetInfo.config.viewType == "GROUP_CALL"
  }

  override fun getMeetingConfig(): MeetingConfig {
    return this.meetingInfo.config
  }

  override fun setRoomUuid(roomUuid: String) {
    this.roomUuid = roomUuid
  }

  override fun getRoomUuid(): String {
    return roomUuid
  }

  override fun getIsHive(): Boolean {
    return useHive!!
  }

  override fun setIsHive(isHive: Boolean) {
    useHive = isHive
  }

  override fun getMeetingType(): DyteMeetingType {
    return DyteMeetingType.fromViewTypeString(participantInfo.presetInfo.config.viewType)
  }

  override fun getDesignToken(): DyteDesignToken {
    return designToken
  }

  override fun setDesignToken(designToken: DyteDesignToken) {
    this.designToken = designToken
  }

  override fun refreshPeerId(): String {
    val newPeerId = peerIdGenerator.generate()
    meetingUserPeerId = newPeerId
    return newPeerId
  }

  //  private fun setupRoomJoinedListeners() {
  //    controllerContainer.internalEventEmitter.addListener(
  //      object : InternalEvents {
  //        override fun onRoomJoined(webSocketJoinRoomModel: WebSocketJoinRoomModel) {
  //          super.onRoomJoined(webSocketJoinRoomModel)
  //          setUpSpotlightSocketListner()
  //        }
  //      }
  //    )
  //  }

  override fun syncTab(id: String, tabType: ActiveTabType) {
    DyteLogger.info("DyteMeta::syncTab::$id for ${tabType::class.simpleName}")
    scope.launch { spotlightController?.spotlightActiveTab(id, tabType) }
  }

  // TODO: decide if we should give onActiveTabChange callback if spotlighter is self
  private fun handleActiveTabUpdate(activeTab: ActiveTab?) {
    if (activeTab == null) {
      return
    }
    if (controllerContainer.selfController.getSelf().roomJoined) {
      controllerContainer.eventController.triggerEvent(DyteEventType.OnActiveTabUpdate(activeTab))
    }
  }
}

internal data class MeetingInfo(
  val authToken: String,
  val config: MeetingConfig,
  val baseUrl: String,
) {
  internal var isV2Meeting: Boolean = false
  internal var meetingId: String? = null
  internal var roomName: String? = null
  internal var viewType: String = "GROUP_CALL"
  internal var meetingTitle: String? = null
  internal var meetingStartedAt: String? = null

  fun getRoomName(): String {
    return (if (isV2Meeting) meetingId else this.roomName) ?: ""
  }
}

data class MeetingConfig(val enableAudio: Boolean, val enableVideo: Boolean)

internal interface IMetaController {
  val meetingState: DyteMeetingState

  fun getPeerId(): String

  fun getDisplayName(): String

  fun getRoomName(): String

  fun getAuthToken(): String

  fun getOrgId(): String

  fun setMeetingTitle(title: String)

  fun setMeetingStartedTimestamp(timestamp: String)

  fun getMeetingTitle(): String

  fun getMeetingStatedTimestamp(): String

  fun getMeetingId(): String

  fun getMeetingConfig(): MeetingConfig

  fun isWebinar(): Boolean

  fun isGroupCall(): Boolean

  fun setParticipantInfo(participantInfo: ParticipantInfo)

  fun isV2AuthToken(): Boolean

  fun setRoomUuid(roomUuid: String)

  fun getRoomUuid(): String

  fun getIsHive(): Boolean

  fun setIsHive(isHive: Boolean)

  fun getMeetingType(): DyteMeetingType

  fun getDesignToken(): DyteDesignToken

  fun setDesignToken(designToken: DyteDesignToken)

  /**
   * Generates a new peerId for local user.
   *
   * **NOTE**: This is needed as a part of a workaround that we have done to make the socket-service
   * work properly after the reconnection.
   *
   * @return new peerId
   */
  fun refreshPeerId(): String

  fun syncTab(id: String, tabType: ActiveTabType)
}
