package io.dyte.core.platform

import android.content.Context
import io.dyte.core.feat.DyteMeetingParticipant
import io.dyte.core.models.DyteJoinedMeetingParticipant
import io.dyte.core.observability.DyteLogger
import kotlinx.coroutines.MainScope
import kotlinx.coroutines.launch

actual typealias VideoView = io.dyte.core.VideoView

class DyteAndroidVideoUtils
internal constructor(private val utilsProvider: IDytePlatformUtilsProvider) : IDyteVideoUtils {
  private var useCache = true

  private val videoViews = hashMapOf<String, io.dyte.core.VideoView>()
  private val screenShareViews = hashMapOf<String, io.dyte.core.VideoView>()

  override fun getVideoView(participant: DyteJoinedMeetingParticipant): io.dyte.core.VideoView {
    return if (useCache && videoViews.containsKey(participant.id)) {
      val videoView = videoViews.getValue(participant.id)
      videoView
    } else {
      val oldView = videoViews[participant.id]

      val videoView =
        io.dyte.core.VideoView(
          utilsProvider
            .getControllerContainer()
            .platformUtilsProvider
            .getPlatformUtils()
            .getActivity() as Context
        )
      videoView.setup(utilsProvider.getControllerContainer(), participant, false)

      if (useCache) {
        videoViews[participant.id] = videoView
        if (oldView != null) destroyVideoView(oldView)
      }

      videoView
    }
  }

  override fun getSelfPreview(): VideoView {
    val participant = utilsProvider.getControllerContainer().selfController.getSelf()
    val videoView =
      io.dyte.core.VideoView(
        utilsProvider
          .getControllerContainer()
          .platformUtilsProvider
          .getPlatformUtils()
          .getActivity() as Context
      )
    videoView.setup(utilsProvider.getControllerContainer(), participant, false)

    if (useCache) {
      val existing = videoViews["self${participant.id}"]
      if (existing != null) destroyVideoView(existing)
      videoViews["self${participant.id}"] = videoView
    }

    return videoView
  }

  override fun getScreenShareView(participant: DyteJoinedMeetingParticipant): VideoView {
    return if (useCache && screenShareViews.containsKey(participant.id)) {
      val screenShareView = screenShareViews.getValue(participant.id)
      screenShareView
    } else {
      val oldView = screenShareViews[participant.id]

      val videoView =
        io.dyte.core.VideoView(
          utilsProvider
            .getControllerContainer()
            .platformUtilsProvider
            .getPlatformUtils()
            .getActivity() as Context
        )
      videoView.setup(utilsProvider.getControllerContainer(), participant, true)

      if (useCache) {
        screenShareViews[participant.id] = videoView
        if (oldView != null) destroyVideoView(oldView)
      }

      videoView
    }
  }

  override fun destroyView(participant: DyteMeetingParticipant) {
    val videoView = videoViews[participant.id]
    if (videoView != null) destroyVideoView(videoView)
    videoViews.remove(participant.id)
  }

  override fun destroyScreenShare(participant: DyteMeetingParticipant) {
    val screenShareView = screenShareViews[participant.id]
    if (screenShareView != null) destroyVideoView(screenShareView)
    screenShareViews.remove(participant.id)
  }

  private fun destroyVideoView(view: VideoView) {
    try {
      MainScope().launch { view.release() }
    } catch (e: Exception) {
      e.printStackTrace()
    }
  }

  override fun destroyAll() {
    DyteLogger.debug("video_utils::destroyAll")
    videoViews.values.forEach { destroyVideoView(it) }
    videoViews.clear()

    screenShareViews.values.forEach { destroyVideoView(it) }
    screenShareViews.clear()
  }

  override fun disableCache() {
    DyteLogger.info("video_utils::disableCache")
    useCache = false
    destroyAll()
  }

  override fun enableCache() {
    DyteLogger.info("video_utils::enableCache")
    useCache = true
    destroyAll()
  }
}
