package io.dyte.core.spotlight

import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.models.ActiveTabType
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.observability.DyteLogger
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.jsonPrimitive
import kotlinx.serialization.json.put

internal abstract class SpotlightController(
  protected val controllerContainer: IControllerContainer,
  protected val scope: CoroutineScope,
) {
  protected val _spotlitActiveTab = MutableStateFlow<ActiveTab?>(null)
  val spotlitActiveTab: StateFlow<ActiveTab?>
    get() = _spotlitActiveTab

  protected val selfParticipant: DyteSelfParticipant
    get() = controllerContainer.selfController.getSelf()

  protected val selfCanSpotLight: Boolean
    get() = controllerContainer.selfController.getSelf().permissions.miscellaneous.canSpotLight

  protected val logger = DyteLogger

  open fun init() {}

  internal suspend fun spotlightActiveTab(id: String, tabType: ActiveTabType) {
    if (!selfCanSpotLight) {
      logger.error("Spotlight::spotlightActiveTab::user not authorised")
      return
    }

    try {
      spotlightActiveTabInternal(id, tabType)
    } catch (e: Exception) {
      logger.error("Spotlight::spotlightActiveTab::failed ${e.message}")
    }
  }

  protected suspend fun spotlightActiveTabToPeer(
    peerId: String,
    tabId: String,
    tabType: ActiveTabType,
  ) {
    if (!selfCanSpotLight) {
      logger.error("Spotlight::spotlightActiveTabToPeer::user not authorised")
      return
    }

    try {
      spotlightActiveTabToPeerInternal(peerId, tabId, tabType)
    } catch (e: Exception) {
      logger.error("Spotlight::spotlightActiveTabToPeer::failed ${e.message}")
    }
  }

  protected abstract suspend fun spotlightActiveTabInternal(id: String, tabType: ActiveTabType)

  protected abstract suspend fun spotlightActiveTabToPeerInternal(
    peerId: String,
    tabId: String,
    tabType: ActiveTabType,
  )

  protected fun handleSpotlightAssertion(spotlightAssertion: JsonObject) {
    try {
      val spotlighterUserId =
        spotlightAssertion["userId"]?.jsonPrimitive?.content
          ?: kotlin.run {
            logger.warn("SpotlightController::handle spotlight assertion::userId not found")
            "" // patching userId as blank
          }
      val currentTab =
        spotlightAssertion["currentTab"]?.jsonObject ?: error("key currentTab not found")
      val tabId =
        currentTab["id"]?.jsonPrimitive?.content ?: error("key id not found in currentTab object")
      val tabTypeString =
        currentTab["type"]?.jsonPrimitive?.content
          ?: error("key type not found in currentTab object")
      val activeTabType =
        getActiveTab(tabTypeString) ?: error("invalid active tab type $tabTypeString")

      val receivedActiveTab = ActiveTab(tabId, activeTabType, spotlighterUserId)

      logger.info(
        "Spotlight::assertion received",
        mapOf("spotlight" to receivedActiveTab.toString()),
      )

      _spotlitActiveTab.value = receivedActiveTab
    } catch (e: Exception) {
      logger.warn(
        "SpotlightController::handle spotlight assertion failed::${e.message}",
        mapOf("assertion" to spotlightAssertion.toString()),
      )
    }
  }

  protected companion object {
    const val MESSAGE_TYPE_SPOTLIGHT = "spotlight"

    fun ActiveTabType.getValue(): String {
      return when (this) {
        ActiveTabType.SCREENSHARE -> "screenshare"
        ActiveTabType.PLUGIN -> "plugin"
      }
    }

    fun getCurrentTabJsonObject(id: String, tabType: ActiveTabType): JsonObject = buildJsonObject {
      put("id", id)
      put("type", tabType.getValue())
    }

    fun getActiveTab(value: String): ActiveTabType? {
      return when (value) {
        "screenshare" -> ActiveTabType.SCREENSHARE
        "plugin" -> ActiveTabType.PLUGIN
        else -> null
      }
    }
  }
}

internal data class SpotlitActiveTab(val spotlighterUserId: String, val activeTab: ActiveTab)
