package io.dyte.core.spotlight

import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.models.ActiveTabType
import io.dyte.core.socket.SocketMessageEventListener
import io.dyte.core.socket.events.InboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.payloadmodel.InboundMeetingEvent
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketMeetingPeerUser
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketRoomMessage
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.jsonPrimitive
import kotlinx.serialization.json.put

internal class RoomNodeSpotlightController(
  controllerContainer: IControllerContainer,
  scope: CoroutineScope,
) : SpotlightController(controllerContainer, scope) {
  private val socketEventListener =
    object : SocketMessageEventListener {
      override suspend fun onMessageEvent(event: InboundMeetingEvent) {
        scope.launch { handleSocketEvent(event) }
      }
    }

  override fun init() {
    super.init()
    setUpActiveTabSocketListeners(selfCanSpotLight)
  }

  private fun setUpActiveTabSocketListeners(selfCanSpotLight: Boolean) {
    controllerContainer.roomNodeSocketService.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_ROOM_MESSAGE,
      socketEventListener,
    )

    controllerContainer.roomNodeSocketService.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PEER_MESSAGE,
      socketEventListener,
    )

    if (selfCanSpotLight) {
      controllerContainer.roomNodeSocketService.addMessageEventListener(
        InboundMeetingEventType.WEB_SOCKET_PEER_JOINED,
        socketEventListener,
      )
    }
  }

  override suspend fun spotlightActiveTabInternal(id: String, tabType: ActiveTabType) {
    val spotlightToRoomPayload = buildJsonObject {
      put("roomMessageType", MESSAGE_TYPE_SPOTLIGHT)
      put("userId", selfParticipant.userId)
      put("currentTab", getCurrentTabJsonObject(id, tabType))
    }
    controllerContainer.roomNodeSocketService.sendMessage(
      OutboundMeetingEventType.ROOM_MESSAGE,
      spotlightToRoomPayload,
    )
  }

  override suspend fun spotlightActiveTabToPeerInternal(
    peerId: String,
    tabId: String,
    tabType: ActiveTabType,
  ) {
    val spotlightToRoomPayload = buildJsonObject {
      put("roomMessageType", MESSAGE_TYPE_SPOTLIGHT)
      put("userId", selfParticipant.userId)
      put("currentTab", getCurrentTabJsonObject(tabId, tabType))
      put("peerId", peerId)
    }
    controllerContainer.roomNodeSocketService.sendMessage(
      OutboundMeetingEventType.PEER_MESSAGE,
      spotlightToRoomPayload,
    )
  }

  private suspend fun handleSocketEvent(event: InboundMeetingEvent) {
    when (event.eventType) {
      InboundMeetingEventType.WEB_SOCKET_ROOM_MESSAGE,
      InboundMeetingEventType.WEB_SOCKET_PEER_MESSAGE -> {
        handleSpotlightBroadcastMessage(event.payload as WebSocketRoomMessage)
      }
      InboundMeetingEventType.WEB_SOCKET_PEER_JOINED -> {
        handlePeerJoinedEvent(event.payload as WebSocketMeetingPeerUser)
      }
      else -> {}
    }
  }

  private fun handleSpotlightBroadcastMessage(webSocketRoomMessage: WebSocketRoomMessage) {
    val spotlightAssertionJsonObject =
      try {
        val roomMessagePayload = webSocketRoomMessage.payload.jsonObject
        val roomMessageType =
          roomMessagePayload["roomMessageType"]?.jsonPrimitive?.content ?: return
        if (roomMessageType != MESSAGE_TYPE_SPOTLIGHT) {
          return
        }
        roomMessagePayload
      } catch (e: Exception) {
        return
      }
    handleSpotlightAssertion(spotlightAssertionJsonObject)
  }

  private suspend fun handlePeerJoinedEvent(webSocketMeetingPeerUser: WebSocketMeetingPeerUser) {
    if (!controllerContainer.selfController.getSelf().roomJoined) {
      return
    }

    val spotlitActiveTab = spotlitActiveTab.value ?: return
    val shouldBroadcastSpotlitActiveTab =
      webSocketMeetingPeerUser.id != selfParticipant.id &&
        spotlitActiveTab.userId == selfParticipant.userId

    if (shouldBroadcastSpotlitActiveTab) {
      spotlightActiveTabToPeer(
        webSocketMeetingPeerUser.id,
        spotlitActiveTab.id,
        spotlitActiveTab.type,
      )
    }
  }
}
