package io.dyte.core.spotlight

import io.dyte.core.controllers.HiveController
import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.models.ActiveTabType
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import io.dyte.core.socket.socketservice.SocketServiceUtils
import io.dyte.core.utils.JsonUtils.decodeFromByteString
import io.dyte.media.utils.sdp.json
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.put
import media.edge.PeerJoinBroadcastResponse
import socket.room.BroadcastMessage

internal class HiveSpotlightController(
  controllerContainer: IControllerContainer,
  scope: CoroutineScope,
) : SpotlightController(controllerContainer, scope) {

  private val socketEventListener =
    object : SocketServiceEventListener {
      override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
        scope.launch {
          if (payload == null) {
            return@launch
          }
          handleSocketEvent(event, payload)
        }
      }
    }

  override fun init() {
    super.init()
    setUpActiveTabSocketListeners(selfCanSpotLight)
  }

  private fun setUpActiveTabSocketListeners(selfCanSpotLight: Boolean) {
    controllerContainer.sockratesSocketService.subscribe(
      SocketServiceUtils.RoomEvent.BROADCAST_MESSAGE.id,
      socketEventListener,
    )

    controllerContainer.sockratesSocketService.subscribe(
      SocketServiceUtils.RoomEvent.BROADCAST_TO_PEERS.id,
      socketEventListener,
    )

    if (selfCanSpotLight) {
      controllerContainer.sockratesSocketService.subscribe(
        SocketServiceUtils.MediaEvent.PEER_JOINED_BROADCAST.id,
        socketEventListener,
      )
    }
  }

  override suspend fun spotlightActiveTabInternal(id: String, tabType: ActiveTabType) {
    val payload = buildJsonObject {
      put("userId", selfParticipant.userId)
      put("currentTab", getCurrentTabJsonObject(id, tabType))
    }
    (controllerContainer.roomNodeController as HiveController).broadcastMessage(
      type = MESSAGE_TYPE_SPOTLIGHT,
      payload = payload,
    )
  }

  private suspend fun handleSocketEvent(event: Int, payload: ByteArray) {
    when (event) {
      SocketServiceUtils.RoomEvent.BROADCAST_MESSAGE.id,
      SocketServiceUtils.RoomEvent.BROADCAST_TO_PEERS.id -> {
        handleSpotlightBroadcastMessage(payload)
      }
      SocketServiceUtils.MediaEvent.PEER_JOINED_BROADCAST.id -> {
        handlePeerJoinedBroadcast(payload)
      }
      else -> {}
    }
  }

  private fun handleSpotlightBroadcastMessage(payload: ByteArray) {
    val broadcastMessage =
      try {
        BroadcastMessage.ADAPTER.decode(payload)
      } catch (e: Exception) {
        return
      }

    if (broadcastMessage.type != MESSAGE_TYPE_SPOTLIGHT) {
      return
    }

    val spotlightAssertionJsonObject =
      try {
        val payloadJsonElement = json.decodeFromByteString(broadcastMessage.payload)
        payloadJsonElement.jsonObject
      } catch (e: Exception) {
        logger.warn(
          "SpotlightController::invalid spotlight assertion received::${e.message}",
          mapOf("room_message" to broadcastMessage.toString()),
        )
        return
      }
    handleSpotlightAssertion(spotlightAssertionJsonObject)
  }

  private suspend fun handlePeerJoinedBroadcast(payload: ByteArray) {
    if (!controllerContainer.selfController.getSelf().roomJoined) {
      return
    }

    val participant =
      try {
        val peerJoinBroadcastResponse = PeerJoinBroadcastResponse.ADAPTER.decode(payload)
        peerJoinBroadcastResponse.participant ?: return
      } catch (e: Exception) {
        return
      }

    val spotlitActiveTab = spotlitActiveTab.value ?: return
    val shouldBroadcastSpotlitActiveTab =
      participant.peer_id != selfParticipant.id && spotlitActiveTab.userId == selfParticipant.userId

    if (shouldBroadcastSpotlitActiveTab) {
      spotlightActiveTabToPeer(participant.peer_id, spotlitActiveTab.id, spotlitActiveTab.type)
    }
  }

  override suspend fun spotlightActiveTabToPeerInternal(
    peerId: String,
    tabId: String,
    tabType: ActiveTabType,
  ) {
    val payload = buildJsonObject {
      put("userId", selfParticipant.userId)
      put("currentTab", getCurrentTabJsonObject(tabId, tabType))
    }
    (controllerContainer.roomNodeController as HiveController).broadcastMessageToPeers(
      type = MESSAGE_TYPE_SPOTLIGHT,
      peerIds = listOf(peerId),
      payload = payload,
    )
  }
}
