package io.dyte.core.media

import io.dyte.core.controllers.SelfController
import io.dyte.core.observability.DyteLogger
import io.dyte.webrtc.*
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking

internal open class DyteCommonMediaUtils(
  forceSoftwareEncoder: Boolean,
  forceSoftwareDecoder: Boolean,
) : IDyteCommonMediaUtils {
  private var platform: String = "android"

  init {
    try {
      WebRtc.initialize(
        forceSoftwareEncoder = forceSoftwareEncoder,
        forceSoftwareDecoder = forceSoftwareDecoder,
      )
    } catch (e: Exception) {
      e.printStackTrace()
    }
  }

  private var audioTrack: AudioStreamTrack? = null
  private var videoTrack: VideoStreamTrack? = null
  private var screenshareTrack: VideoStreamTrack? = null

  private var videoMediaStream: MediaStream? = null
  private var audioMediaStream: MediaStream? = null
  private var screenshareStream: MediaStream? = null

  private var cameraFacingUser = true

  override fun getVideoStream(): MediaStream? {
    if (
      videoMediaStream != null &&
        videoMediaStream?.videoTracks?.first()?.readyState is MediaStreamTrackState.Live
    ) {
      DyteLogger.debug("DyteCommonMediaUtils::getVideoStream::getUserMedia::stream_exists")
      return videoMediaStream!!
    }

    DyteLogger.debug("DyteCommonMediaUtils::getVideoStream::getUserMedia")

    val camExists = runBlocking { MediaDevices.enumerateDevices().isNotEmpty() }

    if (camExists) {
      runBlocking {
        try {
          if (platform == "iOS") {
            videoMediaStream =
              DyteUserMedia.getUserMedia {
                video {
                  facingMode(if (cameraFacingUser) FacingMode.User else FacingMode.Environment)
                }
              }
          } else {
            videoMediaStream =
              DyteUserMedia.getUserMedia {
                video {
                  facingMode(if (cameraFacingUser) FacingMode.User else FacingMode.Environment)
                  height(640)
                  width(480)
                  frameRate(30.0)
                }
              }
          }
          DyteLogger.debug("DyteCommonMediaUtils::getVideoStream::getUserMedia::success")
        } catch (e: Exception) {
          DyteLogger.error("DyteCommonMediaUtils::getVideoStream::getUserMedia::error", e)
        }
      }
    }

    return videoMediaStream
  }

  override fun getAudioStream(): MediaStream? {
    if (
      audioMediaStream != null &&
        audioMediaStream?.audioTracks?.first()?.readyState is MediaStreamTrackState.Live
    ) {
      DyteLogger.debug("DyteCommonMediaUtils::getAudioStream::stream_exists")
      return audioMediaStream!!
    }

    DyteLogger.debug("DyteCommonMediaUtils::getAudioStream::getUserMedia")
    runBlocking {
      try {
        audioMediaStream = DyteUserMedia.getUserMedia { audio(true) }
        DyteLogger.debug("DyteCommonMediaUtils::getAudioStream::getUserMedia::success")
      } catch (e: Exception) {
        DyteLogger.error("DyteCommonMediaUtils::getAudioStream::getUserMedia::error", e)
      }
    }

    return audioMediaStream
  }

  var _isSSInit = false

  override fun getScreenshareStream(
    ssStartedCallback: suspend (VideoStreamTrack, MediaStream) -> Unit,
    coroutineScope: CoroutineScope,
    selfController: SelfController,
  ) {
    if (!_isSSInit) {
      _isSSInit = true
      DyteLogger.debug("DyteCommonMediaUtils::getScreenshareStream::getUserMedia")

      val gotScreenShareStreamCallback: (MediaStream) -> Unit = { stream ->
        DyteLogger.info(
          "DyteCommonMediaUtils::getScreenshareStream::getUserMedia::gotScreenShareStream"
        )
        screenshareStream = stream
        coroutineScope.launch {
          try {
            ssStartedCallback.invoke(screenshareStream?.videoTracks?.first()!!, screenshareStream!!)
            _isSSInit = false
          } catch (e: Exception) {
            DyteLogger.error(
              "DyteCommonMediaUtils::failed to invoke screenshare started callback",
              e,
            )
            _isSSInit = false
            selfController.emitEvent { it.onScreenShareStartFailed() }
          }
        }
      }
      try {
        DyteUserMedia.getDisplayMedia(gotScreenShareStreamCallback, selfController)
        DyteLogger.debug("DyteCommonMediaUtils::getScreenshareStream::getUserMedia::success")
      } catch (e: Exception) {
        DyteLogger.error("DyteCommonMediaUtils::getScreenshareStream::getUserMedia::error", e)
      }
    } else {
      DyteLogger.debug(
        "DyteCommonMediaUtils::getScreenshareStream::screenshare already initialized"
      )
    }
  }

  override fun createAudioTrack(): AudioStreamTrack? {
    DyteLogger.info("DyteCommonMediaUtils::createAudioTrack")
    audioTrack = getAudioStream()?.audioTracks?.first()
    return audioTrack
  }

  override fun createVideoTrack(): VideoStreamTrack? {
    DyteLogger.info("DyteCommonMediaUtils::createVideoTrack")
    videoTrack = getVideoStream()?.videoTracks?.first()
    return videoTrack
  }

  //  override fun createScreenshareTrack(ssStartedCallback: suspend (VideoStreamTrack) -> Unit):
  // VideoStreamTrack? {
  //    screenshareTrack = getScreenshareStream()?.videoTracks?.first()
  //    return screenshareTrack
  //  }

  override fun stopVideo() {
    videoTrack?.enabled = false
    DyteLogger.info("DyteCommonMediaUtils | video stopped")
  }

  override fun stopAudio() {
    audioTrack?.enabled = false
    DyteLogger.info("DyteCommonMediaUtils | audio stopped")
  }

  override fun cleanupScreenshare() {
    screenshareStream = null
    DyteUserMedia.cleanupDisplay()
  }

  override fun resumeVideo() {
    videoTrack?.enabled = true
    DyteLogger.debug("DyteCommonMediaUtils | video resumed")
  }

  override fun switchCamera(deviceId: String?) {
    runBlocking {
      if (videoTrack?.enabled == true) {
        DyteLogger.info("DyteCommonMediaUtils | switchCamera | switching camera")
        videoTrack?.switchCamera(deviceId)
        cameraFacingUser = !cameraFacingUser
      } else {
        DyteLogger.info(
          "DyteCommonMediaUtils | switchCamera | video track is not enabled, not switching camera"
        )
      }
    }
  }

  override fun dispose() {
    // WebRtc.dispose()
    try {
      stopVideo()
      stopAudio()

      videoMediaStream?.release()
      audioMediaStream?.release()
      screenshareStream?.release()
    } catch (e: Exception) {
      DyteLogger.warn("DyteCommonMediaUtils | dispose mediaStream release failed", e)
    }
    _isSSInit = false
    videoMediaStream = null
    audioMediaStream = null
    screenshareStream = null
    DyteLogger.info("DyteCommonMediaUtils | audio/video streams disposed")
  }

  override fun onCameraClosed() {
    videoTrack = null
  }

  override fun getVideoTrack(): VideoStreamTrack? {
    return if (videoTrack == null || videoTrack?.readyState is MediaStreamTrackState.Ended)
      createVideoTrack()
    else videoTrack
  }

  override fun setPlatform(platform: String) {
    this.platform = platform
  }
}

internal interface IDyteCommonMediaUtils {

  fun getVideoStream(): MediaStream?

  fun getAudioStream(): MediaStream?
  /**
   * Switch camera
   *
   * toggles between front camera and back camera
   */
  fun switchCamera(deviceId: String? = null)

  /**
   * Create audio track
   *
   * @return instance of AudioTrack from webrtc
   */
  fun createAudioTrack(): AudioStreamTrack?

  fun stopAudio()

  /**
   * Create video track
   *
   * @return instance of videoTrack from webrtc
   */
  fun createVideoTrack(): VideoStreamTrack?

  fun stopVideo()

  fun resumeVideo()

  fun dispose()

  fun onCameraClosed()

  // EglContext for Android can be retrieved by doing: WebRtc.rootEglBase.eglBaseContext

  fun getVideoTrack(): VideoStreamTrack?

  fun setPlatform(platform: String)

  fun getScreenshareStream(
    ssStartedCallback: suspend (VideoStreamTrack, MediaStream) -> Unit,
    coroutineScope: CoroutineScope,
    selfController: SelfController,
  )

  fun cleanupScreenshare()
}
