package io.dyte.core.controllers

import io.dyte.core.DyteError
import io.dyte.core.PresetEditorErrorCodes.No_Active_Recording_Found
import io.dyte.core.PresetEditorErrorCodes.Record_Permission_Not_Given
import io.dyte.core.PresetEditorErrorCodes.Recording_Pause_Exception
import io.dyte.core.PresetEditorErrorCodes.Recording_Resume_Exception
import io.dyte.core.PresetEditorErrorCodes.Recording_Start_Exception
import io.dyte.core.PresetEditorErrorCodes.Recording_Stop_Exception
import io.dyte.core.Result
import io.dyte.core.controllers.DyteRecordingState.IDLE
import io.dyte.core.controllers.DyteRecordingState.PAUSED
import io.dyte.core.controllers.DyteRecordingState.RECORDING
import io.dyte.core.controllers.DyteRecordingState.STARTING
import io.dyte.core.controllers.DyteRecordingState.STOPPING
import io.dyte.core.events.EventEmitter
import io.dyte.core.listeners.DyteRecordingEventsListener
import io.dyte.core.network.ApiClient
import io.dyte.core.network.info.RecordingInfo
import io.dyte.core.observability.DyteLogger
import kotlinx.coroutines.CoroutineScope

internal abstract class RecordingController(
  private val apiClient: ApiClient,
  private val canRecord: Boolean,
  protected val scope: CoroutineScope,
) : EventEmitter<DyteRecordingEventsListener>(), IRecordingController {
  private lateinit var recordingInfo: RecordingInfo
  private var recordingState: DyteRecordingState = IDLE

  fun init() {
    setupEvents()
  }

  abstract fun setupEvents()

  override suspend fun start(): Result<Unit, DyteError> {
    DyteLogger.info("DyteRecording::start::")
    if (canRecord.not()) {
      DyteLogger.debug("DyteRecording::start::not_permitted")
      return Result.Failure(DyteError(Record_Permission_Not_Given))
    }

    return try {
      updateState(STARTING)
      recordingInfo = apiClient.startRecording()
      Result.Success(Unit)
    } catch (e: Exception) {
      DyteLogger.error("RecordingController::start", e)
      updateState(IDLE)
      emitEvent { it.onMeetingRecordingStopError(e) }
      Result.Failure(DyteError(Recording_Start_Exception, e.message ?: "Unknown exception occurs"))
    }
  }

  override suspend fun pause(): Result<Unit, DyteError> {
    DyteLogger.info("DyteRecording::pause::")
    if (canRecord.not()) {
      DyteLogger.debug("DyteRecording::pause::not_permitted")
      return Result.Failure(DyteError(Record_Permission_Not_Given))
    }

    if (recordingState != RECORDING) {
      DyteLogger.debug("DyteRecording::pause::not_recording")
      return Result.Failure(DyteError(No_Active_Recording_Found))
    }

    try {
      if (::recordingInfo.isInitialized.not()) {
        recordingInfo = apiClient.getActiveRecording()
      }
      return if (recordingState == RECORDING) {
        updateState(STOPPING)
        apiClient.pauseRecording(recordingInfo)
        updateState(PAUSED)
        Result.Success(Unit)
      } else {
        Result.Failure(DyteError(No_Active_Recording_Found))
      }
    } catch (e: Exception) {
      DyteLogger.error("DyteRecording::pause", e)
      updateState(RECORDING)
      emitEvent { it.onMeetingRecordingPauseError(e) }
      return Result.Failure(
        DyteError(Recording_Pause_Exception, e.message ?: "Unknown exception occurs")
      )
    }
  }

  override suspend fun resume(): Result<Unit, DyteError> {
    DyteLogger.info("DyteRecording::resume::")
    if (canRecord.not()) {
      DyteLogger.debug("DyteRecording::resume::not_permitted")
      return Result.Failure(DyteError(Record_Permission_Not_Given))
    }
    if (recordingState != IDLE && recordingState != PAUSED) {
      DyteLogger.debug("DyteRecording::resume::nothing to resume")
      return Result.Failure(DyteError(No_Active_Recording_Found))
    }
    try {
      if (::recordingInfo.isInitialized.not()) {
        recordingInfo = apiClient.getActiveRecording()
      }
      return if (recordingState == IDLE || recordingState == PAUSED) {
        updateState(STARTING)
        apiClient.resumeRecording(recordingInfo)
        updateState(RECORDING)
        Result.Success(Unit)
      } else {
        Result.Failure(DyteError(No_Active_Recording_Found))
      }
    } catch (e: Exception) {
      DyteLogger.error("DyteRecording::resume", e)
      updateState(IDLE)
      emitEvent { it.onMeetingRecordingResumeError(e) }
      return Result.Failure(
        DyteError(Recording_Resume_Exception, e.message ?: "Unknown exception occurs")
      )
    }
  }

  override suspend fun stop(): Result<Unit, DyteError> {
    DyteLogger.info("DyteRecording::stop::")
    if (canRecord.not()) {
      DyteLogger.debug("DyteRecording::stop::not_permitted")
      return Result.Failure(DyteError(Record_Permission_Not_Given))
    }
    if (recordingState != RECORDING) {
      return Result.Failure(DyteError(No_Active_Recording_Found))
    }
    try {
      if (::recordingInfo.isInitialized.not()) {
        recordingInfo = apiClient.getActiveRecording()
      }
      return if (recordingState == RECORDING) {
        updateState(STOPPING)
        apiClient.stopRecording(recordingInfo)
        updateState(IDLE)
        Result.Success(Unit)
      } else {
        Result.Failure(DyteError(No_Active_Recording_Found))
      }
    } catch (e: Exception) {
      DyteLogger.error("DyteRecording::stop", e)
      updateState(RECORDING)
      emitEvent { it.onMeetingRecordingStopError(e) }
      return Result.Failure(
        DyteError(Recording_Stop_Exception, e.message ?: "Unknown exception occurs")
      )
    }
  }

  override fun getState(): DyteRecordingState {
    return recordingState
  }

  private fun updateState(newState: DyteRecordingState) {
    recordingState = newState
    emitEvent { it.onMeetingRecordingStateUpdated(recordingState) }
  }

  protected fun onRecordingPeerJoined() {
    DyteLogger.debug("DyteRecording::onRecordingPeerJoined::")
    updateState(RECORDING)
    emitEvent { it.onMeetingRecordingStarted() }
  }

  protected fun onRecordingPeerLeft() {
    DyteLogger.debug("DyteRecording::onRecordingPeerLeft::")
    updateState(IDLE)
    emitEvent { it.onMeetingRecordingEnded() }
  }

  protected fun onRecordingPaused() {
    DyteLogger.debug("DyteRecording::onRecordingPaused::")
    updateState(PAUSED)
    emitEvent { it.onMeetingRecordingEnded() }
  }
}

internal interface IRecordingController {
  suspend fun start(): Result<Unit, DyteError>

  suspend fun stop(): Result<Unit, DyteError>

  suspend fun pause(): Result<Unit, DyteError>

  suspend fun resume(): Result<Unit, DyteError>

  fun getState(): DyteRecordingState
}

enum class DyteRecordingState {
  IDLE,
  STARTING,
  RECORDING,
  PAUSED,
  STOPPING;

  internal companion object {
    fun fromMap(stage: String): DyteRecordingState {
      return when (stage) {
        "idle" -> IDLE
        "starting" -> STARTING
        "recording" -> RECORDING
        "stopping" -> STOPPING
        else -> IDLE
      }
    }
  }

  fun toMap(): Map<String, String> {
    val strState =
      when (this) {
        IDLE -> "idle"
        STARTING -> "starting"
        RECORDING -> "recording"
        PAUSED -> "paused"
        STOPPING -> "stopping"
      }

    val map: HashMap<String, String> = HashMap()

    map["state"] = strState

    return map
  }
}
