package io.dyte.core.plugins

import io.dyte.core.Result
import io.dyte.core.controllers.DyteEventType
import io.dyte.core.controllers.IEventController
import io.dyte.core.events.EventEmitter
import io.dyte.core.feat.DytePlugin
import io.dyte.core.listeners.DytePluginEventsListener
import io.dyte.core.network.info.PluginPermissions
import io.dyte.core.observability.DyteLogger
import io.dyte.core.utils.JsonUtils
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.jsonPrimitive

internal abstract class PluginsController(
  protected val scope: CoroutineScope,
  private val selfPluginsPermissions: PluginPermissions,
  protected val pluginsInternalRoomJoinListenerStore: PluginsInternalRoomJoinListenerStore,
  private val flutterNotifier: IEventController,
  plugins: List<DytePlugin>,
) :
  EventEmitter<DytePluginEventsListener>(),
  PluginHostMeeting,
  PluginFileProvider,
  PluginsInternalRoomJoinListener {
  protected val _allPlugins = mutableMapOf<String, DytePlugin>()
  protected val _activePlugins = mutableMapOf<String, DytePlugin>()

  protected val logger: DyteLogger = DyteLogger

  val allPlugins: List<DytePlugin>
    get() = _allPlugins.values.toList()

  val activePlugins: List<DytePlugin>
    get() = _activePlugins.values.toList()

  init {
    // populate _allPlugins map
    plugins.forEach { plugin ->
      _allPlugins[plugin.id] = plugin
      plugin.setHostMeeting(this)
      plugin.setFileProvider(this)
    }
  }

  override fun onRoomJoined() {
    scope.launch {
      try {
        loadActivePlugins()
      } catch (e: Exception) {
        logger.error("DytePlugin::loadActivePlugins::failure", e)
      }
    }
  }

  override suspend fun activatePlugin(
    pluginId: String,
    staggered: Boolean,
  ): Result<Unit, Exception> {
    return withContext(scope.coroutineContext) {
      if (!selfPluginsPermissions.canLaunch) {
        logger.error("DytePlugin::activatePlugin::failure permission denied")
        return@withContext Result.Failure(
          UnsupportedOperationException("Not permitted to activate plugin")
        )
      }

      return@withContext addPlugin(pluginId, staggered)
    }
  }

  override suspend fun deactivatePlugin(pluginId: String): Result<Unit, Exception> {
    return withContext(scope.coroutineContext) {
      if (!selfPluginsPermissions.canClose) {
        logger.error("DytePlugin::deactivatePlugin::failure permission denied")
        return@withContext Result.Failure(
          UnsupportedOperationException("Not permitted to deactivate plugin")
        )
      }

      return@withContext removePlugin(pluginId)
    }
  }

  override suspend fun onFileRequest(plugin: DytePlugin) {
    withContext(scope.coroutineContext) {
      if (_activePlugins.containsKey(plugin.id)) {
        emitEvent { it.onPluginFileRequest(plugin) }
      }
    }
  }

  @Throws(Exception::class) protected abstract suspend fun loadActivePlugins()

  protected abstract suspend fun addPlugin(
    pluginId: String,
    staggered: Boolean,
  ): Result<Unit, Exception>

  protected abstract suspend fun removePlugin(pluginId: String): Result<Unit, Exception>

  protected fun submitEventToPlugin(pluginId: String, payload: JsonObject) {
    _activePlugins[pluginId]?.submitMessage(PluginMessage(payload))
  }

  protected suspend fun enablePluginLocallyAndNotify(pluginId: String, enabledBy: String) {
    _allPlugins[pluginId]?.let { plugin ->
      val result = plugin.enableLocal(enabledBy)
      if (result is Result.Success) {
        _activePlugins[plugin.id] = plugin
        emitEvent { it.onPluginActivated(plugin) }
        flutterNotifier.triggerEvent(DyteEventType.OnPluginsUpdates(activePlugins))
      }
    }
  }

  protected suspend fun disablePluginLocallyAndNotify(pluginId: String) {
    _activePlugins[pluginId]?.let { plugin ->
      val result = plugin.disableLocal()
      if (result is Result.Success) {
        _activePlugins.remove(plugin.id)
        emitEvent { it.onPluginDeactivated(plugin) }
        flutterNotifier.triggerEvent(DyteEventType.OnPluginsUpdates(activePlugins))
      }
    }
  }

  /** Called when this PluginsController is no longer used and will be destroyed. */
  internal open suspend fun onCleared() {
    withContext(scope.coroutineContext) { _allPlugins.values.forEach { plugin -> plugin.clear() } }
  }

  protected fun sendCustomPluginEventToParent(plugin: DytePlugin, eventPayload: JsonObject) {
    val eventName =
      eventPayload["eventName"]?.jsonPrimitive?.content
        ?: kotlin.run {
          logger.info("DytePlugin::sendCustomPluginEventToParent::error Event name is null")
          return
        }
    val eventDataJsonElement = eventPayload["data"]

    if (eventDataJsonElement == null) {
      emitEvent { it.onPluginMessage(plugin, eventName, null) }
      return
    }

    val eventData =
      try {
        JsonUtils.deserialize(eventDataJsonElement)
      } catch (e: Exception) {
        logger.info("DytePlugin::sendCustomPluginEventToParent::error ${e.message}")
        return
      }
    emitEvent { it.onPluginMessage(plugin, eventName, eventData) }
  }
}
