package io.dyte.core.chat

import io.dyte.core.ChatErrorCodes
import io.dyte.core.DyteError
import io.dyte.core.Result
import io.dyte.core.chat.ChatFileUploader.UploadFileResult
import io.dyte.core.chat.models.ChatChannel
import io.dyte.core.chat.models.GetMessagesResult
import io.dyte.core.events.EventEmitter
import io.dyte.core.feat.DyteChatMessage
import io.dyte.core.listeners.DyteChatEventsListener
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.observability.DyteLogger
import io.dyte.core.platform.IDytePlatformUtilsProvider
import io.dyte.core.platform.PlatformFile
import io.dyte.core.platform.Uri
import io.dyte.core.plugins.PluginChatSender
import kotlinx.coroutines.CoroutineScope

/*
 * TODO: Need to setup removeOnMessageListener somewhere.
 * Ideal way would be to listen to `roomJoined` event to set listeners and
 * `roomLeft` event to remove listeners.
 * */
internal abstract class BaseChatController(
  private val platformUtilsProvider: IDytePlatformUtilsProvider,
  private val selfParticipant: DyteSelfParticipant,
  private val canSendPrivateChatText: Boolean,
  private val canSendPrivateChatFile: Boolean,
  private val chatFileUploader: ChatFileUploader,
  protected val scope: CoroutineScope,
) : EventEmitter<DyteChatEventsListener>(), IChatController, PluginChatSender {

  private lateinit var roomUUID: String

  /** TODO: Remove this */
  override fun setRoomUUID(uuid: String) {
    roomUUID = uuid
    chatFileUploader.setRoomUUID(uuid)
  }

  override val messages = arrayListOf<DyteChatMessage>()

  override val channels: MutableList<ChatChannel> = mutableListOf()

  /**
   * Sets a message listener on the ChatSocketHandler for messages coming from the socket service.
   *
   * TODO: Need to figure out a way to remove listener from ChatSocketHandler. Ideal way would be to
   *   listen to `roomJoined` event to set listeners and `roomLeft` event to remove listeners.
   */
  override suspend fun loadChatMessages() {
    try {
      val newMessages = this.getChatMessages()
      handleMultipleMessages(newMessages)
    } catch (e: Exception) {
      DyteLogger.error("DyteChat::loadChatMessages::error", e)
    }
  }

  protected fun handleMultipleMessages(newMessages: List<DyteChatMessage>) {
    messages.clear()
    messages.addAll(newMessages)
    if (newMessages.isNotEmpty()) {
      emitEvent { it.onChatUpdates(messages) }
    }
  }

  protected fun handleMessage(newMessage: DyteChatMessage) {
    messages.add(newMessage)
    emitEvent {
      it.onNewChatMessage(newMessage)
      it.onChatUpdates(messages)
    }
  }

  protected abstract suspend fun getChatMessages(): List<DyteChatMessage>

  override suspend fun sendMessageToPeers(
    message: String,
    peerIds: List<String>,
  ): Result<Unit, DyteError> {
    DyteLogger.debug("DyteChat::sendTextMessage ${selfParticipant.permissions.chat.canSendText}")
    if (peerIds.isEmpty() && !selfParticipant.permissions.chat.canSendText) {
      DyteLogger.debug("DyteChat::sendMessageToPeers::public_chat_permission_denied")
      return Result.Failure(DyteError(ChatErrorCodes.Chat_Not_Supported))
    }
    if (peerIds.isNotEmpty() && !canSendPrivateChatText) {
      DyteLogger.debug("DyteChat::sendMessageToPeers::private_chat_permission_denied")
      return Result.Failure(DyteError(ChatErrorCodes.Chat_Not_Supported))
    }

    if (message.isEmpty()) {
      DyteLogger.debug("DyteChat::sendTextMessage::message_can_not_be_empty")
      throw IllegalArgumentException("message can not be empty")
    }
    return sendTextMessageSocket(message, peerIds)
  }

  protected abstract suspend fun sendTextMessageSocket(
    message: String,
    peerIds: List<String>,
  ): Result<Unit, DyteError>

  override suspend fun sendFileMessageToPeers(
    fileUri: Uri,
    peerIds: List<String>,
  ): Result<Unit, DyteError> {
    DyteLogger.debug("DyteChat::sendFileMessage")
    if (peerIds.isEmpty() && !selfParticipant.permissions.chat.canSendFiles) {
      // This means we are going to send a public chat which is for every member in the meeting
      DyteLogger.debug("DyteChat::sendFileMessageToPeers::chat_permission_denied")
      throw UnsupportedOperationException("not allowed to send file message")
    }
    if (peerIds.isNotEmpty() && !canSendPrivateChatFile) {
      DyteLogger.debug("DyteChat::sendFileMessageToPeers::private_chat_permission_denied")
      throw UnsupportedOperationException("not allowed to send file message to selected peers")
    }

    val file =
      platformUtilsProvider.getPlatformUtils().getPlatformFile(fileUri)
        ?: run {
          DyteLogger.error("DyteChat::sendFileMessageToPeers::chat_read_file_failed")
          throw Exception("Requested file doesn't exist or is missing read permission")
        }

    return sendFileMessageInternal(file, peerIds)
  }

  override suspend fun sendFileMessageToPeers(
    filePath: String,
    peerIds: List<String>,
  ): Result<Unit, DyteError> {
    DyteLogger.debug("DyteChat::sendFileMessage")
    if (peerIds.isEmpty() && !selfParticipant.permissions.chat.canSendFiles) {
      DyteLogger.debug("DyteChat::sendFileMessageToPeers::chat_permission_denied")
      return Result.Failure(DyteError(ChatErrorCodes.Chat_Files_Not_Supported))
    }
    if (peerIds.isNotEmpty() && !canSendPrivateChatFile) {
      DyteLogger.debug("DyteChat::sendFileMessageToPeers::private_chat_permission_denied")
      return Result.Failure(DyteError(ChatErrorCodes.Chat_Files_Not_Supported))
    }

    val file =
      platformUtilsProvider.getPlatformUtils().getPlatformFile(filePath)
        ?: run {
          DyteLogger.warn("DyteChat::sendFileMessage::chat_read_file_failed")
          throw Exception("Requested file doesn't exist or is missing read permission")
        }

    return sendFileMessageInternal(file, peerIds)
  }

  override suspend fun sendImageMessageToPeers(
    imageUri: Uri,
    peerIds: List<String>,
  ): Result<Unit, DyteError> {
    DyteLogger.debug("DyteChat::sendImageMessage")
    if (peerIds.isEmpty() && !selfParticipant.permissions.chat.canSendFiles) {
      DyteLogger.warn("DyteChat::sendImageMessageToPeers::permission_denied")
      return Result.Failure(DyteError(ChatErrorCodes.Chat_Files_Not_Supported))
    }

    if (peerIds.isNotEmpty() && !canSendPrivateChatFile) {
      DyteLogger.warn("DyteChat::sendImageMessageToPeers::permission_denied")
      return Result.Failure(DyteError(ChatErrorCodes.Chat_Files_Not_Supported))
    }

    val file =
      platformUtilsProvider.getPlatformUtils().getPlatformFile(imageUri)
        ?: run {
          DyteLogger.error("DyteChat::sendImageMessage::required_argument_image_can_not_be_empty")
          throw Exception("Requested image doesn't exist or is missing read permission")
        }

    return sendImageMessageInternal(file, peerIds)
  }

  override suspend fun sendImageMessageToPeers(
    imagePath: String,
    peerIds: List<String>,
  ): Result<Unit, DyteError> {
    DyteLogger.debug("DyteChat::sendImageMessage")
    if (peerIds.isEmpty() && !selfParticipant.permissions.chat.canSendFiles) {
      DyteLogger.debug("DyteChat::sendImageMessageToPeers::permission_denied")
      return Result.Failure(DyteError(ChatErrorCodes.Chat_Files_Not_Supported))
    }
    if (peerIds.isNotEmpty() && !canSendPrivateChatFile) {
      DyteLogger.debug("DyteChat::sendImageMessageToPeers::permission_denied")
      return Result.Failure(DyteError(ChatErrorCodes.Chat_Files_Not_Supported))
    }

    val file =
      platformUtilsProvider.getPlatformUtils().getPlatformFile(imagePath)
        ?: throw Exception("Requested image doesn't exist or is missing read permission")
    return sendImageMessageInternal(file, peerIds)
  }

  private suspend fun sendImageMessageInternal(
    file: PlatformFile,
    peerIds: List<String>,
  ): Result<Unit, DyteError> {
    if (file.mimeType !in ACCEPTED_IMAGE_TYPES) {
      DyteLogger.error("sendImageMessage::image_type_not_supported")
      throw UnsupportedOperationException("Unsupported image file found -> ${file.mimeType}")
    }

    return when (val uploadImageResult = chatFileUploader.uploadFile(file)) {
      is UploadFileResult.Uploaded -> {
        this.sendImageMessageSocket(uploadImageResult.getLocation, peerIds)
        Result.Success(Unit)
      }
      is UploadFileResult.UploadFailed -> {
        DyteLogger.error("DyteChat::sendImageMessage::${uploadImageResult.reason}")
        Result.Failure(DyteError(ChatErrorCodes.Chat_Files_Not_Supported))
      }
    }
  }

  protected abstract suspend fun sendImageMessageSocket(
    path: String,
    peerIds: List<String>,
  ): Result<Unit, DyteError>

  private suspend fun sendFileMessageInternal(
    file: PlatformFile,
    peerIds: List<String>,
  ): Result<Unit, DyteError> {
    return when (val uploadFileResult = chatFileUploader.uploadFile(file)) {
      is UploadFileResult.Uploaded -> {
        sendFileMessageSocket(uploadFileResult.getLocation, file.name, file.size, peerIds)
        Result.Success(Unit)
      }
      is UploadFileResult.UploadFailed -> {
        DyteLogger.error("DyteChat::sendFileMessage::failure_upload_file")
        Result.Failure(DyteError(ChatErrorCodes.Chat_Files_Not_Supported))
      }
    }
  }

  protected abstract suspend fun sendFileMessageSocket(
    fileUrl: String,
    fileName: String,
    fileSize: Long,
  ): Result<Unit, DyteError>

  protected abstract suspend fun sendFileMessageSocket(
    fileUrl: String,
    fileName: String,
    fileSize: Long,
    peerIds: List<String>,
  ): Result<Unit, DyteError>

  override suspend fun sendTextMessageFromPlugin(message: String) {
    sendMessageToPeers(message, listOf())
  }

  protected fun parseTime(t: Long?): String {
    return platformUtilsProvider
      .getPlatformUtils()
      .getUserDisplayableTime(t?.times(1000) ?: error("Message time is null"))
  }

  companion object {
    private val ACCEPTED_IMAGE_TYPES = listOf("image/jpeg", "image/png")
  }
}

internal interface IChatController {
  val messages: List<DyteChatMessage>

  val channels: List<ChatChannel>

  fun setRoomUUID(uuid: String)

  suspend fun loadChatMessages()

  suspend fun sendMessageToPeers(message: String, peerIds: List<String>): Result<Unit, DyteError>

  suspend fun sendImageMessageToPeers(imageUri: Uri, peerIds: List<String>): Result<Unit, DyteError>

  suspend fun sendImageMessageToPeers(
    imagePath: String,
    peerIds: List<String>,
  ): Result<Unit, DyteError>

  suspend fun sendFileMessageToPeers(fileUri: Uri, peerIds: List<String>): Result<Unit, DyteError>

  suspend fun sendFileMessageToPeers(
    filePath: String,
    peerIds: List<String>,
  ): Result<Unit, DyteError>

  suspend fun getMessagesPaginated(
    timestamp: Long,
    size: Int,
    offset: Int,
    reversed: Boolean,
    channelId: String?,
  ): Result<GetMessagesResult, String>
}

internal enum class MessageType(val type: Int) {
  TEXT(0),
  IMAGE(1),
  FILE(2)
}
