package io.dyte.core.controllers.stage

import io.dyte.core.controllers.DyteStageStatus
import io.dyte.core.controllers.ParticipantController
import io.dyte.core.controllers.SelfController
import io.dyte.core.events.EventEmitter
import io.dyte.core.events.InternalEvents
import io.dyte.core.media.IDyteCommonMediaUtils
import io.dyte.core.models.DyteJoinedMeetingParticipant
import io.dyte.core.models.DyteMediaPermission
import io.dyte.core.models.DyteMeetingType
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.observability.DyteLogger
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketJoinRoomModel
import io.dyte.core.socket.socketservice.ISockratesSocketService
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import io.dyte.core.socket.socketservice.SocketServiceUtils
import io.dyte.core.utils.ReadHeavyMutableList
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import socket.livestreaming.DenyStageAccessRequest
import socket.livestreaming.GetStageRequestsResponse
import socket.livestreaming.GrantStageAccessRequest
import socket.livestreaming.LeaveStageRequest
import socket.livestreaming.StageRequest
import socket.room.PeerStatusUpdate
import socket.room.StageType

internal class StageSocketServiceController(
  val self: DyteSelfParticipant,
  private val socketService: ISockratesSocketService,
  scope: CoroutineScope,
  private val participantController: ParticipantController,
  private val selfController: SelfController,
  private val mediaUtils: IDyteCommonMediaUtils,
  private val internalEventsEmitter: EventEmitter<InternalEvents>,
  private val meetingType: DyteMeetingType,
) : BaseStageController(self, scope) {
  private val logger = DyteLogger

  private val stageRequests = ReadHeavyMutableList<DyteJoinedMeetingParticipant>()

  override val accessRequests: List<DyteJoinedMeetingParticipant>
    get() = stageRequests.toSafeList()

  // we dont show any viewers in case of livestream stage. this is only to support webinar
  override val viewers: List<DyteJoinedMeetingParticipant>
    get() =
      if (meetingType == DyteMeetingType.WEBINAR) {
        participantController.joinedParticipants.filter {
          it.stageStatus != DyteStageStatus.ON_STAGE
        }
      } else {
        emptyList()
      }

  private val socketEventListener =
    object : SocketServiceEventListener {
      override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
        scope.launch { handleStageSocketEvent(event, payload) }
      }
    }

  override fun setupEvents() {
    DyteLogger.info("DyteStage::using socket-service stage")
    socketService.subscribe(
      SocketServiceUtils.RoomEvent.PEER_STAGE_STATUS_UPDATE.id,
      socketEventListener,
    )

    if (self.permissions.host.canAcceptRequests) {
      socketService.subscribe(
        SocketServiceUtils.RoomEvent.GET_STAGE_REQUESTS.id,
        socketEventListener,
      )
    }

    // When a participant joins or leave stage
    socketService.subscribe(SocketServiceUtils.RoomEvent.GET_STAGE_PEERS.id, socketEventListener)

    socketService.subscribe(
      SocketServiceUtils.RoomEvent.GRANT_STAGE_REQUEST.id,
      socketEventListener,
    )

    socketService.subscribe(SocketServiceUtils.RoomEvent.DENY_STAGE_REQUEST.id, socketEventListener)
  }

  override fun onRoomJoined(webSocketJoinRoomModel: WebSocketJoinRoomModel) {
    DyteLogger.info("DyteStage::onRoomJoined::")
    val selfInitialStageStatus =
      DyteStageStatus.fromWebsocketStageStatus(webSocketJoinRoomModel.stageStatus)
    /*
     * Note(swapnil): This is needed in case of reconnection because if Self is a host then currently
     * the server sends the stageStatus as ON_STAGE even if self was OFF_STAGE before disconnecting.
     * Additionally, we are doing a force-update and emitting it even when joining first time for flutter-core.
     * */
    updateSelfStageStatusAndEmit(selfInitialStageStatus, forceUpdate = true)

    if (self.permissions.host.canAcceptRequests) {
      runBlocking(scope.coroutineContext) {
        val socketResponse =
          try {
            socketService.requestResponse(
              event = SocketServiceUtils.RoomEvent.GET_STAGE_REQUESTS.id,
              payload = null,
            )
          } catch (e: Exception) {
            logger.warn("DyteStage::onRoomJoined::failed to get stage requests -> ${e.message}")
            return@runBlocking
          }

        val getStageRequestsResponse =
          if (socketResponse != null) {
            try {
              GetStageRequestsResponse.ADAPTER.decode(socketResponse)
            } catch (e: Exception) {
              DyteLogger.warn("DyteStage::onRoomJoined::failed to decode GetStageRequestsResponse")
              GetStageRequestsResponse(emptyList())
            }
          } else {
            // currently null response means empty request list
            GetStageRequestsResponse(emptyList())
          }

        refreshAccessRequests(getStageRequestsResponse.stage_requests)

        if (accessRequests.isNotEmpty()) {
          emitEvent { it.onStageRequestsUpdated(accessRequests) }
        }
      }
    }
  }

  override suspend fun requestAccess() {
    DyteLogger.info("DyteStage::requestAccess::")
    if (!self.permissions.miscellaneous.stageEnabled) {
      logger.error("DyteStage::requestAccess::stage is disabled")
      return
    }

    val selfStageStatus = self.stageStatus // for thread-safety
    if (selfStageStatus !== DyteStageStatus.OFF_STAGE) {
      logger.warn("DyteStage::requestAccess::stage status is already $selfStageStatus")
      return
    }

    // beyond this Self is OFF_STAGE
    if (self.permissions.miscellaneous.stageAccess == DyteMediaPermission.NOT_ALLOWED) {
      logger.warn("DyteStage::requestAccess::self not allowed to send stage request")
      return
    }

    if (self.permissions.miscellaneous.stageAccess === DyteMediaPermission.ALLOWED) {
      updateSelfStageStatusAndEmit(DyteStageStatus.ACCEPTED_TO_JOIN_STAGE)
      return
    }

    try {
      socketService.send(SocketServiceUtils.RoomEvent.REQUEST_STAGE_ACCESS.id, null)
      updateSelfStageStatusAndEmit(DyteStageStatus.REQUESTED_TO_JOIN_STAGE)
    } catch (e: Exception) {
      logger.error("DyteStage::requestAccess::socket request failed ${e.message}")
    }
  }

  // TODO: add permission checks
  override suspend fun denyAccess(id: String) {
    DyteLogger.info("DyteStage::denyAccess::$id")
    val participantToDenyAccess =
      accessRequests.find { it.id == id }
        ?: kotlin.run {
          logger.error("DyteStage::denyAccess::stage request for peerId $id not found")
          return
        }
    val userIdToDenyAccess = participantToDenyAccess.userId
    try {
      socketService.send(
        SocketServiceUtils.RoomEvent.DENY_STAGE_REQUEST.id,
        DenyStageAccessRequest(listOf(userIdToDenyAccess)).encode(),
      )
    } catch (e: Exception) {
      logger.error("DyteStage::denyAccess::socket request failed ${e.message}")
    }
  }

  // TODO: add permission checks
  override suspend fun denyAccessAll() {
    DyteLogger.info("DyteStage::denyAccessAll::")
    val userIds = accessRequests.map { it.userId }
    try {
      socketService.send(
        SocketServiceUtils.RoomEvent.DENY_STAGE_REQUEST.id,
        DenyStageAccessRequest(userIds).encode(),
      )
    } catch (e: Exception) {
      logger.error("DyteStage::denyAccessAll::socket request failed ${e.message}")
    }
  }

  // TODO: add permission checks
  override suspend fun grantAccess(id: String) {
    // covers both accept stage request and invite to stage
    DyteLogger.info("DyteStage::grantAccess::$id")
    val participantToGrantAccess =
      participantController.joinedParticipants.find { it.id == id }
        ?: kotlin.run {
          logger.error("DyteStage::grantAccess::participant with peerId $id not found")
          return
        }
    val userIdToGrantAccess = participantToGrantAccess.userId
    try {
      socketService.send(
        SocketServiceUtils.RoomEvent.GRANT_STAGE_REQUEST.id,
        GrantStageAccessRequest(listOf(userIdToGrantAccess)).encode(),
      )
    } catch (e: Exception) {
      logger.error("DyteStage::grantAccess::socket request failed ${e.message}")
    }
  }

  // TODO: add permission checks
  override suspend fun grantAccessAll() {
    DyteLogger.info("DyteStage::grantAccessAll::")
    val userIds = accessRequests.map { it.userId }
    try {
      socketService.send(
        SocketServiceUtils.RoomEvent.GRANT_STAGE_REQUEST.id,
        GrantStageAccessRequest(userIds).encode(),
      )
    } catch (e: Exception) {
      logger.error("DyteStage::grantAccessAll::socket request failed ${e.message}")
    }
  }

  override suspend fun join() {
    DyteLogger.info("DyteStage::join::")
    if (self.stageStatus == DyteStageStatus.ON_STAGE) {
      logger.warn("DyteStage::join::self already ON_STAGE")
      return
    }

    if (
      self.permissions.miscellaneous.stageAccess != DyteMediaPermission.ALLOWED &&
        self.stageStatus != DyteStageStatus.ACCEPTED_TO_JOIN_STAGE
    ) {
      logger.warn("DyteStage::join::self not allowed to join stage")
      return
    }

    try {
      socketService.send(SocketServiceUtils.RoomEvent.JOIN_STAGE.id, null)

      updateSelfStageStatusAndEmit(DyteStageStatus.ON_STAGE)

      if (meetingType == DyteMeetingType.WEBINAR) {
        internalEventsEmitter.emitEvent { it.createProducers() }
      } else if (meetingType == DyteMeetingType.LIVESTREAM) {
        internalEventsEmitter.emitEvent { it.connectToRoomNode() }
        // internalEventsEmitter.emitEvent { it.connectMedia() }
      }
    } catch (e: Exception) {
      logger.error("DyteStage::join::socket request failed ${e.message}")
    }
  }

  override suspend fun leave() {
    DyteLogger.info("DyteStage::leave::")
    if (
      !(self.stageStatus == DyteStageStatus.ON_STAGE ||
        self.stageStatus == DyteStageStatus.ACCEPTED_TO_JOIN_STAGE)
    ) {
      logger.warn("DyteStage::leave::self is currently ${self.stageStatus}")
      return
    }

    // TODO: Move this logic below send socket message
    // Leave stage operation can fail, in that case we don't want to close producers
    if (meetingType == DyteMeetingType.WEBINAR) {
      DyteLogger.info("DyteStage::leave::webinar_stage")
      disableMediaProduction()
    } else if (meetingType == DyteMeetingType.LIVESTREAM) {
      DyteLogger.info("DyteStage::leave::livestream_stage")
      internalEventsEmitter.emitEvent { it.disconnectFromRoomNode() }
    }

    val request = LeaveStageRequest(listOf(self.userId)).encode()
    try {
      socketService.send(SocketServiceUtils.RoomEvent.LEAVE_STAGE.id, request)
      updateSelfStageStatusAndEmit(DyteStageStatus.OFF_STAGE)
    } catch (e: Exception) {
      logger.error("DyteStage::leave::socket request failed ${e.message}")
    }
  }

  // TODO: add acceptStageRequests permission check
  override suspend fun kick(id: String) {
    if (!self.permissions.host.canAcceptRequests) {
      logger.warn("DyteStage::kick::self not allowed to kick a participant")
      return
    }

    if (id == self.id) {
      leave()
      return
    }

    val participantToKick =
      participantController.joinedParticipants.find { it.id == id }
        ?: kotlin.run {
          logger.error("DyteStage::kick::participant $id not found")
          return
        }

    if (
      !(participantToKick.stageStatus == DyteStageStatus.ON_STAGE ||
        participantToKick.stageStatus == DyteStageStatus.ACCEPTED_TO_JOIN_STAGE)
    ) {
      logger.error("DyteStage::kick::participant $id is currently ${participantToKick.stageStatus}")
      return
    }

    try {
      val leaveStageRequest = LeaveStageRequest(listOf(participantToKick.userId)).encode()
      socketService.send(SocketServiceUtils.RoomEvent.LEAVE_STAGE.id, leaveStageRequest)
    } catch (e: Exception) {
      logger.error("DyteStage::grantAccess::socket request failed ${e.message}")
    }
  }

  override suspend fun cancelRequestAccess() {
    DyteLogger.info("DyteStage::cancelRequestAccess::")
    if (!self.permissions.miscellaneous.stageEnabled) {
      logger.error("DyteStage::cancelRequestAccess::stage is disabled")
      return
    }

    if (self.permissions.miscellaneous.stageAccess != DyteMediaPermission.CAN_REQUEST) {
      logger.warn("DyteStage::cancelRequestAccess::self cannot send or cancel stage request")
      return
    }

    // beyond this Self has CAN_REQUEST permission
    val selfStageStatus = self.stageStatus // for thread-safety
    if (selfStageStatus !== DyteStageStatus.REQUESTED_TO_JOIN_STAGE) {
      logger.warn("DyteStage::cancelRequestAccess::stage status is not REQUESTED_TO_JOIN_STAGE")
      return
    }

    // Self stageStatus is REQUESTED_TO_JOIN_STAGE
    try {
      socketService.send(SocketServiceUtils.RoomEvent.CANCEL_STAGE_REQUEST.id, null)
      updateSelfStageStatusAndEmit(DyteStageStatus.OFF_STAGE)
    } catch (e: Exception) {
      logger.error("DyteStage::cancelRequestAccess::socket request failed ${e.message}")
    }
  }

  private fun handleStageSocketEvent(event: Int, payload: ByteArray?) {
    when (event) {
      SocketServiceUtils.RoomEvent.PEER_STAGE_STATUS_UPDATE.id -> {
        DyteLogger.info("DyteStage::handleStageSocketEvent::peer stage status update")
        if (payload == null) {
          return
        }

        val peerStageStatusUpdate =
          try {
            PeerStatusUpdate.ADAPTER.decode(payload)
          } catch (e: Exception) {
            DyteLogger.warn(
              "DyteStage::handle PEER_STAGE_STATUS_UPDATE socket event::failed to decode PeerStatusUpdate"
            )
            return
          }

        val updatedStageStatus =
          peerStageStatusUpdate.stage_type?.toDyteStageStatus()
            ?: kotlin.run {
              logger.warn(
                "DyteStage::handle PEER_STAGE_STATUS_UPDATE socket event::stage status received is null"
              )
              return
            }

        /*
         * note: Temporarily handling self removed from stage scenario using PEER_STAGE_STATUS_UPDATE.
         * In coming days, core will receive LEAVE_STAGE for this scenario.
         * */
        if (peerStageStatusUpdate.peer_id == self.id) {
          handleSelfStageStatusUpdate(updatedStageStatus)
          return
        }

        val updatePeerStageStatusResult =
          participantController.updatePeerStageStatus(
            peerStageStatusUpdate.peer_id,
            updatedStageStatus,
          ) ?: return

        val oldStageStatus = updatePeerStageStatusResult.oldStageStatus

        when (updatedStageStatus) {
          // started presenting
          // should we also check for invalid transition like OFF_STAGE to ON_STAGE?
          DyteStageStatus.ON_STAGE -> {
            runBlocking { // note: do not provide scope.context!!
              refreshGridParticipantsIfViewModeActive()
            }
            emitEvent { it.onParticipantStartedPresenting(updatePeerStageStatusResult.updatedPeer) }
          }

          // stopped presenting
          // should we also check for invalid transition like ON_STAGE to REQUESTED?
          DyteStageStatus.OFF_STAGE -> {
            /*
             * TODO: Check if there is any difference in resultant stageStatus is an ACCEPTED participant
             *  leaves the stage vs they are removed from stage.
             * */
            if (oldStageStatus == DyteStageStatus.ON_STAGE) {
              runBlocking { // note: do not provide scope.context!!
                refreshGridParticipantsIfViewModeActive()
              }
              emitEvent {
                it.onParticipantStoppedPresenting(updatePeerStageStatusResult.updatedPeer)
              }
            }
          }

          // requested to join stage
          DyteStageStatus.REQUESTED_TO_JOIN_STAGE -> {
            // emit only if self is host
            if (
              oldStageStatus == DyteStageStatus.OFF_STAGE && self.permissions.host.canAcceptRequests
            ) {
              emitEvent { it.onPresentRequestAdded(updatePeerStageStatusResult.updatedPeer) }
              return
            }
          }
          else -> {
            // ignore
          }
        }

        /*
         * currently no support for emitting following peer events:
         * - accepted to join stage (REQUESTED -> ACCEPTED)
         * - invited to join stage (OFF_STAGE -> ACCEPTED)
         * - rejected to join stage (REQUESTED -> OFF_STAGE)
         * - request withdrawn
         * */

        // should we emit DyteParticipantEventsListener.onUpdate?
      }
      /** triggered whenever there is any change in stage access requests */
      SocketServiceUtils.RoomEvent.GET_STAGE_REQUESTS.id -> {
        if (!self.permissions.host.canAcceptRequests) {
          return
        }

        DyteLogger.info("DyteStage::handleStageSocketEvent::stage requests updated")

        val getStageRequestsResponse =
          if (payload != null) {
            try {
              GetStageRequestsResponse.ADAPTER.decode(payload)
            } catch (e: Exception) {
              DyteLogger.warn(
                "DyteStage::handle GET_STAGE_REQUESTS socket event::failed to decode GetStageRequestsResponse"
              )
              GetStageRequestsResponse(emptyList())
            }
          } else {
            // currently null response means empty request list
            GetStageRequestsResponse(emptyList())
          }

        refreshAccessRequests(getStageRequestsResponse.stage_requests)

        emitEvent { it.onStageRequestsUpdated(accessRequests) }
      }
      SocketServiceUtils.RoomEvent.GRANT_STAGE_REQUEST.id -> {
        if (self.permissions.miscellaneous.stageAccess == DyteMediaPermission.ALLOWED) {
          return
        }
        updateSelfStageStatusAndEmit(DyteStageStatus.ACCEPTED_TO_JOIN_STAGE)
        emitEvent { it.onPresentRequestReceived() }
      }
      SocketServiceUtils.RoomEvent.DENY_STAGE_REQUEST.id -> {
        // ignore event if self is host (presenter)
        if (self.permissions.miscellaneous.stageAccess == DyteMediaPermission.ALLOWED) {
          return
        }
        updateSelfStageStatusAndEmit(DyteStageStatus.OFF_STAGE)
      }
      else -> {}
    }
  }

  private fun updateSelfStageStatusAndEmit(
    stageStatus: DyteStageStatus,
    forceUpdate: Boolean = false,
  ) {
    if (self.stageStatus == stageStatus && !forceUpdate) {
      return
    }

    self._stageStatus = stageStatus
    emitEvent { it.onStageStatusUpdated(stageStatus) }
    selfController.emitEvent { it.onStageStatusUpdated(stageStatus) }
    participantController.triggerOnUpdate()
  }

  /** Handles stageUpdate for Self when they go ON_STAGE or OFF_STAGE. */
  private fun handleSelfStageStatusUpdate(updatedStageStatus: DyteStageStatus) {
    // val selfOldStageStatus = self.stageStatus
    val updateIsSelfInitiated = self.stageStatus == updatedStageStatus

    updateSelfStageStatusAndEmit(updatedStageStatus)

    if (updatedStageStatus == DyteStageStatus.ON_STAGE) {
      runBlocking { // note: do not provide scope.context!!
        refreshGridParticipantsIfViewModeActive()
      }
      if (!updateIsSelfInitiated) {
        emitEvent { it.onAddedToStage() }
      }
      return
    }

    if (updatedStageStatus == DyteStageStatus.OFF_STAGE) {

      if (meetingType == DyteMeetingType.WEBINAR) {
        if (!updateIsSelfInitiated) {
          disableMediaProduction()
        }
        runBlocking { // note: do not provide scope.context!!
          refreshGridParticipantsIfViewModeActive()
        }
        if (!updateIsSelfInitiated) {
          emitEvent { it.onRemovedFromStage() }
        }
      } else if (meetingType == DyteMeetingType.LIVESTREAM) {
        internalEventsEmitter.emitEvent { it.disconnectFromRoomNode() }
        if (!updateIsSelfInitiated) {
          emitEvent { it.onRemovedFromStage() }
        }
      }

      return
    }
  }

  private fun refreshAccessRequests(updatedStageRequests: List<StageRequest>) {
    stageRequests.clear()
    for (stageRequest in updatedStageRequests) {
      val requestingPeer =
        participantController.joinedParticipants.find { it.id == stageRequest.peer_id } ?: continue
      stageRequests.add(requestingPeer)
    }
  }

  private suspend fun refreshGridParticipantsIfViewModeActive() {
    if (participantController.getCurrentPageNumber() == 0) {
      participantController.refreshGridParticipantsHive()
    }
  }

  private fun disableMediaProduction() {
    internalEventsEmitter.emitEvent { it.closeProducers() }
    selfController.getSelf().disableAudio()
    selfController.getSelf().disableVideo()
    mediaUtils.stopAudio()
    mediaUtils.stopVideo()
  }

  companion object {
    private fun StageType.toDyteStageStatus(): DyteStageStatus {
      return when (this) {
        StageType.STAGE_TYPE_ON_STAGE -> DyteStageStatus.ON_STAGE
        StageType.STAGE_TYPE_APPROVED_STAGE -> DyteStageStatus.ACCEPTED_TO_JOIN_STAGE
        StageType.STAGE_TYPE_REQUESTED_STAGE -> DyteStageStatus.REQUESTED_TO_JOIN_STAGE
        else -> DyteStageStatus.OFF_STAGE
      }
    }
  }
}
