package io.dyte.core.controllers.polls

import io.dyte.core.feat.DytePollMessage
import io.dyte.core.feat.DytePollOption
import io.dyte.core.feat.DytePollVote
import io.dyte.core.network.info.PollPermissions
import io.dyte.core.observability.DyteLogger
import io.dyte.core.socket.socketservice.ISockratesSocketService
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import io.dyte.core.socket.socketservice.SocketServiceUtils
import socket.polls.GetPollsResponse
import socket.polls.NewPollRequest
import socket.polls.Poll
import socket.polls.UpdatePollResponse
import socket.polls.VotePollRequest

internal class PollsSocketServiceController(
  permissions: PollPermissions,
  private val socket: ISockratesSocketService,
) : BasePollsController(permissions) {

  override fun setupEvents() {
    DyteLogger.info("DytePolls::using socket-service polls::")
    socket.subscribe(PollSocketEvent.CREATE_POLL.id, pollSocketListener)
    socket.subscribe(PollSocketEvent.UPDATE_POLL.id, pollSocketListener)
  }

  private val pollSocketListener =
    object : SocketServiceEventListener {
      override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
        if (payload == null) {
          DyteLogger.warn(
            "PollsSocketServiceController::socketEvent $event::received payload is null"
          )
          return
        }
        when (event) {
          PollSocketEvent.CREATE_POLL.id -> {
            DyteLogger.info("DytePolls::onEvent::create_poll")
            val decodedPayload = UpdatePollResponse.ADAPTER.decode(payload)
            decodedPayload.poll?.let { onNewPoll(parsePoll(it), true) }
          }
          PollSocketEvent.UPDATE_POLL.id -> {
            DyteLogger.info("DytePolls::onEvent::update_poll")
            val decodedPayload = UpdatePollResponse.ADAPTER.decode(payload)
            decodedPayload.poll?.let { onNewPoll(parsePoll(it), true) }
          }
        }
      }
    }

  override suspend fun loadPolls() {
    DyteLogger.info("DytePolls::loadPolls::")
    try {
      val payload = socket.requestResponse(PollSocketEvent.GET_POLLS.id, null)
      payload?.let {
        val decodedPayload = GetPollsResponse.ADAPTER.decode(payload)
        decodedPayload.polls.forEach { onNewPoll(parsePoll(it), false) }
        if (polls.isNotEmpty()) {
          emitEvent { it.onPollUpdates(polls) }
        }
      }
    } catch (e: Exception) {
      // ignore exception
    }
  }

  override suspend fun sendPoll(
    question: String,
    options: List<String>,
    anonymous: Boolean,
    hideVotes: Boolean,
  ) {
    DyteLogger.info("DytePolls::sendPoll::")
    val newPollRequest = NewPollRequest(question, options, anonymous)
    socket.send(
      event = PollSocketEvent.CREATE_POLL.id,
      payload = NewPollRequest.ADAPTER.encode(newPollRequest),
    )
  }

  override suspend fun sendVote(pollMessage: DytePollMessage, pollOption: DytePollOption) {
    DyteLogger.info("DytePolls::sendVote::")
    socket.send(
      PollSocketEvent.VOTE_POLL.id,
      VotePollRequest.ADAPTER.encode(
        VotePollRequest(pollMessage.id, pollMessage.options.indexOf(pollOption).toLong())
      ),
    )
  }

  private fun parsePoll(poll: Poll): DytePollMessage {
    val options = ArrayList<DytePollOption>()
    poll.options.forEach { option ->
      val votes = ArrayList<DytePollVote>()
      val updateOption = DytePollOption(option.text, votes, option.count?.toInt() ?: 0)
      option.votes.forEach { vote -> votes.add(DytePollVote(vote.user_id, vote.name)) }
      options.add(updateOption)
    }
    return DytePollMessage(
      poll.poll_id,
      poll.question,
      poll.anonymous,
      poll.hide_votes,
      poll.created_by,
      options,
    )
  }
}

internal enum class PollSocketEvent(slug: Int) {
  CREATE_POLL(0),
  GET_POLLS(1),
  VOTE_POLL(2),
  UPDATE_POLL(3);

  private val base: Int = 3
  val id: Int = SocketServiceUtils.generateSocketEventId(base, slug)
}
