package io.dyte.core.controllers

import io.dyte.core.controllers.DyteEventType.OnViewerCountUpdated
import io.dyte.core.controllers.LiveStreamState.ERRORED
import io.dyte.core.controllers.LiveStreamState.NONE
import io.dyte.core.controllers.LiveStreamState.STARTED
import io.dyte.core.controllers.LiveStreamState.STOPPED
import io.dyte.core.observability.DyteLogger
import io.dyte.core.socket.socketservice.ISockratesSocketService
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import io.dyte.core.socket.socketservice.SocketServiceUtils.RoomEvent
import kotlinx.coroutines.launch
import socket.livestreaming.*
import socket.room.RoomPeerCountResponse

internal class LiveStreamController(
  controllerContainer: IControllerContainer,
  private val socketService: ISockratesSocketService,
) : BaseController(controllerContainer), ILiveStreamController {
  private var _liveStreamUrl: String? = null

  private var _state: LiveStreamState = NONE

  private var _viewerCount: Int = 0

  override val livestreamData: DyteLivestreamData
    get() =
      DyteLivestreamData(
        _liveStreamUrl,
        controllerContainer.metaController.getRoomName(),
        _state,
        _viewerCount,
      )

  private val socketEventListener =
    object : SocketServiceEventListener {
      override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
        if (payload == null) {
          DyteLogger.warn("DyteLiveStream::socketEvent $event::received payload is null")
          return
        }
        when (event) {
          RoomEvent.STARTED.id -> {
            DyteLogger.info("DyteLiveStream::onEvent::Started")
            controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamStarting)
            _state = STARTED
            val decodedPayload = LiveStreamingEvent.ADAPTER.decode(payload)
            _liveStreamUrl = decodedPayload.playback_url
            controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamStarted)
          }
          RoomEvent.STOPPED.id -> {
            DyteLogger.info("DyteLiveStream::onEvent::Stopped")
            controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamEnding)
            _state = STOPPED
            controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamEnded)
          }
          RoomEvent.ERRORED.id -> {
            DyteLogger.info("DyteLiveStream::onEvent::Errored")
            _state = ERRORED
            controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamErrored)
          }
          RoomEvent.ROOM_PEER_COUNT.id -> {
            DyteLogger.info("DyteLiveStream::onEvent::Room_Peer_Count")
            val response = RoomPeerCountResponse.ADAPTER.decode(payload)
            _viewerCount = response.count.toInt()
            controllerContainer.eventController.triggerEvent(OnViewerCountUpdated(_viewerCount))
          }
          else -> {}
        }
      }
    }

  override fun init() {
    socketService.subscribe(RoomEvent.STARTED.id, socketEventListener)
    socketService.subscribe(RoomEvent.STOPPED.id, socketEventListener)
    socketService.subscribe(RoomEvent.ERRORED.id, socketEventListener)
    socketService.subscribe(RoomEvent.ROOM_PEER_COUNT.id, socketEventListener)
    loadLiveStreamUrl()
  }

  private fun loadLiveStreamUrl() {
    serialScope.launch {
      try {
        _liveStreamUrl = controllerContainer.apiClient.getLiveStreamUrl()
        _state = STARTED
        controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamStarted)
      } catch (e: Exception) {
        _state = NONE
        DyteLogger.info("LivestreamController::loadURL::failed")
      }
    }
  }

  override suspend fun loadData() {
    socketService.send(RoomEvent.GET_STAGE_PEERS.id, null)
    socketService.send(RoomEvent.GET_STAGE_REQUESTS.id, null)
  }

  override suspend fun start() {
    DyteLogger.info("DyteLiveStream::start::")
    controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamStarting)
    try {
      controllerContainer.apiClient.startLiveStream()
    } catch (e: Exception) {
      controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamErrored)
      DyteLogger.error("LivestreamController::start::failed", e)
      return
    }
  }

  override suspend fun stop() {
    DyteLogger.info("DyteLiveStream::stop::")
    controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamEnding)
    try {
      controllerContainer.apiClient.stopLiveStream()
    } catch (e: Exception) {
      controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamErrored)
      DyteLogger.error("LivestreamController::stop::failed", e)
      return
    }
  }
}

internal interface ILiveStreamController {
  val livestreamData: DyteLivestreamData

  suspend fun start()

  suspend fun stop()

  suspend fun loadData()
}

enum class LiveStreamState {
  NONE,
  STARTING,
  STARTED,
  STOPPING,
  STOPPED,
  ERRORED;

  companion object {
    fun fromMap(stage: String): LiveStreamState {
      return when (stage) {
        "none" -> NONE
        "starting" -> STARTING
        "started" -> STARTED
        "stopping" -> STOPPING
        "stopped" -> STOPPED
        "errored" -> ERRORED
        else -> NONE
      }
    }
  }

  fun toMap(): Map<String, String> {
    val strState =
      when (this) {
        NONE -> "none"
        STARTING -> "stopping"
        STARTED -> "started"
        STOPPING -> "stopping"
        STOPPED -> "stopped"
        ERRORED -> "errored"
      }

    val map: HashMap<String, String> = HashMap()

    map["state"] = strState

    return map
  }
}

data class DyteLivestreamData(
  val liveStreamUrl: String?,
  val roomName: String,
  val state: LiveStreamState,
  val viewerCount: Int,
) {
  companion object {
    fun fromMap(map: Map<String, Any?>): DyteLivestreamData {

      val lvsUrl = map["livestreamUrl"] as String
      val roomName = map["roomName"] as String
      val viewerCount = map["viewerCount"] as Int
      val state = LiveStreamState.fromMap(map["state"] as String)

      return DyteLivestreamData(lvsUrl, roomName, state, viewerCount)
    }
  }

  fun toMap(): Map<String, Any?> {
    val map = HashMap<String, Any?>()
    map["livestreamUrl"] = liveStreamUrl
    map["roomName"] = roomName
    map["viewerCount"] = viewerCount
    map["state"] = state.toMap()["state"]
    return map
  }
}
