package io.dyte.core.chat.socketservice

import io.dyte.core.DyteError
import io.dyte.core.Result
import io.dyte.core.chat.BaseChatController
import io.dyte.core.chat.ChatFileUploader
import io.dyte.core.chat.MessageType
import io.dyte.core.chat.models.GetMessagesResult
import io.dyte.core.chat.models.SocketServiceFileMessage
import io.dyte.core.feat.DyteChatMessage
import io.dyte.core.feat.DyteFileMessage
import io.dyte.core.feat.DyteImageMessage
import io.dyte.core.feat.DyteTextMessage
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.observability.DyteLogger
import io.dyte.core.platform.IDytePlatformUtilsProvider
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import io.dyte.core.socket.socketservice.SocketServiceUtils
import io.dyte.core.socket.socketservice.SockratesSocketService
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.jsonPrimitive
import kotlinx.serialization.json.long
import socket.chat.ChatMessage
import socket.chat.GetChatMessagesResponse
import socket.chat.GetPaginatedChatMessageRoomRequest
import socket.chat.GetPaginatedChatMessageRoomResponse
import socket.chat.SendChatMessageToPeersRequest
import socket.chat.SendChatMessageToPeersResponse
import socket.chat.SendChatMessageToRoomRequest
import socket.chat.SendChatMessageToRoomResponse

internal class ChatSocketServiceController(
  private val platformUtilsProvider: IDytePlatformUtilsProvider,
  private val selfParticipant: DyteSelfParticipant,
  private val canSendPrivateChatText: Boolean,
  private val canSendPrivateChatFile: Boolean,
  private val socketService: SockratesSocketService,
  chatFileUploader: ChatFileUploader,
  scope: CoroutineScope,
) :
  BaseChatController(
    platformUtilsProvider,
    selfParticipant,
    canSendPrivateChatText,
    canSendPrivateChatFile,
    chatFileUploader,
    scope,
  ) {
  private val roomMessageEventListener =
    object : SocketServiceEventListener {
      override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
        scope.launch { payload?.let { onSocketServiceMessageToRoom(payload) } }
      }
    }

  private val peerMessageEventListener =
    object : SocketServiceEventListener {
      override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
        scope.launch { payload?.let { onSocketServiceMessageToPeers(payload) } }
      }
    }

  init {
    setupEvents()
  }

  override suspend fun getChatMessages(): List<DyteChatMessage> {
    val response =
      socketService.requestResponse(event = ChatSocketEvent.GET_MESSAGES.id, payload = null)

    response?.let {
      val getChatMessagesResponse = GetChatMessagesResponse.ADAPTER.decode(response)
      val socketChatMessages = mutableListOf<DyteChatMessage>()
      for (socketServiceChatMessage in getChatMessagesResponse.messages) {
        try {
          val socketChatMessage = fromSocketServiceChatMessage(socketServiceChatMessage)
          socketChatMessages.add(socketChatMessage)
        } catch (e: Exception) {
          DyteLogger.error("ChatSocketHandler::getChatMessages::error", e)
        }
      }
      return socketChatMessages.toList()
    }
    return emptyList()
  }

  override suspend fun sendTextMessageSocket(
    message: String,
    peerIds: List<String>,
  ): Result<Unit, DyteError> {
    return if (peerIds.isNotEmpty()) {
      sendMessageToPeers(MessageType.TEXT.type, message, peerIds)
    } else {
      sendMessageToRoom(MessageType.TEXT.type, message)
    }
  }

  override suspend fun sendImageMessageSocket(
    path: String,
    peerIds: List<String>,
  ): Result<Unit, DyteError> {
    return if (peerIds.isNotEmpty()) {
      sendMessageToPeers(MessageType.IMAGE.type, path, peerIds)
    } else {
      sendMessageToRoom(MessageType.IMAGE.type, path)
    }
  }

  private fun fromSocketServiceChatMessage(socketServerChatMessage: ChatMessage): DyteChatMessage {
    when (socketServerChatMessage.payload_type) {
      0 -> {
        return DyteTextMessage(
          message = socketServerChatMessage.payload,
          userId = socketServerChatMessage.user_id,
          displayName = socketServerChatMessage.display_name,
          time = parseTime(socketServerChatMessage.created_at),
          pluginId = null,
          read = false,
        )
      }
      1 -> {
        return DyteImageMessage(
          link = socketServerChatMessage.payload,
          userId = socketServerChatMessage.user_id,
          displayName = socketServerChatMessage.display_name,
          time = parseTime(socketServerChatMessage.created_at),
          pluginId = null,
          read = false,
        )
      }
      2 -> {
        val fileMessagePayload = Json.parseToJsonElement(socketServerChatMessage.payload).jsonObject

        return DyteFileMessage(
          name = fileMessagePayload.getValue("name").jsonPrimitive.content,
          size = fileMessagePayload.getValue("size").jsonPrimitive.long,
          link = fileMessagePayload.getValue("link").jsonPrimitive.content,
          userId = socketServerChatMessage.user_id,
          displayName = socketServerChatMessage.display_name,
          time = parseTime(socketServerChatMessage.created_at),
          pluginId = null,
          read = false,
        )
      }
      else -> {
        throw UnsupportedOperationException(
          "Message type ${socketServerChatMessage.payload_type} not supported"
        )
      }
    }
  }

  override suspend fun sendFileMessageSocket(
    fileUrl: String,
    fileName: String,
    fileSize: Long,
    peerIds: List<String>,
  ): Result<Unit, DyteError> {
    val message = SocketServiceFileMessage(fileUrl, fileName, fileSize)
    return if (peerIds.isNotEmpty()) {
      sendMessageToPeers(MessageType.FILE.type, Json.encodeToString(message), peerIds)
    } else {
      sendMessageToRoom(MessageType.FILE.type, Json.encodeToString(message))
    }
  }

  override suspend fun sendFileMessageSocket(
    fileUrl: String,
    fileName: String,
    fileSize: Long,
  ): Result<Unit, DyteError> {
    val message = SocketServiceFileMessage(fileUrl, fileName, fileSize)
    sendMessageToRoom(MessageType.FILE.type, Json.encodeToString(message))
    return Result.Success(Unit)
  }

  override suspend fun getMessagesPaginated(
    timestamp: Long,
    size: Int,
    offset: Int,
    reversed: Boolean,
    channelId: String?,
  ): Result<GetMessagesResult, String> {
    DyteLogger.info("ChatController.getMessagesPaginated")

    // TODO: Confirm if we should check any permission here or not

    val getPaginatedChatMessageRoomRequest =
      GetPaginatedChatMessageRoomRequest(timestamp, size, offset, reversed, channelId)
    val socketResponse =
      try {
        socketService.requestResponse(
          event = ChatSocketEvent.GET_PAGINATED_MESSAGES.id,
          payload =
            GetPaginatedChatMessageRoomRequest.ADAPTER.encode(getPaginatedChatMessageRoomRequest),
        )
      } catch (e: Exception) {
        DyteLogger.error("ChatController::getMessagesPaginated::socket request failed ${e.message}")
        return Result.Failure(e.message ?: "Failed to request messages from server")
      }
        ?: kotlin.run {
          DyteLogger.warn("ChatController::getMessagesPaginated::socket response is null")
          return Result.Success(GetMessagesResult(emptyList(), false))
        }

    val getPaginatedChatMessageRoomResponse =
      try {
        GetPaginatedChatMessageRoomResponse.ADAPTER.decode(socketResponse)
      } catch (e: Exception) {
        DyteLogger.warn(
          "ChatController::getMessagesPaginated::failed to decode GetPaginatedChatMessageRoomResponse"
        )
        return Result.Success(GetMessagesResult(emptyList(), false))
      }

    val messages =
      getPaginatedChatMessageRoomResponse.messages.map { fromSocketServiceChatMessage(it) }
    return Result.Success(GetMessagesResult(messages, getPaginatedChatMessageRoomResponse.next))
  }

  private suspend fun sendMessageToRoom(
    messageType: Int,
    message: String,
  ): Result<Unit, DyteError> {
    val sendChatMessageToRoomRequest =
      SendChatMessageToRoomRequest(payload_type = messageType, payload = message)
    socketService.send(
      event = ChatSocketEvent.SEND_MESSAGE_TO_ROOM.id,
      payload = SendChatMessageToRoomRequest.ADAPTER.encode(sendChatMessageToRoomRequest),
    )
    return Result.Success(Unit)
  }

  private suspend fun sendMessageToPeers(
    messageType: Int,
    message: String,
    userIDs: List<String>,
  ): Result<Unit, DyteError> {
    val sendChatMessageToPeerRequest =
      SendChatMessageToPeersRequest(
        payload_type = messageType,
        payload = message,
        peer_ids = userIDs,
      )
    socketService.send(
      event = ChatSocketEvent.SEND_MESSAGE_TO_PEERS.id,
      payload = SendChatMessageToPeersRequest.ADAPTER.encode(sendChatMessageToPeerRequest),
    )
    return Result.Success(Unit)
  }

  private fun unSubscribeToSocketServiceChatEvents() {
    unSubscribeToSendMessageToRoomEvent()
  }

  private fun setupEvents() {
    socketService.subscribe(
      event = ChatSocketEvent.SEND_MESSAGE_TO_ROOM.id,
      listener = roomMessageEventListener,
    )
    socketService.subscribe(
      event = ChatSocketEvent.SEND_MESSAGE_TO_PEERS.id,
      listener = peerMessageEventListener,
    )
  }

  private fun unSubscribeToSendMessageToRoomEvent() {
    socketService.unsubscribe(
      event = ChatSocketEvent.SEND_MESSAGE_TO_ROOM.id,
      listener = roomMessageEventListener,
    )
  }

  private fun onSocketServiceMessageToPeers(payload: ByteArray) {
    // TODO: Check if this event would ever be applicable to ChatController in a CHAT meeting.
    // If not then we can directly return without decoding the payload if meetingType is CHAT.
    try {
      val sendChatMessageToPeerResponse = SendChatMessageToPeersResponse.ADAPTER.decode(payload)
      val message = sendChatMessageToPeerResponse.message ?: return
      if (message.channel_id != null) {
        return
      }
      val convertedMessage = fromSocketServiceChatMessage(message)
      handleMessage(convertedMessage)
    } catch (e: Exception) {
      DyteLogger.error("ChatSocketHandler::getPrivateChatMessagesSocketService::error", e)
    }
  }

  private fun onSocketServiceMessageToRoom(payload: ByteArray) {
    // TODO: Check if this event would ever be applicable to ChatController in a CHAT meeting.
    // If not then we can directly return without decoding the payload if meetingType is CHAT.
    try {
      val sendChatMessageToRoomResponse = SendChatMessageToRoomResponse.ADAPTER.decode(payload)
      val message = sendChatMessageToRoomResponse.message ?: return
      if (message.channel_id != null) {
        return
      }
      val convertedMessage = fromSocketServiceChatMessage(message)
      handleMessage(convertedMessage)
    } catch (e: Exception) {
      DyteLogger.error("ChatSocketHandler::getChatMessagesSocketService::error", e)
    }
  }
}

internal enum class ChatSocketEvent(slug: Int) {
  GET_MESSAGES(0),
  SEND_MESSAGE_TO_ROOM(1),
  SEND_MESSAGE_TO_PEERS(2),
  EDIT_MESSAGE(3),
  DELETE_MESSAGE(4),
  GET_PAGINATED_MESSAGES(5),
  SEND_MESSAGE_TO_CHANNEL(6),
  SEARCH_CHANNEL_MESSAGES(7),
  GET_ALL_CHAT_CHANNELS(8),
  MARK_CHANNEL_INDEX_AS_READ(9);

  private val base: Int = 1
  val id: Int = SocketServiceUtils.generateSocketEventId(base, slug)
}
