package io.dyte.core.utils

import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonArray
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonNull
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive
import kotlinx.serialization.json.booleanOrNull
import kotlinx.serialization.json.buildJsonArray
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.doubleOrNull
import kotlinx.serialization.json.floatOrNull
import kotlinx.serialization.json.int
import kotlinx.serialization.json.intOrNull
import kotlinx.serialization.json.longOrNull
import okio.ByteString
import okio.ByteString.Companion.encodeUtf8

internal object JsonUtils {
  fun Json.encodeToByteString(jsonElement: JsonElement): ByteString {
    return encodeToString(jsonElement).encodeUtf8()
  }

  fun Json.decodeFromByteString(byteString: ByteString): JsonElement {
    return parseToJsonElement(byteString.utf8())
  }

  fun JsonPrimitive.getPrimitiveValue(): Any? {
    if (this.isString) {
      return this.content
    }

    this.booleanOrNull?.let { booleanValue ->
      return booleanValue
    }

    this.intOrNull?.let { intValue ->
      return intValue
    }

    this.doubleOrNull?.let { doubleValue ->
      return doubleValue
    }

    this.floatOrNull?.let { floatValue ->
      return floatValue
    }

    this.longOrNull?.let { longValue ->
      return longValue
    }

    return try {
      this.int.toUInt()
    } catch (e: NumberFormatException) {
      null
    }
  }

  fun List<*>.toJsonArray(): JsonArray {
    return buildJsonArray {
      for (element in this@toJsonArray) {
        val jsonElement = serializeAny(element)
        if (jsonElement != null) {
          add(jsonElement)
        }
      }
    }
  }

  fun Array<*>.toJsonArray(): JsonArray {
    return buildJsonArray {
      for (element in this@toJsonArray) {
        val jsonElement = serializeAny(element)
        if (jsonElement != null) {
          add(jsonElement)
        }
      }
    }
  }

  fun Map<*, *>.toJsonObject(): JsonObject {
    return buildJsonObject {
      for (entry in this@toJsonObject) {
        val key = entry.key
        if (key !is String) {
          continue
        }
        val jsonElement = serializeAny(entry.value)
        if (jsonElement != null) {
          put(key, jsonElement)
        }
      }
    }
  }

  fun String.toJsonPrimitive(): JsonPrimitive = JsonPrimitive(this)

  fun Number.toJsonPrimitive(): JsonPrimitive = JsonPrimitive(this)

  fun Char.toJsonPrimitive(): JsonPrimitive = JsonPrimitive(this.toString())

  fun Boolean.toJsonPrimitive(): JsonPrimitive = JsonPrimitive(this)

  fun serializeAny(data: Any?): JsonElement? {
    // Try-catch is just a safety measure
    return try {
      when (data) {
        null -> JsonNull
        is Map<*, *> -> {
          data.toJsonObject()
        }
        is List<*> -> {
          data.toJsonArray()
        }
        is Array<*> -> {
          data.toJsonArray()
        }
        is Boolean -> {
          data.toJsonPrimitive()
        }
        is String -> {
          data.toJsonPrimitive()
        }
        is Char -> {
          data.toJsonPrimitive()
        }
        is Number -> {
          data.toJsonPrimitive()
        }
        else -> {
          null
        }
      }
    } catch (e: Exception) {
      null
    }
  }

  fun deserialize(jsonElement: JsonElement): Any? {
    return when (jsonElement) {
      is JsonNull -> null
      is JsonPrimitive -> jsonElement.getPrimitiveValue()
      is JsonObject -> jsonElement.toKotlinMap()
      is JsonArray -> jsonElement.toKotlinList()
    }
  }

  fun JsonObject.toKotlinMap(): Map<String, Any?> {
    val resultMap = mutableMapOf<String, Any?>()
    for (entry in this) {
      // TODO: skip the unsupported types
      resultMap[entry.key] = deserialize(entry.value)
    }
    return resultMap
  }

  fun JsonArray.toKotlinList(): List<Any?> {
    val resultList = mutableListOf<Any?>()
    for (element in this) {
      // TODO: skip the unsupported types
      val deserializedElement = deserialize(element)
      resultList.add(deserializedElement)
    }
    return resultList
  }
}
