package io.dyte.core.models

import io.dyte.core.controllers.DyteStageStatus
import io.dyte.core.controllers.IMetaController
import io.dyte.core.controllers.ParticipantController
import io.dyte.core.controllers.PermissionController
import io.dyte.core.controllers.PermissionType
import io.dyte.core.controllers.SelfController
import io.dyte.core.feat.ParticipantFlags
import io.dyte.core.host.IHostController
import io.dyte.core.media.DyteVideoMiddleware
import io.dyte.core.media.DyteVideoMiddlewares
import io.dyte.core.models.DyteMediaPermission.ALLOWED
import io.dyte.core.models.WaitListStatus.ACCEPTED
import io.dyte.core.models.WaitListStatus.NONE
import io.dyte.core.network.info.DyteMediaRoomType
import io.dyte.core.network.info.ParticipantInfo
import io.dyte.core.platform.VideoView
import kotlinx.coroutines.*

// TODO : add enabledPreview, disablePreview
// TODO : getAudioDevices, getVideoDevices, getSpeakerDevices, getDeviceBydId, getAllDevices,
// setDevice
// TODO : setIsPinned, pin, unpin,
// TODO : add roomJoined, remove userId
// TODO : wrap audioTrack and videoTrack inside preview
// TODO : we can get rid of audioTrack, its not been used anywhere.
class DyteSelfParticipant
internal constructor(
  id: String,
  override val userId: String,
  override var name: String,
  override val picture: String?,
  override val isHost: Boolean,
  override val clientSpecificId: String?,
  override val flags: ParticipantFlags,
  override val presetName: String,
  internal val selfController: SelfController,
  private val participantInfo: ParticipantInfo,
  participantController: ParticipantController,
  private val metaController: IMetaController,
  hostController: IHostController,
  private val permissionController: PermissionController,
) :
  DyteJoinedMeetingParticipant(
    id,
    userId,
    name,
    picture,
    isHost,
    clientSpecificId,
    flags,
    presetName,
    participantController,
    hostController,
    participantInfo.presetInfo.permissions,
  ) {
  val roomJoined: Boolean
    get() = selfController._roomJoined

  override val id: String
    get() = metaController.getPeerId()

  val permissions = participantInfo.presetInfo.permissions

  val presetInfo = participantInfo.presetInfo

  val screenshareEnabled: Boolean
    get() = _screenShareTrack != null

  val mediaRoomType: DyteMediaRoomType
    get() = permissions.mediaRoomType

  internal var _waitListStatus: WaitListStatus = NONE
  val waitListStatus: WaitListStatus
    get() = _waitListStatus

  val isCameraPermissionGranted: Boolean
    get() = permissionController.isPermissionGrated(PermissionType.CAMERA)

  val isMicrophonePermissionGranted: Boolean
    get() = permissionController.isPermissionGrated(PermissionType.MICROPHONE)

  val designToken = participantInfo.presetInfo.ui

  fun enableAudio() {
    serialScope.launch { selfController.enableAudio() }
  }

  /** This method is used to mute the local participant's audio. */
  override fun disableAudio() {
    serialScope.launch { selfController.disableAudio() }
  }

  /** This method is used to start streaming the local participant's video to the meeting. */
  fun enableVideo() {
    serialScope.launch { selfController.enableVideo() }
  }

  /** This participant is used to disable the local participant's video. */
  override fun disableVideo() {
    serialScope.launch { selfController.disableVideo() }
  }

  /** This method starts capturing the screen and shares it to the remote participants */
  fun enableScreenshare() {
    selfController.enableScreenshare()
  }

  /** This participant is used to disable the local participant's screenshare. */
  fun disableScreenshare() {
    serialScope.launch { selfController.disableScreenshare() }
  }

  fun getSelfPreview(): VideoView {
    return selfController.getSelfPreview()
  }

  fun setDisplayName(name: String) {
    serialScope.launch { this@DyteSelfParticipant.name = name }
  }

  fun getAudioDevices(): List<DyteAudioDevice> {
    return runBlocking { selfController.getAudioDevices() }
  }

  fun getVideoDevices(): List<DyteVideoDevice> {
    return runBlocking { selfController.getVideoDevices() }
  }

  fun setAudioDevice(dyteAndroidDevice: DyteAudioDevice) {
    selfController.setDevice(dyteAndroidDevice)
  }

  fun setVideoDevice(dyteVideoDevice: DyteVideoDevice) {
    serialScope.launch { selfController.setDevice(dyteVideoDevice) }
  }

  fun switchCamera() {
    val devices = getVideoDevices()
    val device = devices.firstOrNull { it.type != getSelectedVideoDevice()?.type }
    device?.let { setVideoDevice(it) }
  }

  fun getSelectedVideoDevice(): DyteVideoDevice? {
    return selfController.getSelectedVideoDevice()
  }

  fun getSelectedAudioDevice(): DyteAudioDevice? {
    return selfController.getSelectedAudioDevice()
  }

  fun canDoParticipantHostControls(): Boolean {
    return runBlocking {
      withContext(serialScope.coroutineContext) { permissions.canDoParticipantHostControls() }
    }
  }

  fun shouldShowSetupScreen(): Boolean {
    return runBlocking {
      withContext(serialScope.coroutineContext) {
        permissions.media.audioPermission == ALLOWED ||
          permissions.media.video.permission == ALLOWED
      }
    }
  }

  fun shouldJoinMediaRoom(): Boolean {
    return if (waitListStatus == ACCEPTED) {
      true
    } else {
      selfController.getSelf().stageStatus == DyteStageStatus.ON_STAGE
    }
  }

  fun addVideoMiddleware(middleware: DyteVideoMiddleware) =
    DyteVideoMiddlewares.addVideoMiddleware(middleware)

  fun removeVideoMiddleware(middleware: DyteVideoMiddleware) =
    DyteVideoMiddlewares.removeVideoMiddleware(middleware)

  fun getVideoMiddlewares(): List<DyteVideoMiddleware> = DyteVideoMiddlewares.getVideoMiddlewares()

  override fun toMap(): Map<String, Any?> {
    val map = HashMap(super.toMap())
    map["selfPermissions"] = permissions.toMap()
    map["systemPermissions"] =
      hashMapOf(
        "isCameraPermissionGranted" to isCameraPermissionGranted,
        "isMicrophonePermissionGranted" to isMicrophonePermissionGranted,
      )

    return map
  }
}
