package io.dyte.core.controllers.polls

import io.dyte.core.feat.DytePollMessage
import io.dyte.core.feat.DytePollOption
import io.dyte.core.feat.DytePollVote
import io.dyte.core.network.info.PollPermissions
import io.dyte.core.observability.DyteLogger
import io.dyte.core.socket.IRoomNodeSocketService
import io.dyte.core.socket.SocketMessageEventListener
import io.dyte.core.socket.events.InboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.payloadmodel.InboundMeetingEvent
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPoll
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPollModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPollsModel
import kotlinx.serialization.json.JsonArray
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive

class PollsRoomNodeController(
  permissions: PollPermissions,
  private val socket: IRoomNodeSocketService,
) : BasePollsController(permissions) {
  override fun init() {
    setupEvents()
  }

  override fun setupEvents() {
    DyteLogger.info("DytePolls::using room-node polls::")
    socket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_ON_POLL,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          DyteLogger.info("DytePolls::new poll::")
          val wsPoll = event.payload as WebSocketPollModel
          onNewPoll(parsePoll(wsPoll.poll), true)
        }
      },
    )
    socket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_ON_POLLS,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          DyteLogger.info("DytePolls::new polls::")
          val wsPolls = event.payload as WebSocketPollsModel
          wsPolls.polls.forEach { onNewPoll(parsePoll(it.value), true) }
        }
      },
    )
  }

  override suspend fun loadPolls() {
    DyteLogger.info("DytePolls::loadPolls::")
    val pollsResponse =
      socket.sendMessageParsed<WebSocketPollsModel>(OutboundMeetingEventType.GET_POLLS, null)
    pollsResponse.polls.values.forEach { onNewPoll(parsePoll(it), false) }
    if (polls.isNotEmpty()) {
      emitEvent { it.onPollUpdates(polls) }
    }
  }

  override suspend fun sendPoll(
    question: String,
    options: List<String>,
    anonymous: Boolean,
    hideVotes: Boolean,
  ) {
    DyteLogger.info("DytePolls::sendPoll::")
    val content = HashMap<String, JsonElement>()
    content["question"] = JsonPrimitive(question)
    content["options"] = JsonArray(options.map { JsonPrimitive(it) })
    content["anonymous"] = JsonPrimitive(anonymous)
    content["hideVotes"] = JsonPrimitive(hideVotes)
    socket.sendMessage(OutboundMeetingEventType.NEW_POLL, JsonObject(content))
  }

  override suspend fun sendVote(pollMessage: DytePollMessage, pollOption: DytePollOption) {
    DyteLogger.info("DytePolls::sendVote::")
    val content = HashMap<String, JsonElement>()
    content["index"] = JsonPrimitive(pollMessage.options.indexOf(pollOption))
    content["pollId"] = JsonPrimitive(pollMessage.id)
    socket.sendMessage(OutboundMeetingEventType.VOTE_POLL, JsonObject(content))
  }

  private fun parsePoll(webSocketPoll: WebSocketPoll): DytePollMessage {
    val options = ArrayList<DytePollOption>()
    webSocketPoll.options.forEach { option ->
      val votes = ArrayList<DytePollVote>()
      val updateOption = DytePollOption(option.text, votes, option.count)
      option.votes.forEach { vote -> votes.add(DytePollVote(vote.id, vote.name)) }
      options.add(updateOption)
    }
    return DytePollMessage(
      webSocketPoll.id,
      webSocketPoll.question,
      webSocketPoll.anonymous,
      webSocketPoll.hideVotes,
      webSocketPoll.createdBy,
      options,
    )
  }
}
