package io.dyte.core.chat

import io.dyte.core.feat.DyteChatMessage
import io.dyte.core.feat.DyteFileMessage
import io.dyte.core.feat.DyteImageMessage
import io.dyte.core.feat.DyteTextMessage
import io.dyte.core.platform.IDytePlatformUtils
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.jsonPrimitive
import kotlinx.serialization.json.long
import socket.chat.ChatMessage as SocketServiceChatMessage

internal class ChatMessageMapper(private val timeFormatter: IDytePlatformUtils) {
  fun toDyteChatMessage(socketChatMessage: SocketServiceChatMessage): DyteChatMessage {
    when (socketChatMessage.payload_type) {
      0 -> {
        return DyteTextMessage(
          message = socketChatMessage.payload,
          userId = socketChatMessage.user_id,
          displayName = socketChatMessage.display_name,
          time = parseTime(socketChatMessage.created_at) ?: error("Message time is null"),
          pluginId = null,
          read = false,
          channelId = socketChatMessage.channel_id,
        )
      }
      1 -> {
        return DyteImageMessage(
          link = socketChatMessage.payload,
          userId = socketChatMessage.user_id,
          displayName = socketChatMessage.display_name,
          time = parseTime(socketChatMessage.created_at) ?: error("Message time is null"),
          pluginId = null,
          read = false,
          channelId = socketChatMessage.channel_id,
        )
      }
      2 -> {
        val fileMessagePayload = Json.parseToJsonElement(socketChatMessage.payload).jsonObject

        return DyteFileMessage(
          name = fileMessagePayload.getValue("name").jsonPrimitive.content,
          size = fileMessagePayload.getValue("size").jsonPrimitive.long,
          link = fileMessagePayload.getValue("link").jsonPrimitive.content,
          userId = socketChatMessage.user_id,
          displayName = socketChatMessage.display_name,
          time = parseTime(socketChatMessage.created_at) ?: error("Message time is null"),
          pluginId = null,
          read = false,
          channelId = socketChatMessage.channel_id,
        )
      }
      else -> {
        throw UnsupportedOperationException(
          "Message type ${socketChatMessage.payload_type} not supported"
        )
      }
    }
  }

  private fun parseTime(t: Long): String? {
    return try {
      timeFormatter.getUserDisplayableTime(t.times(1000))
    } catch (e: Exception) {
      return null
    }
  }
}
