package io.dyte.core.host

import io.dyte.core.DyteError
import io.dyte.core.HostControlErrorCodes.Kick_All
import io.dyte.core.HostControlErrorCodes.Kick_Peer
import io.dyte.core.HostControlErrorCodes.Mute_All_Audio
import io.dyte.core.HostControlErrorCodes.Mute_All_Video
import io.dyte.core.HostControlErrorCodes.Mute_Audio
import io.dyte.core.HostControlErrorCodes.Mute_Video
import io.dyte.core.HostControlErrorCodes.Pin_Peer
import io.dyte.core.HostControlErrorCodes.Unpin_Peer
import io.dyte.core.Result
import io.dyte.core.Result.Failure
import io.dyte.core.Result.Success
import io.dyte.core.network.info.HostPermissions
import io.dyte.core.observability.DyteLogger
import io.dyte.core.room.RoomSocketHandler
import io.dyte.core.socket.IRoomNodeSocketService
import io.dyte.core.socket.events.OutboundMeetingEventType
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.put

internal class HostRoomNodeController(
  private val selfHostPermissions: HostPermissions,
  private val socketController: IRoomNodeSocketService,
  private val roomSocketHandler: RoomSocketHandler,
) : IHostController {
  override suspend fun kickPeer(peerId: String): Result<Unit, DyteError> {
    DyteLogger.info("DyteParticipant::kickPeer::$peerId")
    return try {
      // Kick peer from room-node
      val kickPeerPayload = buildJsonObject { put("id", peerId) }
      val result = socketController.sendPacket(OutboundMeetingEventType.KICK_PEER, kickPeerPayload)
      val roomNodeKickPeerResult = evaluate(result, DyteError(Kick_Peer))
      if (roomNodeKickPeerResult is Result.Failure) {
        return roomNodeKickPeerResult
      }

      // Kick peer from socketService room
      roomSocketHandler.kickPeer(peerId)
      Result.Success(Unit)
    } catch (e: Exception) {
      DyteLogger.error("DyteParticipant::kickPeer::failed", e)
      Result.Failure(DyteError(Kick_Peer, "DyteParticipant::kick_peer_failed::${e.message}"))
    }
  }

  override suspend fun kickAll(): Result<Unit, DyteError> {
    DyteLogger.info("DyteParticipants::kickAll")
    if (!selfHostPermissions.canKickParticipant) {
      DyteLogger.error("HostController::kickAll::self not permitted")
      return Result.Failure(
        DyteError(Kick_All, "Unauthorized: User does not have permission to kick peers.")
      )
    }
    return try {
      val result = socketController.sendPacket(OutboundMeetingEventType.KICK_ALL, null)
      val roomNodeKickAllResult = evaluate(result, DyteError(Kick_All))
      if (roomNodeKickAllResult is Result.Failure) {
        return roomNodeKickAllResult
      }

      // Kick all from socketService room
      roomSocketHandler.kickAll()
      Result.Success(Unit)
    } catch (e: Exception) {
      DyteLogger.error("DyteParticipant::kickAll::failed", e)
      Result.Failure(DyteError(Kick_Peer, "DyteParticipants::kick_all_failed:${e.message}"))
    }
  }

  override suspend fun muteAllAudio(allowUnmute: Boolean?): Result<Unit, DyteError> {
    DyteLogger.info("DyteHost::muteAllAudio::")
    val content = HashMap<String, JsonElement>()
    content["allowUnMute"] = JsonPrimitive(allowUnmute)
    val result =
      socketController.sendPacket(OutboundMeetingEventType.MUTE_ALL_AUDIO, JsonObject(content))
    return evaluate(result, DyteError(Mute_All_Audio))
  }

  override suspend fun muteAllVideo(): Result<Unit, DyteError> {
    DyteLogger.info("DyteHost::muteAllVideo::")
    val result = socketController.sendPacket(OutboundMeetingEventType.MUTE_ALL_VIDEO, null)
    return evaluate(result, DyteError(Mute_All_Video))
  }

  override suspend fun muteAudio(peerId: String): Result<Unit, DyteError> {
    DyteLogger.info("DyteHost::muteAudio::$peerId")
    val content = HashMap<String, JsonElement>()
    content["id"] = JsonPrimitive(peerId)
    val result =
      socketController.sendPacket(OutboundMeetingEventType.MUTE_PEER_AUDIO, JsonObject(content))
    return evaluate(result, DyteError(Mute_Audio))
  }

  override suspend fun muteVideo(peerId: String): Result<Unit, DyteError> {
    DyteLogger.info("DyteHost::muteVideo::$peerId")
    val content = HashMap<String, JsonElement>()
    content["id"] = JsonPrimitive(peerId)
    val result =
      socketController.sendPacket(OutboundMeetingEventType.MUTE_PEER_VIDEO, JsonObject(content))
    return evaluate(result, DyteError(Mute_Video))
  }

  override suspend fun pinPeer(peerId: String): Result<Unit, DyteError> {
    DyteLogger.info("DyteHost::pinPeer::$peerId")
    val content = HashMap<String, JsonElement>()
    content["id"] = JsonPrimitive(peerId)
    val result =
      socketController.sendPacket(OutboundMeetingEventType.PIN_PARTICIPANT, JsonObject(content))
    return evaluate(result, DyteError(Pin_Peer))
  }

  override suspend fun unpinPeer(): Result<Unit, DyteError> {
    DyteLogger.info("DyteHost::unpinPeer::")
    val content = HashMap<String, JsonElement>()
    val nullString: String? = null
    content["id"] = JsonPrimitive(nullString)
    val result =
      socketController.sendPacket(OutboundMeetingEventType.PIN_PARTICIPANT, JsonObject(content))
    return evaluate(result, DyteError(Unpin_Peer))
  }

  private fun evaluate(
    result: Result<String, DyteError>,
    localError: DyteError,
  ): Result<Unit, DyteError> {
    return when (result) {
      is Failure -> {
        Failure(localError)
      }
      is Success -> {
        Success(Unit)
      }
    }
  }
}
