package io.dyte.core.controllers

import io.dyte.core.Result.Failure
import io.dyte.core.Result.Success
import io.dyte.core.events.InternalEvents
import io.dyte.core.media.IDyteCommonMediaUtils
import io.dyte.core.media.IDyteMediaSoupUtils
import io.dyte.core.models.DyteMeetingType.LIVESTREAM
import io.dyte.core.models.DyteMeetingType.WEBINAR
import io.dyte.core.models.WaitListStatus.WAITING
import io.dyte.core.network.BaseApiService
import io.dyte.core.network.models.IceServerData
import io.dyte.core.observability.DyteLogger
import io.dyte.core.socket.SocketMessageEventListener
import io.dyte.core.socket.events.InboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType.CREATE_WEB_RTC_TRANSPORT
import io.dyte.core.socket.events.OutboundMeetingEventType.GET_ROOM_STATE
import io.dyte.core.socket.events.OutboundMeetingEventType.GET_ROUTER_RTP_CAPABILITIES
import io.dyte.core.socket.events.payloadmodel.InboundMeetingEvent
import io.dyte.core.socket.events.payloadmodel.inbound.Device
import io.dyte.core.socket.events.payloadmodel.outbound.CreateWebRTCTransportPayloadRequestModel
import io.dyte.core.socket.events.payloadmodel.outbound.JoinRoomPayloadRequestModel
import io.dyte.core.socket.events.payloadmodel.outbound.RouterCapabilitiesModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebRtcCreateTransportModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketJoinRoomModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketRoomStateModel
import kotlinx.coroutines.MainScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.withContext
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.encodeToJsonElement

internal class RoomNodeController(
  controllerContainer: IControllerContainer,
  private val mediaUtils: IDyteCommonMediaUtils,
) : IRoomNodeController, BaseController(controllerContainer), InternalEvents {

  private lateinit var iceServers: List<IceServerData>
  private lateinit var jsonParser: Json

  private lateinit var routerCapabilities: RouterCapabilitiesModel
  private lateinit var roomState: WebSocketRoomStateModel
  private lateinit var webRtcCreateTransportModelConsumer: WebRtcCreateTransportModel
  private lateinit var webRtcCreateTransportModelProducer: WebRtcCreateTransportModel
  private lateinit var joinRoom: WebSocketJoinRoomModel

  // Cached callbacks coming from clients that use the callback version of joinRoom method.
  private var onRoomJoined: (() -> Unit)? = null
  private var onRoomJoinFailed: (() -> Unit)? = null

  override fun connectToRoomNode() {
    super.connectToRoomNode()
    serialScope.launch {
      controllerContainer.roomNodeSocketService.connect()
      joinRoom()
    }
  }

  override fun disconnectFromRoomNode() {
    super.disconnectFromRoomNode()
    // should we also call videoUtils().destroyAll() here?
    runBlocking(serialScope.coroutineContext) {
      controllerContainer.sfuUtils.leaveCall()
      controllerContainer.roomNodeSocketService.disconnect()
      leaveRoom()
    }
  }

  override fun onWaitlistEntryRejected() {
    super.onWaitlistEntryRejected()
    this@RoomNodeController.onRoomJoinFailed?.let { onRoomJoinedFailedCallback ->
      MainScope().launch { onRoomJoinedFailedCallback.invoke() }
      this@RoomNodeController.onRoomJoinFailed = null
    }
  }

  override fun init() {
    jsonParser = BaseApiService.json
    setupMediaListeners()
    controllerContainer.roomNodeSocketService.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_STARTED_PRESENTING,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          connectToMediaProduction()
          if (controllerContainer.selfController.getSelf().audioEnabled) {
            if (!controllerContainer.metaController.getIsHive())
              controllerContainer.roomNodeSocketService.sendMessage(
                OutboundMeetingEventType.UN_MUTE_SELF_AUDIO,
                null,
              )
          }
        }
      },
    )
    controllerContainer.roomNodeSocketService.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_STOPPED_PRESENTING,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          controllerContainer.selfController.disableAudio()
          controllerContainer.selfController.disableVideo()
        }
      },
    )
    controllerContainer.roomNodeSocketService.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_KICKED,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          mediaUtils.dispose()
          controllerContainer.selfController.onRemovedFromMeeting()
          leaveRoom()
        }
      },
    )
    serialScope.launch {
      try {
        iceServers = getICEServers()
      } catch (e: Exception) {
        DyteLogger.error("RoomNodeController::init::ice_server_fetch_failed", e)
      }
    }
  }

  override suspend fun joinRoom(onRoomJoined: (() -> Unit)?, onRoomJoinFailed: (() -> Unit)?) {
    withContext(serialScope.coroutineContext) {
      DyteLogger.info("SelfController.joinRoom")

      // When joinRoom is called internally from WaitlistController, we do not want to clear the
      // cached callbacks.
      if (this@RoomNodeController.onRoomJoined == null) {
        this@RoomNodeController.onRoomJoined = onRoomJoined
      }
      if (this@RoomNodeController.onRoomJoinFailed == null) {
        this@RoomNodeController.onRoomJoinFailed = onRoomJoinFailed
      }

      if (controllerContainer.selfController.getSelf().shouldJoinMediaRoom()) {
        try {
          getRoomState()
          _joinRoom()
          if (controllerContainer.selfController.getSelf().waitListStatus == WAITING) {
            return@withContext
          }
          connectToMediaConsumption()
          connectToMediaProduction()

          DyteLogger.info("room state processed")

          DyteLogger.info(
            "can produce audio ? ${controllerContainer.selfController.canPublishAudio()} } video ? ${controllerContainer.selfController.canPublishVideo()}"
          )

          val selfTracks =
            (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)?.getSelfTrack()
              ?: Pair(null, null)
          val selfParticipant = controllerContainer.selfController.getSelf()
          // selfParticipant.audioTrack = selfTracks.first
          selfParticipant._videoTrack = selfTracks.second
          controllerContainer.selfController._roomJoined = true

          if (controllerContainer.selfController.getSelf().audioEnabled) {
            if (!controllerContainer.metaController.getIsHive())
              controllerContainer.sfuUtils.unmuteSelfAudio()
            controllerContainer.roomNodeSocketService.sendMessage(
              OutboundMeetingEventType.UN_MUTE_SELF_AUDIO,
              null,
            )
          }

          if (controllerContainer.metaController.getMeetingType() == LIVESTREAM) {
            controllerContainer.liveStreamController.loadData()
          }

          controllerContainer.eventController.triggerEvent(
            DyteEventType.OnMeetingRoomJoined(controllerContainer.selfController.getSelf())
          )

          controllerContainer.eventController.triggerEvent(DyteEventType.OnMetaUpdate)
          DyteLogger.info("joined!")

          if (controllerContainer.metaController.getMeetingType() == WEBINAR) {
            controllerContainer.stageController.join()
          }

          controllerContainer.pollsController.loadPolls()
          controllerContainer.chatController.loadChatMessages()
          controllerContainer.platformUtilsProvider.getMediaUtils().routeAudioFromSpeakerphone()
          controllerContainer.participantController.setPage(0)
          this@RoomNodeController.onRoomJoined?.let { onRoomJoinedCallback ->
            MainScope().launch { onRoomJoinedCallback.invoke() }
            this@RoomNodeController.onRoomJoined = null
          }
        } catch (e: Exception) {
          DyteLogger.error("RoomNodeController::join_room_error", e)
          controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRoomJoinFailed(e))
          this@RoomNodeController.onRoomJoinFailed?.let { onRoomJoinedFailedCallback ->
            MainScope().launch { onRoomJoinedFailedCallback.invoke() }
            this@RoomNodeController.onRoomJoinFailed = null
          }
        }
      } else {
        try {
          controllerContainer.platformUtilsProvider.getMediaUtils().routeAudioFromSpeakerphone()
          if (controllerContainer.metaController.getMeetingType() != LIVESTREAM) {
            getRoomState()
            _joinRoom()
            if (controllerContainer.selfController.getSelf().waitListStatus == WAITING) {
              return@withContext
            }
            controllerContainer.selfController._roomJoined = true
            connectToMediaConsumption()
            controllerContainer.participantController.handleRoomJoined(joinRoom)
            controllerContainer.participantController.setPage(0)
          }
          if (controllerContainer.selfController.getSelf().waitListStatus == WAITING) {
            return@withContext
          }
          controllerContainer.pollsController.loadPolls()
          controllerContainer.chatController.loadChatMessages()

          controllerContainer.eventController.triggerEvent(
            DyteEventType.OnMeetingRoomJoined(controllerContainer.selfController.getSelf())
          )
          controllerContainer.eventController.triggerEvent(DyteEventType.OnMetaUpdate)
          this@RoomNodeController.onRoomJoined?.let { onRoomJoinedCallback ->
            MainScope().launch { onRoomJoinedCallback.invoke() }
            this@RoomNodeController.onRoomJoined = null
          }
        } catch (e: Exception) {
          DyteLogger.error("RoomNodeController::join_room_error", e)
          controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRoomJoinFailed(e))
          this@RoomNodeController.onRoomJoinFailed?.let { onRoomJoinedFailedCallback ->
            MainScope().launch { onRoomJoinedFailedCallback.invoke() }
            this@RoomNodeController.onRoomJoinFailed = null
          }
        }
      }
    }
  }

  override fun leaveRoom() {
    controllerContainer.selfController._roomJoined = false
    onRoomJoinFailed = null
    onRoomJoined = null
  }

  private suspend fun getICEServers(): List<IceServerData> {
    try {
      return controllerContainer.apiClient.getICEServers().iceServers
    } catch (e: Exception) {
      DyteLogger.error("DyteLog: RoomNodeController::getICEServers::failed", e)
      val emptyIceServerList: List<IceServerData> = emptyList()
      return emptyIceServerList
    }
  }

  private fun setupMediaListeners() {
    controllerContainer.internalEventEmitter.addListener(
      object : InternalEvents {
        override fun connectMedia() {
          if (controllerContainer.selfController._roomJoined) {
            controllerContainer.platformUtilsProvider.getVideoUtils().destroyAll()
            serialScope.launch { joinRoom() }
          }
        }

        override fun disconnectMedia() {
          if (controllerContainer.selfController._roomJoined) {
            serialScope.launch {
              controllerContainer.sfuUtils.cleanupConsumers()
              controllerContainer.sfuUtils.cleanupProducers()
              controllerContainer.sfuUtils.cleanupTransport()
            }
            controllerContainer.platformUtilsProvider.getVideoUtils().destroyAll()
          }
        }
      }
    )
  }

  override suspend fun connectToMediaProduction() {
    val producerResponse =
      controllerContainer.roomNodeSocketService.sendPacket(
        CREATE_WEB_RTC_TRANSPORT,
        jsonParser.encodeToJsonElement(
          CreateWebRTCTransportPayloadRequestModel(
            forceTcp = false,
            producing = true,
            consuming = false,
          )
        ),
      )
    when (producerResponse) {
      is Success -> {
        webRtcCreateTransportModelProducer =
          controllerContainer.socketMessageResponseParser
            .parseResponse(producerResponse.value)
            .payload as WebRtcCreateTransportModel
        (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)?.createWebRtcTransportProd(
          webRtcCreateTransportModelProducer,
          iceServers,
        )
        DyteLogger.info("created produce transport")
      }
      else -> {}
    }
  }

  private suspend fun connectToMediaConsumption() {
    (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)?.loadRouterRtpCapabilities(
      routerCapabilities
    )

    val consumerResponse =
      controllerContainer.roomNodeSocketService.sendPacket(
        CREATE_WEB_RTC_TRANSPORT,
        jsonParser.encodeToJsonElement(
          CreateWebRTCTransportPayloadRequestModel(
            forceTcp = false,
            producing = false,
            consuming = true,
          )
        ),
      )

    when (consumerResponse) {
      is Success -> {
        webRtcCreateTransportModelConsumer =
          (controllerContainer.socketMessageResponseParser
            .parseResponse(consumerResponse.value)
            .payload as WebRtcCreateTransportModel)
        (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)?.createWebRtcTransportRecv(
          webRtcCreateTransportModelConsumer,
          iceServers,
        )
        DyteLogger.info("created receive transport")
      }
      is Failure -> {
        DyteLogger.info("receive transport creation got failed")
      }
    }
  }

  private suspend fun getRoomState() {
    val roomStateResponse =
      controllerContainer.roomNodeSocketService.sendPacket(GET_ROOM_STATE, null)

    when (roomStateResponse) {
      is Success -> {
        roomState =
          controllerContainer.socketMessageResponseParser
            .parseResponse(roomStateResponse.value)
            .payload as WebSocketRoomStateModel

        DyteLogger.info("room state received")

        /*
         * note: This step is required only in Mediasoup meetings as we get roomUUID while joining the room.
         * For Hive meetings, meetingId acts as roomUUID. Since it is available during the init operation,
         * we do not call setRoomUUID explicitly but pass the meetingId directly while initialising the Chat module.
         * */
        if (roomState.roomState.roomUUID.isNotBlank()) {
          controllerContainer.chatController.setRoomUUID(roomState.roomState.roomUUID)
        }
        // check for remote peers joined before local user
        controllerContainer.participantController.handleRoomState(roomState)
      }
      else -> {}
    }
  }

  @Throws(Exception::class)
  private suspend fun _joinRoom() {
    val routerCapabilitiesResponse =
      controllerContainer.roomNodeSocketService.sendPacket(GET_ROUTER_RTP_CAPABILITIES, null)
    when (routerCapabilitiesResponse) {
      is Success -> {
        routerCapabilities =
          (controllerContainer.socketMessageResponseParser
            .parseResponse(routerCapabilitiesResponse.value)
            .payload as RouterCapabilitiesModel)
        DyteLogger.info("received and processed router capabilities")

        val joinRoomPayload = JoinRoomPayloadRequestModel()
        joinRoomPayload.device =
          Device(
            osName = controllerContainer.platformUtilsProvider.getPlatformUtils().getOsName(),
            osVersionName =
              controllerContainer.platformUtilsProvider.getPlatformUtils().getOsVersion(),
          )
        joinRoomPayload.displayName = controllerContainer.selfController.getSelf().name
        joinRoomPayload.audioMuted = !controllerContainer.selfController.getSelf().audioEnabled
        joinRoomPayload.rtpCapabilities = routerCapabilities
        val payload = jsonParser.encodeToJsonElement(joinRoomPayload)
        val joinRoomResponse =
          controllerContainer.roomNodeSocketService.sendPacket(
            OutboundMeetingEventType.JOIN_ROOM,
            payload,
          )
        when (joinRoomResponse) {
          is Success -> {
            joinRoom =
              controllerContainer.socketMessageResponseParser
                .parseResponse(joinRoomResponse.value)
                .payload as WebSocketJoinRoomModel

            DyteLogger.info("join room done! -> $joinRoom")

            if (joinRoom.waitlisted) {
              DyteLogger.info("aww snap, I'm in waiting room!")
              controllerContainer.selfController.getSelf()._waitListStatus = WAITING
              controllerContainer.selfController.emitEvent { it.onWaitListStatusUpdate(WAITING) }
              return
            }
            controllerContainer.selfController._roomJoined = true
            controllerContainer.selfController.getSelf()._stageStatus =
              DyteStageStatus.fromWebsocketStageStatus(joinRoom.stageStatus)
            controllerContainer.participantController.handleRoomJoined(joinRoom)
            controllerContainer.internalEventEmitter.emitEvent { it.onRoomJoined(joinRoom) }
            joinRoom.startedAt?.let { startedAt ->
              controllerContainer.metaController.setMeetingStartedTimestamp(startedAt)
            } ?: kotlin.run { DyteLogger.warn("RoomNode::joinRoom::started at time is null") }
          }
          is Failure -> {
            DyteLogger.info("Not able to fetch Join room response.")
            throw Exception("Failed to send JOIN_ROOM to Room node")
          }
        }
      }
      is Failure -> {
        DyteLogger.info("Not able to fetch Router Capabilities")
        throw Exception("Failed to fetch router capabilities")
      }
    }
  }
}
