package io.dyte.core.socket

import io.dyte.core.network.BaseApiService
import io.dyte.core.socket.events.InboundMeetingEventType
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_ACTIVE_SPEAKER
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_CLOSE_CONSUMER
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_CONNECT_TRANSPORT
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_CREATE_TRANSPORT
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_DISABLE_AUDIO
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_DISABLE_VIDEO
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_GET_PAGE
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_JOIN_ROOM
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_KICKED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_MUTE_ALL_AUDIO
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_MUTE_ALL_VIDEO
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_NEW_CONSUMER
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_NO_ACTIVE_SPEAKER
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_ON_CHAT_MESSAGE
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_ON_CHAT_MESSAGES
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_ON_POLL
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_ON_POLLS
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PAUSE_CONSUMER
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PEER_ADDED_TO_STAGE
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PEER_JOINED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PEER_LEFT
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PEER_MUTED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PEER_PINNED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PEER_REJECTED_TO_JOIN_STAGE
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PEER_REMOVED_FROM_STAGE
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PEER_STARTED_PRESENTING
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PEER_STOPPED_PRESENTING
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PEER_UNMUTED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PLUGIN_DATA
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PLUGIN_DISABLED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PLUGIN_ENABLED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PLUGIN_EVENT
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PRODUCER_CLOSED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_PRODUCER_CONNECT
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_RECORDING_STARTED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_RECORDING_STOPPED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_REMOVED_FROM_STAGE
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_REQUEST_TO_JOIN_STAGE_ACCEPTED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_REQUEST_TO_JOIN_STAGE_PEER_ADDED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_REQUEST_TO_JOIN_STAGE_PEER_WITHDRAWN
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_REQUEST_TO_JOIN_STAGE_REJECTED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_RESUME_CONSUMER
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_ROOM_MESSAGE
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_ROOM_STATE
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_ROUTER_CAPABILITY
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_SELECTED_PEERS
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_STARTED_PRESENTING
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_STOPPED_PRESENTING
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_WAITLIST_ACCEPTED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_WAITLIST_PEER_ADDED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_WAITLIST_PEER_CLOSED
import io.dyte.core.socket.events.InboundMeetingEventType.WEB_SOCKET_WAITLIST_REJECTED
import io.dyte.core.socket.events.payloadmodel.BasePayloadModel
import io.dyte.core.socket.events.payloadmodel.InboundMeetingEvent
import io.dyte.core.socket.events.payloadmodel.ResponseModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketActiveSpeakerModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketChatMessage
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketChatMessagesModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConnectTransportModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerClosedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerResumedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketEmptyModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketGetPageModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketMeetingPeerUser
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerLeftModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerMuteModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerPinnedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPluginDisabled
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPluginEnabled
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPluginEvent
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPollModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPollsModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketProducerClosedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketProducerConnectModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketRoomMessage
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketSelectedPeersModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWaitlistPeerAdded
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWaitlistPeerClosed
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWebinarRequestToJoinPeerAdded
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWebinarStagePeer
import io.dyte.core.socket.events.payloadmodel.outbound.RouterCapabilitiesModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebRtcCreateTransportModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketJoinRoomModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketRoomStateModel
import kotlinx.serialization.json.JsonObject

class SocketMessageResponseParser : ISocketMessageResponseParser {
  private val jsonParser = BaseApiService.json

  override fun parseResponse(response: String): InboundMeetingEvent {
    val responseModel = jsonParser.decodeFromString(ResponseModel.serializer(), response)
    val typeStr = responseModel.type
    val payloadObj = responseModel.payload

    val eventType =
      getEventType(typeStr) ?: throw IllegalArgumentException("event type not supported $typeStr")
    val payloadModel =
      parsePayloadModel(eventType, payloadObj)
        ?: throw IllegalArgumentException("payload object not supported $payloadObj")

    return InboundMeetingEvent(eventType, payloadModel)
  }

  private fun getEventType(type: String?): InboundMeetingEventType? {
    type ?: return null

    for (event in InboundMeetingEventType.values()) {
      if (event.type == type) {
        return event
      }
    }

    return null
  }

  private fun parsePayloadModel(
    eventType: InboundMeetingEventType,
    jsonObject: JsonObject?,
  ): BasePayloadModel? {
    jsonObject ?: return null

    return when (eventType) {
      WEB_SOCKET_ROOM_STATE ->
        jsonParser.decodeFromJsonElement(WebSocketRoomStateModel.serializer(), jsonObject)
      WEB_SOCKET_ROUTER_CAPABILITY ->
        jsonParser.decodeFromJsonElement(RouterCapabilitiesModel.serializer(), jsonObject)
      WEB_SOCKET_CREATE_TRANSPORT ->
        jsonParser.decodeFromJsonElement(WebRtcCreateTransportModel.serializer(), jsonObject)
      WEB_SOCKET_JOIN_ROOM ->
        jsonParser.decodeFromJsonElement(WebSocketJoinRoomModel.serializer(), jsonObject)
      WEB_SOCKET_PEER_JOINED ->
        jsonParser.decodeFromJsonElement(WebSocketMeetingPeerUser.serializer(), jsonObject)
      WEB_SOCKET_SELECTED_PEERS ->
        jsonParser.decodeFromJsonElement(WebSocketSelectedPeersModel.serializer(), jsonObject)
      WEB_SOCKET_NEW_CONSUMER ->
        jsonParser.decodeFromJsonElement(WebSocketConsumerModel.serializer(), jsonObject)
      WEB_SOCKET_CONNECT_TRANSPORT ->
        jsonParser.decodeFromJsonElement(WebSocketConnectTransportModel.serializer(), jsonObject)
      WEB_SOCKET_PRODUCER_CONNECT ->
        jsonParser.decodeFromJsonElement(WebSocketProducerConnectModel.serializer(), jsonObject)
      WEB_SOCKET_PEER_MUTED ->
        jsonParser.decodeFromJsonElement(WebSocketPeerMuteModel.serializer(), jsonObject)
      WEB_SOCKET_PEER_UNMUTED ->
        jsonParser.decodeFromJsonElement(WebSocketPeerMuteModel.serializer(), jsonObject)
      WEB_SOCKET_PRODUCER_CLOSED ->
        jsonParser.decodeFromJsonElement(WebSocketProducerClosedModel.serializer(), jsonObject)
      WEB_SOCKET_RESUME_CONSUMER ->
        jsonParser.decodeFromJsonElement(WebSocketConsumerResumedModel.serializer(), jsonObject)
      WEB_SOCKET_CLOSE_CONSUMER ->
        jsonParser.decodeFromJsonElement(WebSocketConsumerClosedModel.serializer(), jsonObject)
      WEB_SOCKET_PAUSE_CONSUMER ->
        jsonParser.decodeFromJsonElement(WebSocketConsumerClosedModel.serializer(), jsonObject)
      WEB_SOCKET_PEER_LEFT ->
        jsonParser.decodeFromJsonElement(WebSocketPeerLeftModel.serializer(), jsonObject)
      WEB_SOCKET_ON_CHAT_MESSAGE ->
        jsonParser.decodeFromJsonElement(WebSocketChatMessage.serializer(), jsonObject)
      WEB_SOCKET_ON_CHAT_MESSAGES ->
        jsonParser.decodeFromJsonElement(WebSocketChatMessagesModel.serializer(), jsonObject)
      WEB_SOCKET_ON_POLLS ->
        jsonParser.decodeFromJsonElement(WebSocketPollsModel.serializer(), jsonObject)
      WEB_SOCKET_ON_POLL ->
        jsonParser.decodeFromJsonElement(WebSocketPollModel.serializer(), jsonObject)
      WEB_SOCKET_WAITLIST_PEER_ADDED ->
        jsonParser.decodeFromJsonElement(WebSocketWaitlistPeerAdded.serializer(), jsonObject)
      WEB_SOCKET_WAITLIST_PEER_CLOSED ->
        jsonParser.decodeFromJsonElement(WebSocketWaitlistPeerClosed.serializer(), jsonObject)
      WEB_SOCKET_GET_PAGE -> {
        jsonParser.decodeFromJsonElement(WebSocketGetPageModel.serializer(), jsonObject)
      }
      WEB_SOCKET_ACTIVE_SPEAKER -> {
        jsonParser.decodeFromJsonElement(WebSocketActiveSpeakerModel.serializer(), jsonObject)
      }
      WEB_SOCKET_PEER_PINNED -> {
        jsonParser.decodeFromJsonElement(WebSocketPeerPinnedModel.serializer(), jsonObject)
      }
      WEB_SOCKET_PLUGIN_ENABLED -> {
        jsonParser.decodeFromJsonElement(WebSocketPluginEnabled.serializer(), jsonObject)
      }
      WEB_SOCKET_PLUGIN_DISABLED -> {
        jsonParser.decodeFromJsonElement(WebSocketPluginDisabled.serializer(), jsonObject)
      }
      WEB_SOCKET_PLUGIN_EVENT,
      WEB_SOCKET_PLUGIN_DATA -> {
        WebSocketPluginEvent(jsonObject)
      }
      WEB_SOCKET_PEER_ADDED_TO_STAGE,
      WEB_SOCKET_PEER_REJECTED_TO_JOIN_STAGE,
      WEB_SOCKET_REQUEST_TO_JOIN_STAGE_PEER_WITHDRAWN,
      WEB_SOCKET_PEER_REMOVED_FROM_STAGE,
      WEB_SOCKET_PEER_STARTED_PRESENTING,
      WEB_SOCKET_PEER_STOPPED_PRESENTING -> {
        jsonParser.decodeFromJsonElement(WebSocketWebinarStagePeer.serializer(), jsonObject)
      }
      WEB_SOCKET_REQUEST_TO_JOIN_STAGE_PEER_ADDED -> {
        jsonParser.decodeFromJsonElement(
          WebSocketWebinarRequestToJoinPeerAdded.serializer(),
          jsonObject,
        )
      }
      WEB_SOCKET_ROOM_MESSAGE -> {
        WebSocketRoomMessage(jsonObject)
      }
      WEB_SOCKET_MUTE_ALL_VIDEO,
      WEB_SOCKET_MUTE_ALL_AUDIO,
      WEB_SOCKET_NO_ACTIVE_SPEAKER,
      WEB_SOCKET_RECORDING_STARTED,
      WEB_SOCKET_RECORDING_STOPPED,
      WEB_SOCKET_KICKED,
      WEB_SOCKET_DISABLE_AUDIO,
      WEB_SOCKET_REQUEST_TO_JOIN_STAGE_REJECTED,
      WEB_SOCKET_REMOVED_FROM_STAGE,
      WEB_SOCKET_REQUEST_TO_JOIN_STAGE_ACCEPTED,
      WEB_SOCKET_STARTED_PRESENTING,
      WEB_SOCKET_STOPPED_PRESENTING,
      WEB_SOCKET_DISABLE_VIDEO,
      WEB_SOCKET_WAITLIST_ACCEPTED,
      WEB_SOCKET_WAITLIST_REJECTED -> {
        jsonParser.decodeFromJsonElement(WebSocketEmptyModel.serializer(), jsonObject)
      }
    }
  }
}

interface ISocketMessageResponseParser {
  fun parseResponse(response: String): InboundMeetingEvent
}
