package io.dyte.core.plugins.socketservice

import io.dyte.core.Result
import io.dyte.core.controllers.IEventController
import io.dyte.core.feat.DytePlugin
import io.dyte.core.models.DyteJoinedMeetingParticipant
import io.dyte.core.models.DyteParticipants
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.network.info.PluginPermissions
import io.dyte.core.plugins.LocalPluginEvent
import io.dyte.core.plugins.PluginChatSender
import io.dyte.core.plugins.PluginMessage
import io.dyte.core.plugins.PluginsController
import io.dyte.core.plugins.PluginsInternalRoomJoinListenerStore
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.withContext
import kotlinx.serialization.json.JsonArray
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonNull
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.addJsonObject
import kotlinx.serialization.json.buildJsonArray
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.contentOrNull
import kotlinx.serialization.json.int
import kotlinx.serialization.json.jsonArray
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.jsonPrimitive
import kotlinx.serialization.json.put
import kotlinx.serialization.json.putJsonObject

internal class SocketServicePluginsController(
  private val meetingTitle: String,
  private val roomName: String,
  private val organizationId: String,
  private val pluginsSocketHandler: PluginsSocketHandler,
  private val pluginChatSender: PluginChatSender,
  private val dyteParticipants: DyteParticipants,
  private val selfParticipant: DyteSelfParticipant,
  scope: CoroutineScope,
  selfPluginsPermissions: PluginPermissions,
  pluginsInternalRoomJoinListenerStore: PluginsInternalRoomJoinListenerStore,
  flutterNotifier: IEventController,
  plugins: List<DytePlugin>,
) :
  PluginsController(
    scope,
    selfPluginsPermissions,
    pluginsInternalRoomJoinListenerStore,
    flutterNotifier,
    plugins,
  ) {

  init {
    pluginsSocketHandler
      .listenToPluginSocketMessages()
      .onEach { message -> handlePluginSocketMessage(message) }
      .launchIn(scope)

    pluginsInternalRoomJoinListenerStore.addListener(this)
  }

  override suspend fun onPluginMessage(pluginId: String, pluginMessage: PluginMessage) {
    withContext(scope.coroutineContext) {
      val message = pluginMessage.message
      val type = message["type"]?.jsonPrimitive?.int
      val uuid = message["uuid"]?.jsonPrimitive?.content
      val payload = message["payload"]

      when (type) {
        PluginsSocketEvent.CUSTOM_PLUGIN_EVENT_TO_ROOM.id -> {
          if (payload == null) {
            return@withContext
          }
          pluginsSocketHandler.sendCustomPluginEventToRoom(pluginId, payload, uuid)
        }
        PluginsSocketEvent.CUSTOM_PLUGIN_EVENT_TO_PEERS.id -> {
          if (payload == null || payload !is JsonObject) {
            return@withContext
          }
          val peerIds = extractPeerIdsFromPluginMessagePayload(payload) ?: return@withContext
          pluginsSocketHandler.sendCustomPluginEventToPeers(pluginId, peerIds, payload, uuid)
        }
        PluginsSocketEvent.ENABLE_PLUGIN_FOR_ROOM.id -> {
          pluginsSocketHandler.enablePluginForRoom(pluginId, uuid)
        }
        PluginsSocketEvent.ENABLE_PLUGIN_FOR_PEERS.id -> {
          if (payload == null || payload !is JsonObject) {
            return@withContext
          }
          val peerIds = extractPeerIdsFromPluginMessagePayload(payload) ?: return@withContext
          pluginsSocketHandler.enablePluginForPeers(pluginId, peerIds, uuid)
        }
        PluginsSocketEvent.DISABLE_PLUGIN_FOR_ROOM.id -> {
          pluginsSocketHandler.disablePluginForRoom(pluginId, uuid)
        }
        PluginsSocketEvent.DISABLE_PLUGIN_FOR_PEERS.id -> {
          if (payload == null || payload !is JsonObject) {
            return@withContext
          }
          val peerIds = extractPeerIdsFromPluginMessagePayload(payload) ?: return@withContext
          pluginsSocketHandler.disablePluginForPeers(pluginId, peerIds, uuid)
        }
        PluginsSocketEvent.STORE_INSERT_KEYS.id -> {
          if (payload == null || payload !is JsonObject) {
            return@withContext
          }
          val store = payload["store"]?.jsonPrimitive?.content ?: return@withContext
          val insertKeysJsonArray = payload["insertKeys"]?.jsonArray ?: return@withContext
          val insertKeys = mutableListOf<PluginsSocketHandler.PluginStoreKeyEntry>()
          for (insertKey in insertKeysJsonArray) {
            val key = insertKey.jsonObject["key"]?.jsonPrimitive?.content ?: continue
            val insertKeyPayload = insertKey.jsonObject["payload"]
            insertKeys.add(PluginsSocketHandler.PluginStoreKeyEntry(key, insertKeyPayload))
          }
          pluginsSocketHandler.storeInsertKeys(pluginId, store, insertKeys, uuid)
        }
        PluginsSocketEvent.STORE_GET_KEYS.id -> {
          if (payload == null || payload !is JsonObject) {
            return@withContext
          }
          val store = payload["store"]?.jsonPrimitive?.content ?: return@withContext
          val getKeysJsonArray = payload["getKeys"]?.jsonArray ?: return@withContext
          val getKeys = mutableListOf<String>()
          for (getKey in getKeysJsonArray) {
            val key = getKey.jsonObject["key"]?.jsonPrimitive?.content ?: continue
            getKeys.add(key)
          }
          pluginsSocketHandler.storeGetKeys(pluginId, store, getKeys, uuid)
        }
        PluginsSocketEvent.STORE_DELETE_KEYS.id -> {
          if (payload == null || payload !is JsonObject) {
            return@withContext
          }
          val store = payload["store"]?.jsonPrimitive?.content ?: return@withContext
          val deleteKeysJsonArray = payload["deleteKeys"]?.jsonArray ?: return@withContext
          val deleteKeys = mutableListOf<String>()
          for (deleteKey in deleteKeysJsonArray) {
            val key = deleteKey.jsonObject["key"]?.jsonPrimitive?.content ?: continue
            deleteKeys.add(key)
          }
          pluginsSocketHandler.storeDeleteKeys(pluginId, store, deleteKeys, uuid)
        }
        PluginsSocketEvent.STORE_DELETE.id -> {
          if (payload == null || payload !is JsonObject) {
            return@withContext
          }
          val store = payload["store"]?.jsonPrimitive?.content ?: return@withContext
          pluginsSocketHandler.deleteStore(pluginId, store, uuid)
        }
        LocalPluginEvent.GET_PLUGIN_INITIATOR.id -> {
          val enabledBy = _activePlugins[pluginId]?.enabledBy ?: return@withContext
          val data = buildJsonObject { put("enabledBy", enabledBy) }
          submitMessageToPlugin(pluginId, type, uuid, data)
        }
        LocalPluginEvent.GET_DISPLAY_TITLE.id -> {
          val data = buildJsonObject { put("displayTitle", meetingTitle) }
          submitMessageToPlugin(pluginId, type, uuid, data)
        }
        LocalPluginEvent.GET_ROOM_NAME.id -> {
          val data = buildJsonObject { put("roomName", roomName) }
          submitMessageToPlugin(pluginId, type, uuid, data)
        }
        LocalPluginEvent.CHAT_MESSAGE.id -> {
          if (payload == null || payload !is JsonObject) {
            return@withContext
          }

          val messagePayload = payload["messagePayload"]?.jsonObject
          val peerIds =
            try {
              payload["peerIds"]?.jsonArray
            } catch (e: Exception) {
              return@withContext
            }

          // TODO: Once we re-structure our features, check if Chat feature is enabled, here

          if (!peerIds.isNullOrEmpty()) {
            return@withContext
          }

          // TODO: Once we have support for Private chat, use that API here.
          val messageType = messagePayload?.get("type")?.jsonPrimitive?.content

          /*
           * note(swapnil): Only sending text type chat messages for now.
           *
           * { type: 'image', image: File }, { type: 'file', file: File }
           * TODO: Check the data coming in File object as part of messagePayload for file and image
           *  type payload and find a way to use it with mobile Chat APIs.
           * */
          if (messageType == "text") {
            try {
              val textMessage =
                messagePayload["message"]?.jsonPrimitive?.content ?: return@withContext
              pluginChatSender.sendTextMessageFromPlugin(textMessage)
              val data = buildJsonObject { put("success", true) }
              submitMessageToPlugin(pluginId, type, uuid, data)
            } catch (e: Exception) {
              val data = buildJsonObject { put("error", e.message) }
              submitMessageToPlugin(pluginId, type, uuid, data)
            }
          }
        }
        LocalPluginEvent.GET_PEER.id -> {
          val peerId = payload?.jsonObject?.get("peerId")?.jsonPrimitive?.contentOrNull
          val selfParticipant = this@SocketServicePluginsController.selfParticipant
          val peer: JsonElement =
            if (peerId == null) {
              selfParticipant.toJsonObject(organizationId)
            } else {
              val joinedMeetingParticipant = dyteParticipants.joined.find { it.id == peerId }
              joinedMeetingParticipant?.let {
                if (it.id == selfParticipant.id) {
                  selfParticipant.toJsonObject(organizationId)
                } else {
                  it.toJsonObject()
                }
              } ?: JsonNull
            }

          val data = buildJsonObject { put("peer", peer) }
          submitMessageToPlugin(pluginId, type, uuid, data)
        }
        LocalPluginEvent.GET_PEERS.id -> {
          val serializedPeers = dyteParticipants.joined.map { it.toJsonObject() }
          val data = buildJsonObject { put("peers", JsonArray(serializedPeers)) }
          submitMessageToPlugin(pluginId, type, uuid, data)
        }
        LocalPluginEvent.CUSTOM_PLUGIN_EVENT_TO_PARENT.id -> {
          val plugin = _activePlugins[pluginId] ?: return@withContext
          if (payload == null) {
            logger.info(
              "PluginsController::onPluginMessage CUSTOM_PLUGIN_EVENT_TO_PARENT::payload is null"
            )
            return@withContext
          }

          val payloadJsonObject =
            payload as? JsonObject
              ?: kotlin.run {
                logger.info(
                  "PluginsController::onPluginMessage CUSTOM_PLUGIN_EVENT_TO_PARENT::error Malformed payload json object"
                )
                return@withContext
              }
          sendCustomPluginEventToParent(plugin, payloadJsonObject)
        }
      }
    }
  }

  override suspend fun loadActivePlugins() {
    val roomActivePlugins = pluginsSocketHandler.getActivePlugins()

    // remove the deactivated local plugins, if any
    val roomActivePluginIds = roomActivePlugins.map { it.id }.toSet()
    for (localActivePluginId in _activePlugins.keys) {
      if (!roomActivePluginIds.contains(localActivePluginId)) {
        disablePluginLocallyAndNotify(localActivePluginId)
      }
    }

    // populate the active plugins
    roomActivePlugins.forEach { activePlugin ->
      enablePluginLocallyAndNotify(activePlugin.id, activePlugin.enabledBy)
    }
  }

  override suspend fun addPlugin(pluginId: String, staggered: Boolean): Result<Unit, Exception> {
    logger.info("DytePlugin::addPlugin::")
    return try {
      pluginsSocketHandler.addPlugin(pluginId, staggered)
      logger.info("DytePlugin::addPlugin::success")
      Result.Success(Unit)
    } catch (e: Exception) {
      logger.error("DytePlugin::addPlugin::failure", e)
      Result.Failure(e)
    }
  }

  override suspend fun removePlugin(pluginId: String): Result<Unit, Exception> {
    logger.info("DytePlugin::removePlugin::")
    return try {
      pluginsSocketHandler.removePlugin(pluginId)
      logger.info("DytePlugin::removePlugin::success")
      Result.Success(Unit)
    } catch (e: Exception) {
      logger.error("DytePlugin::removePlugin::failure", e)
      Result.Failure(e)
    }
  }

  override suspend fun onCleared() {
    pluginsInternalRoomJoinListenerStore.removeListener()
    super.onCleared()
  }

  private fun extractPeerIdsFromPluginMessagePayload(payload: JsonObject): List<String>? {
    val peerIdsJsonArray = payload["peerIds"]?.jsonArray ?: return null
    val peerIds = mutableListOf<String>()
    for (peerId in peerIdsJsonArray) {
      peerIds.add(peerId.jsonPrimitive.content)
    }
    return peerIds
  }

  private suspend fun handlePluginSocketMessage(message: PluginSocketMessage) {
    when (message) {
      is PluginSocketMessage.AddPlugin -> {
        handleAddPluginSocketMessage(message)
      }
      is PluginSocketMessage.RemovePlugin -> {
        handleRemovePluginSocketMessage(message)
      }
      is PluginSocketMessage.EnablePluginForPeers -> {
        val data = buildJsonObject { put("enabledBy", message.enabledBy) }
        submitMessageToPlugin(message.pluginId, message.event, message.id, data)
      }
      is PluginSocketMessage.EnablePluginForRoom -> {
        val data = buildJsonObject { put("enabledBy", message.enabledBy) }
        submitMessageToPlugin(message.pluginId, message.event, message.id, data)
      }
      is PluginSocketMessage.DisablePluginForPeers -> {
        val data = buildJsonObject { put("disabledBy", message.disabledBy) }
        submitMessageToPlugin(message.pluginId, message.event, message.id, data)
      }
      is PluginSocketMessage.DisablePluginForRoom -> {
        val data = buildJsonObject { put("disabledBy", message.disabledBy) }
        submitMessageToPlugin(message.pluginId, message.event, message.id, data)
      }
      is PluginSocketMessage.CustomPluginEventToPeers -> {
        val data = buildJsonObject { put("data", message.pluginData) }
        submitMessageToPlugin(message.pluginId, message.event, message.id, data)
      }
      is PluginSocketMessage.CustomPluginEventToRoom -> {
        val data = buildJsonObject { put("data", message.pluginData) }
        submitMessageToPlugin(message.pluginId, message.event, message.id, data)
      }
      is PluginSocketMessage.StoreInsertKeys -> {
        val data = buildJsonObject {
          put("storeName", message.pluginStore.storeName)
          put("storeItems", message.pluginStore.getStoreItemsAsJsonArray())
        }
        submitMessageToPlugin(message.pluginStore.pluginId, message.event, message.id, data)
      }
      is PluginSocketMessage.StoreGetKeys -> {
        val data = buildJsonObject {
          put("storeName", message.pluginStore.storeName)
          put("storeItems", message.pluginStore.getStoreItemsAsJsonArray())
        }
        submitMessageToPlugin(message.pluginStore.pluginId, message.event, message.id, data)
      }
      is PluginSocketMessage.StoreDeleteKeys -> {
        val data = buildJsonObject {
          put("storeName", message.pluginStore.storeName)
          put("storeItems", message.pluginStore.getStoreItemsAsJsonArray())
        }
        submitMessageToPlugin(message.pluginStore.pluginId, message.event, message.id, data)
      }
      is PluginSocketMessage.StoreDelete -> {
        val data = buildJsonObject { put("storeName", message.storeName) }
        submitMessageToPlugin(message.pluginId, message.event, message.id, data)
      }
    }
  }

  private suspend fun handleAddPluginSocketMessage(message: PluginSocketMessage.AddPlugin) {
    var pluginActive = false
    _activePlugins[message.pluginId]?.let { plugin ->
      if (plugin.isActive) {
        pluginActive = true
      }
    }

    if (!pluginActive) {
      enablePluginLocallyAndNotify(message.pluginId, message.enabledBy)
    }
  }

  private suspend fun handleRemovePluginSocketMessage(message: PluginSocketMessage.RemovePlugin) {
    disablePluginLocallyAndNotify(message.pluginId)
  }

  private fun submitMessageToPlugin(
    pluginId: String,
    messageType: Int,
    messageId: String?,
    data: JsonObject,
  ) {
    val eventPayload = buildJsonObject {
      put("type", messageType)
      put("uuid", messageId)
      put("payload", data)
    }
    submitEventToPlugin(pluginId, eventPayload)
  }

  companion object {
    private fun PluginStore.getStoreItemsAsJsonArray(): JsonArray {
      return buildJsonArray {
        storeItems.forEach { pluginStoreItem ->
          addJsonObject {
            put("timestamp", pluginStoreItem.timestamp)
            put("peerId", pluginStoreItem.peerId)
            put("payload", pluginStoreItem.payload)
            put("key", pluginStoreItem.key)
          }
        }
      }
    }

    /*
     * note(swapnil): The following properties are not present in the DyteJoinedMeetingParticipant of mobile-core:
     * 1. device: DeviceConfig
     * 2. screenShareEnabled: Boolean
     * 3. supportsRemoteControl: Boolean
     * 4. presetName: String?
     * 5. webinarStageStatus: WebinarStageStatus
     *
     * These properties won't be sent to the Plugin.
     * */
    private fun DyteJoinedMeetingParticipant.toJsonObject(): JsonObject {
      return buildJsonObject {
        put("id", id)
        put("userId", userId)
        put("name", name)
        put("picture", picture)
        put("isHost", isHost)
        put("clientSpecificId", clientSpecificId)
        putJsonObject("flags") {
          put("recorder", flags.recorder)
          put("hidden_participant", flags.hiddenParticipant)
          // web-core has both hidden_participant & hiddenParticipant. So covering both the cases.
          put("hiddenParticipant", flags.hiddenParticipant)
        }
        put("videoEnabled", videoEnabled)
        put("audioEnabled", audioEnabled)
      }
    }

    /*
     * note(swapnil): The following properties are not present in the DyteSelfParticipant of mobile-core:
     * 1. device: DeviceConfig
     * 2. role: Any
     * 3. supportsRemoteControl: Boolean
     * 4. presetName: String?
     * 5. stageStatus: StageStatus
     * 6. isRecorder: Boolean
     *
     * These properties won't be sent to the Plugin.
     *
     * We could have passed the presetName from MetaController but we are deliberately skipping it until
     * it is absolutely required by a Plugin to work.
     * Reason: Uniformity. We are not including the presetName in serialization of DyteJoinedMeetingParticipant as well.
     * */
    private fun DyteSelfParticipant.toJsonObject(organizationId: String): JsonObject {
      return buildJsonObject {
        put("id", id)
        put("userId", userId)
        put("name", name)
        put("picture", picture)
        put("isHost", isHost)
        put("clientSpecificId", clientSpecificId)
        putJsonObject("flags") {
          put("recorder", flags.recorder)
          put("hidden_participant", flags.hiddenParticipant)
          // web-core has both hidden_participant & hiddenParticipant. So covering both the cases.
          put("hiddenParticipant", flags.hiddenParticipant)
        }
        put("videoEnabled", videoEnabled)
        put("audioEnabled", audioEnabled)

        val waitlistStatus = waitListStatus.name.lowercase()
        put("waitlistStatus", waitlistStatus)

        put("organizationId", organizationId)
        put("isRecorder", permissions.miscellaneous.isRecorder)
        put("isHidden", permissions.miscellaneous.isHiddenParticipant)
      }
    }
  }
}
