package io.dyte.core.models

import io.dyte.core.controllers.ParticipantController
import io.dyte.core.feat.DyteMeetingParticipant
import io.dyte.core.feat.ParticipantFlags
import io.dyte.core.host.IHostController
import io.dyte.core.network.info.SelfPermissions
import io.dyte.core.platform.VideoView
import io.dyte.webrtc.VideoStreamTrack
import kotlinx.coroutines.launch

open class DyteJoinedMeetingParticipant
internal constructor(
  override val id: String,
  override val userId: String,
  override var name: String,
  override val picture: String?,
  override val isHost: Boolean,
  override val clientSpecificId: String?,
  override val flags: ParticipantFlags,
  override val presetName: String,
  participantController: ParticipantController,
  internal val hostController: IHostController,
  internal val selfPermissions: SelfPermissions,
) :
  DyteMeetingParticipant(
    id,
    userId,
    name,
    picture,
    isHost,
    clientSpecificId,
    flags,
    presetName,
    participantController,
  ) {

  /** The video track for the local user. */
  internal var _videoTrack: VideoStreamTrack? = null
  val videoTrack: VideoStreamTrack?
    get() = _videoTrack

  internal var _screenShareTrack: VideoStreamTrack? = null
  val screenShareTrack: VideoStreamTrack?
    get() = _screenShareTrack

  /**
   * Disables the audio of the participant if it is enabled
   *
   * Requires *localUser.permissions.host.canMuteAudio*
   */
  @Throws(Exception::class)
  open fun disableAudio() {
    serialScope.launch {
      if (selfPermissions.host.canMuteAudio.not()) {
        // TODO : Send out error to client
        // Result.Failure(DyteError(code = 12, localisedDescription = "cant disable participant
        // audio"))
        return@launch
      }
      hostController.muteAudio(id)
    }
  }

  /**
   * Disables the video of the participant if it is enabled
   *
   * Requires *localUser.permissions.host.canMuteVideo*
   */
  @Throws(Exception::class)
  open fun disableVideo() {
    serialScope.launch {
      if (selfPermissions.host.canMuteVideo.not()) {
        // TODO : Send out error to client
        // Result.Failure(DyteError(code = 12, localisedDescription = "cant produce video"))
        return@launch
      }
      hostController.muteVideo(id)
    }
  }

  /**
   * Pins the participant, so the participant is always in the active list
   *
   * Requires *localUser.permissions.host.canPinParticipant*
   */
  @Throws(Exception::class)
  open fun pin() {
    serialScope.launch {
      if (selfPermissions.host.canPinParticipant.not()) {
        // TODO : Send out error to client
        // Result.Failure(DyteError(code = 12, localisedDescription = "Not allowed to pin
        // participant"))
        return@launch
      }

      hostController.pinPeer(id)
    }
  }

  /**
   * Unpins the participant
   *
   * Requires *localUser.permissions.host.canPinParticipant*
   */
  @Throws(Exception::class)
  open fun unpin() {
    serialScope.launch {
      if (selfPermissions.host.canPinParticipant.not()) {
        // TODO : Send out error to client
        // Result.Failure(DyteError(code = 12, localisedDescription = "Not allowed to un-pin
        // participant"))
        return@launch
      }

      hostController.unpinPeer()
    }
  }

  @Suppress("UNUSED") // Public API
  val isPinned: Boolean
    get() = participantController.pinnedParticipant?.id == this.id

  /**
   * Removes the participant from the meeting
   *
   * Requires *localUser.permissions.host.canKickParticipant*
   */
  @Throws(Exception::class)
  open fun kick() {
    serialScope.launch {
      if (selfPermissions.host.canKickParticipant.not()) {
        // TODO : Send out error to client
        // Result.Failure(DyteError(code = 12, localisedDescription = "not allowed to kick
        // participant"))
        return@launch
      }
      hostController.kickPeer(id)
    }
  }

  open fun getVideoView(): VideoView? {
    return participantController.getVideoView(this)
  }

  fun getScreenShareVideoView(): VideoView {
    return participantController.getScreenShareView(this)
  }
}
