package io.dyte.core.waitingroom.roomnode

import io.dyte.core.Result
import io.dyte.core.controllers.ParticipantController
import io.dyte.core.events.EventEmitter
import io.dyte.core.events.InternalEvents
import io.dyte.core.feat.DyteWaitlistedParticipant
import io.dyte.core.host.IHostController
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.models.WaitListStatus.ACCEPTED
import io.dyte.core.models.WaitListStatus.REJECTED
import io.dyte.core.socket.RoomNodeSocketService
import io.dyte.core.socket.SocketMessageEventListener
import io.dyte.core.socket.events.InboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType.ACCEPT_WAITLIST_REQUEST
import io.dyte.core.socket.events.OutboundMeetingEventType.REJECT_WAITLIST_REQUEST
import io.dyte.core.socket.events.payloadmodel.InboundMeetingEvent
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketMeetingPeerUser
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWaitlistPeerAdded
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWaitlistPeerClosed
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketJoinRoomModel
import io.dyte.core.waitingroom.BaseWaitlistController
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive

internal class RoomNodeWaitlistController(
  private val roomNodeSocket: RoomNodeSocketService,
  internalEventsEmitter: EventEmitter<InternalEvents>,
  selfParticipant: DyteSelfParticipant,
  scope: CoroutineScope,
  participantController: ParticipantController,
  hostController: IHostController,
) :
  BaseWaitlistController(
    internalEventsEmitter,
    selfParticipant,
    scope,
    participantController,
    hostController,
  ) {
  init {
    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PEER_JOINED,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          val payload = event.payload as WebSocketMeetingPeerUser
          this@RoomNodeWaitlistController.scope.launch { refreshWaitlistOnPeerJoined(payload.id) }
        }
      },
    )

    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_WAITLIST_PEER_ADDED,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          val payload = event.payload as WebSocketWaitlistPeerAdded
          this@RoomNodeWaitlistController.scope.launch {
            addParticipantToWaitlist(id = payload.id, userId = null, displayName = payload.name)
          }
        }
      },
    )

    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_WAITLIST_PEER_CLOSED,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          val payload = event.payload as WebSocketWaitlistPeerClosed
          this@RoomNodeWaitlistController.scope.launch { removeParticipantFromWaitlist(payload.id) }
        }
      },
    )

    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_WAITLIST_REJECTED,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          selfParticipant._waitListStatus = REJECTED
          internalEventsEmitter.emitEvent { it.onWaitlistEntryRejected() }
        }
      },
    )

    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_WAITLIST_ACCEPTED,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          selfParticipant._waitListStatus = ACCEPTED
          internalEventsEmitter.emitEvent { it.onWaitlistEntryAccepted() }
          internalEventsEmitter.emitEvent { it.connectToRoomNode() }
        }
      },
    )
  }

  override fun onRoomJoined(webSocketJoinRoomModel: WebSocketJoinRoomModel) {
    super.onRoomJoined(webSocketJoinRoomModel)
    if (selfPermissions.waitingRoom.canAcceptRequests) {
      runBlocking(scope.coroutineContext) {
        webSocketJoinRoomModel.waitlistedPeers.forEach {
          addParticipantToWaitlist(it.id, null, it.name)
        }
      }
    }
  }

  override suspend fun acceptAllWaitingRequestsInternal(
    waitlist: List<DyteWaitlistedParticipant>
  ): Result<List<DyteWaitlistedParticipant>, Exception> {
    val acceptedParticipants = mutableListOf<DyteWaitlistedParticipant>()
    for (participant in waitlist) {
      val result = acceptWaitlistedRequestInternal(participant)
      if (result is Result.Failure) {
        logger.error(
          "WaitlistController::acceptAllWaitingRequests::failed to accept ${participant.id}",
          result.value,
        )
        continue
      }
      acceptedParticipants.add(participant)
    }
    if (acceptedParticipants.isEmpty()) {
      return Result.Failure(Exception("Failed to accept all waiting requests"))
    }

    return Result.Success(acceptedParticipants)
  }

  override suspend fun acceptWaitlistedRequestInternal(
    waitlistedParticipant: DyteWaitlistedParticipant
  ): Result<Unit, Exception> {
    val content = HashMap<String, JsonElement>()
    content["id"] = JsonPrimitive(waitlistedParticipant.id)
    return try {
      roomNodeSocket.sendMessage(ACCEPT_WAITLIST_REQUEST, JsonObject(content))
      Result.Success(Unit)
    } catch (e: Exception) {
      Result.Failure(e)
    }
  }

  override suspend fun rejectWaitlistedRequestInternal(
    waitlistedParticipant: DyteWaitlistedParticipant
  ): Result<Unit, Exception> {
    val content = HashMap<String, JsonElement>()
    content["id"] = JsonPrimitive(waitlistedParticipant.id)
    return try {
      roomNodeSocket.sendMessage(REJECT_WAITLIST_REQUEST, JsonObject(content))
      Result.Success(Unit)
    } catch (e: Exception) {
      Result.Failure(e)
    }
  }
}
