package io.dyte.core.socket

import ACKFn
import io.dyte.core.ControllerScopeProvider
import io.dyte.core.DyteError
import io.dyte.core.Result
import io.dyte.core.SocketErrorCode
import io.dyte.core.controllers.DyteEventType.OnMeetingStateChanged
import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.media.IDyteMediaSoupUtils
import io.dyte.core.network.info.MeetingSessionInfo
import io.dyte.core.observability.DyteLogger
import io.dyte.core.socket.events.InboundMeetingEventType
import io.dyte.core.socket.events.InboundMeetingEventType.*
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.payloadmodel.BasePayloadModel
import io.dyte.core.socket.events.payloadmodel.InboundMeetingEvent
import io.dyte.core.socket.events.payloadmodel.OutboundMeetingEvents
import io.dyte.core.socket.events.payloadmodel.inbound.*
import io.dyte.socketio.src.ExternalLogger
import io.dyte.socketio.src.Logger
import io.dyte.socketio.src.engine.asString
import kotlinx.coroutines.*
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive

private const val QUERY_PARAM_ROOM_URL = "roomURL"
private const val QUERY_PARAM_PEER_ID = "peerId"
private const val QUERY_PARAM_AUTH_TOKEN = "authToken"
private const val QUERY_PARAM_VERSION = "version"
private const val VERSION = "0.5.0"

internal class RoomNodeSocketService(val controllerContainer: IControllerContainer) :
  ControllerScopeProvider(), IRoomNodeSocketService {
  private lateinit var socket: Socket
  private lateinit var roomNodeUrl: String

  // https://dyte.slack.com/archives/C043X7B6RHC/p1681805925170109
  private var isKicked = false

  private var isSocketInReconnectState = false
  @OptIn(DelicateCoroutinesApi::class, ExperimentalCoroutinesApi::class)
  private var socketSerialScope = CoroutineScope(newSingleThreadContext("SocketSerialScope"))

  private val socketEventManager = SocketEventManager()

  suspend fun init(meetingSessionData: MeetingSessionInfo) {
    DyteLogger.info("RoomNodeSocketService::init::")
    withContext(serialScope.coroutineContext) {
      controllerContainer.metaController.setMeetingTitle(meetingSessionData.title)
      roomNodeUrl = meetingSessionData.roomNodeLink

      val roomNodeLink =
        getRoomSocketLinkQueryParams(
          roomNodeUrl,
          controllerContainer.metaController.getPeerId(),
          controllerContainer.metaController.getMeetingId(),
          controllerContainer.metaController.getAuthToken(),
        )

      val params =
        HashMap<String, String>().apply {
          put(QUERY_PARAM_ROOM_URL, controllerContainer.metaController.getMeetingId())
          put(QUERY_PARAM_PEER_ID, controllerContainer.metaController.getPeerId())
          put(QUERY_PARAM_AUTH_TOKEN, controllerContainer.metaController.getAuthToken())
          put(QUERY_PARAM_VERSION, VERSION)
        }

      socket = createSocket(roomNodeLink, params)
      Logger.setExternalLogger(
        object : ExternalLogger {
          override fun debug(message: String) {
            DyteLogger.debug(message)
          }

          override fun error(message: String) {
            DyteLogger.error(message)
          }

          override fun info(message: String) {
            DyteLogger.info(message)
          }

          override fun warn(message: String) {
            DyteLogger.warn(message)
          }
        }
      )
    }
  }

  override suspend fun connect() {
    withContext(serialScope.coroutineContext) {
      if (!socket.isConnected()) {
        socket.connect()
      }
    }
  }

  override suspend fun sendMessage(
    outboundMeetingEventType: OutboundMeetingEventType,
    payload: JsonElement?,
  ): String {
    DyteLogger.info("RoomNodeSocketService::sendMessage::${outboundMeetingEventType.type}")
    val result =
      withContext(serialScope.coroutineContext) {
        val eventData = createEventData(outboundMeetingEventType, payload).toString()
        return@withContext socket.emitAck(OutboundMeetingEvents.SEND_MESSAGE.eventName, eventData)
          ?: throw Exception("Room node response for ${outboundMeetingEventType.type} is null")
      }
    DyteLogger.info(
      "RoomNodeSocketService::sendMessage::response_for_${outboundMeetingEventType.type}::$result"
    )
    return result
  }

  override suspend fun <T> sendMessageParsed(
    outboundMeetingEventType: OutboundMeetingEventType,
    payload: JsonElement?,
  ): T {
    DyteLogger.info("RoomNodeSocketService::sendMessageParsed::${outboundMeetingEventType.type}")
    val resp = sendMessage(outboundMeetingEventType, payload)
    val parsed = controllerContainer.socketMessageResponseParser.parseResponse(resp)
    DyteLogger.info(
      "RoomNodeSocketService::sendMessageParsed::response_for_${outboundMeetingEventType.type}"
    )
    @Suppress("UNCHECKED_CAST") return parsed.payload as T
  }

  override suspend fun sendPacket(
    outboundMeetingEventType: OutboundMeetingEventType,
    payload: JsonElement?,
  ): Result<String, DyteError> {
    DyteLogger.info("RoomNodeSocketService::sendPacket::${outboundMeetingEventType.type}")
    val result =
      withContext(serialScope.coroutineContext) {
        val eventData = createEventData(outboundMeetingEventType, payload).toString()

        val response =
          socket.emitAck(OutboundMeetingEvents.SEND_MESSAGE.eventName, eventData)
            ?: return@withContext Result.Failure(
              DyteError(
                SocketErrorCode.Get_Socket_Response_Is_Null,
                "Expected response from server is Null",
              )
            )
        if (response == "NO ACK") {
          return@withContext Result.Failure(
            DyteError(
              SocketErrorCode.Get_Socket_Response_Timeout,
              "Server is not responding in within given time limit",
            )
          )
        }

        return@withContext Result.Success(response)
      }
    return result
  }

  override suspend fun sendMessageAsync(
    outboundMeetingEventType: OutboundMeetingEventType,
    payload: JsonElement?,
  ) {
    serialScope.launch {
      val eventData = createEventData(outboundMeetingEventType, payload).toString()
      socket.emitAckAsync(OutboundMeetingEvents.SEND_MESSAGE.eventName, eventData)
    }
  }

  override suspend fun sendMessageSyncGeneric(payload: JsonObject): String {
    return withContext(serialScope.coroutineContext) {
      val response =
        socket.emitAck(OutboundMeetingEvents.SEND_MESSAGE.eventName, payload.toString())
          ?: throw Exception("Room node response is null")
      response
    }
  }

  override fun addConnectionListener(listener: SocketConnectionEventListener) {
    serialScope.launch { socketEventManager.addConnectionEventListener(listener) }
  }

  override fun removeConnectionListener(listener: SocketConnectionEventListener) {
    serialScope.launch { socketEventManager.removeConnectionEventListener(listener) }
  }

  override fun addMessageEventListener(
    event: InboundMeetingEventType,
    listener: SocketMessageEventListener,
  ) {
    serialScope.launch { socketEventManager.addMessageEventListener(event, listener) }
  }

  override fun removeMessageEventListener(
    event: InboundMeetingEventType,
    listener: SocketMessageEventListener,
  ) {
    serialScope.launch { socketEventManager.removeMessageEventListener(event, listener) }
  }

  override fun clear() {
    socketEventManager.clear()
  }

  override fun refreshUrl(peerId: String) {
    val roomNodeLink =
      getRoomSocketLinkQueryParams(
        roomNodeUrl,
        peerId,
        controllerContainer.metaController.getMeetingId(),
        controllerContainer.metaController.getAuthToken(),
      )

    val params =
      HashMap<String, String>().apply {
        put(QUERY_PARAM_ROOM_URL, controllerContainer.metaController.getMeetingId())
        put(QUERY_PARAM_PEER_ID, peerId)
        put(QUERY_PARAM_AUTH_TOKEN, controllerContainer.metaController.getAuthToken())
        put(QUERY_PARAM_VERSION, VERSION)
        put("EIO", "4")
      }

    socket = createSocket(roomNodeLink, params)
  }

  override fun disconnect() {
    if (this::socket.isInitialized) {
      socket.disconnect()
    }
  }

  private suspend fun handleEvent(
    inboundMeetingEventType: InboundMeetingEventType,
    payloadModel: BasePayloadModel,
    onDone: ACKFn,
  ) {
    when (inboundMeetingEventType) {
      WEB_SOCKET_NEW_CONSUMER -> handleNewConsumer(payloadModel as WebSocketConsumerModel, onDone)
      WEB_SOCKET_CLOSE_CONSUMER -> handleCloseConsumer(payloadModel as WebSocketConsumerClosedModel)
      WEB_SOCKET_KICKED -> {
        handleKicked()
      }
      else -> {
        // do nothing
      }
    }
  }

  private fun handleKicked() {
    isKicked = true
  }

  private fun createEventData(type: OutboundMeetingEventType?, payload: JsonElement?): JsonObject {
    val mapper = hashMapOf<String, JsonElement>()
    type?.type?.let { mapper.put("type", JsonPrimitive(it)) }
    payload?.let { mapper.put("payload", payload) }
    return JsonObject(mapper)
  }

  /**
   * Get room socket link query params
   *
   * adds roomName, authToken, peerId to the room node link, we receive in the meeting session
   * graphql api
   *
   * @param roomNodeLink
   * @param peerId
   * @param meetingName
   * @param authToken
   * @return
   */
  private fun getRoomSocketLinkQueryParams(
    roomNodeLink: String,
    peerId: String,
    meetingName: String,
    authToken: String,
  ): String {
    val params =
      HashMap<String, String>().apply {
        put(
          QUERY_PARAM_ROOM_URL,
          controllerContainer.platformUtilsProvider
            .getPlatformUtils()
            .getUrlEncodedString(meetingName),
        )
        put(QUERY_PARAM_PEER_ID, peerId)
        put(QUERY_PARAM_AUTH_TOKEN, authToken)
        put(QUERY_PARAM_VERSION, VERSION)
      }

    val map = HashMap<String, String>()
    params.forEach { map[it.key] = it.value }

    val urlBuilder = StringBuilder(roomNodeLink)

    if (!map.isEmpty()) {
      urlBuilder.append("?")

      map.keys.forEachIndexed { index, key ->
        urlBuilder.append(key).append("=").append(map[key])

        if (index < map.keys.size - 1) {
          urlBuilder.append("&")
        }
      }
    }

    return urlBuilder.toString()
  }

  private fun handleCloseConsumer(payloadModel: WebSocketConsumerClosedModel) {
    (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)?.handleCloseConsumer(payloadModel)
  }

  private fun handleNewConsumer(webSocketConsumerModel: WebSocketConsumerModel, onDone: ACKFn) {
    controllerContainer.mediaSoupController.onNewConsumer(webSocketConsumerModel)
    (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)?.handleNewConsumer(
      webSocketConsumerModel,
      fun() {
        onDone?.invoke("")
      },
    )
  }

  /*
    private fun handleConnectTransport(
      payloadModel: WebSocketConnectTransportModel,
    ) {
      controllerContainer.platformUtilsProvider.getMediaSoupUtils().connectTransport(
        payloadModel.id ?: return,
        payloadModel.producing ?: false,
      )
    }
  */

  private fun createSocket(roomNodeLink: String, params: Map<String, String>): Socket {
    socket = Socket(endpoint = roomNodeLink, config = SocketOptions(queryParams = params))
    val sc = socket.socketIo
    sc.on(EVENT_CONNECT) {
      controllerContainer.eventController.triggerEvent(
        OnMeetingStateChanged(EVENT_CONNECT, isSocketInReconnectState)
      )

      if (isSocketInReconnectState) {
        DyteLogger.info("SocketController | Socket Reconnected")
        isSocketInReconnectState = false
      } else {
        DyteLogger.info("SocketController | Socket Connected")
      }

      socketEventManager.submitConnectionEvent(SocketConnectionEvent.Connected)
    }

    sc.on(EVENT_DISCONNECT) {
      DyteLogger.info("SocketController | Socket Disconnected")
      controllerContainer.eventController.triggerEvent(OnMeetingStateChanged(EVENT_DISCONNECT))
      socketEventManager.submitConnectionEvent(
        SocketConnectionEvent.Disconnected(1000, it as String)
      )
    }

    sc.on(EVENT_ERROR) {
      DyteLogger.info(
        "SocketController | Socket Connection error -> ${(it as? Exception)?.message ?: "Socket error"}"
      )
      controllerContainer.eventController.triggerEvent(OnMeetingStateChanged(EVENT_ERROR))
      //        socketEventManager.submitConnectionEvent(SocketEvent.Error)

    }

    sc.io.on(EVENT_RECONNECT) {
      DyteLogger.info("SocketController | Socket Reconnecting")

      isSocketInReconnectState = true
      controllerContainer.eventController.triggerEvent(OnMeetingStateChanged(EVENT_RECONNECT))

      socketEventManager.submitConnectionEvent(SocketConnectionEvent.Reconnected)
    }

    sc.io.on(EVENT_RECONNECT_ATTEMPT) {
      DyteLogger.info("SocketController | Socket Reconnection attempt -> $it")

      controllerContainer.eventController.triggerEvent(
        OnMeetingStateChanged(EVENT_RECONNECT_ATTEMPT)
      )

      socketEventManager.submitConnectionEvent(
        SocketConnectionEvent.ReconnectionAttemptFailed(it as Int)
      )
    }

    sc.on(EVENT_PING) {
      controllerContainer.eventController.triggerEvent(OnMeetingStateChanged(EVENT_PING))
    }

    sc.on(EVENT_PONG) {
      controllerContainer.eventController.triggerEvent(OnMeetingStateChanged(EVENT_PONG))
    }

    sc.on(EVENT_MESSAGE) {
      controllerContainer.eventController.triggerEvent(OnMeetingStateChanged(EVENT_MESSAGE))
    }

    sc.onEvent("event://server-simple-message") { response, onDone ->
      if (!isKicked) {
        socketSerialScope.launch {
          try {
            response.first().asString()?.let {
              val parsedResponse = controllerContainer.socketMessageResponseParser.parseResponse(it)
              DyteLogger.info(parsedResponse.eventType.name)
              socketEventManager.submitMessageEvent(parsedResponse)
              handleEvent(parsedResponse.eventType, parsedResponse.payload, onDone)
            }
          } catch (e: Exception) {
            // DyteLogger.error("SocketController | Error parsing server event", e)
          }
        }
      }
    }
    return socket
  }
}

interface IRoomNodeSocketService {
  suspend fun connect()

  fun disconnect()

  suspend fun sendMessage(
    outboundMeetingEventType: OutboundMeetingEventType,
    payload: JsonElement?,
  ): String

  suspend fun <T> sendMessageParsed(
    outboundMeetingEventType: OutboundMeetingEventType,
    payload: JsonElement?,
  ): T

  suspend fun sendPacket(
    outboundMeetingEventType: OutboundMeetingEventType,
    payload: JsonElement?,
  ): Result<String, DyteError>

  suspend fun sendMessageAsync(
    outboundMeetingEventType: OutboundMeetingEventType,
    payload: JsonElement?,
  )

  suspend fun sendMessageSyncGeneric(payload: JsonObject): String

  fun addConnectionListener(listener: SocketConnectionEventListener)

  fun removeConnectionListener(listener: SocketConnectionEventListener)

  fun addMessageEventListener(event: InboundMeetingEventType, listener: SocketMessageEventListener)

  fun removeMessageEventListener(
    event: InboundMeetingEventType,
    listener: SocketMessageEventListener,
  )

  fun clear()

  /**
   * Generates a new RoomNode socket link with the provided peerId.
   *
   * **NOTE**: This is needed as a part of a workaround that we have done to make the socket-service
   * work properly after the reconnection.
   */
  fun refreshUrl(peerId: String)
}

interface SocketConnectionEventListener {
  fun onConnectionEvent(event: SocketConnectionEvent)
}

interface SocketMessageEventListener {
  suspend fun onMessageEvent(event: InboundMeetingEvent)
}

internal class SocketEventManager {
  @OptIn(DelicateCoroutinesApi::class, ExperimentalCoroutinesApi::class)
  private var eventSerialScope = CoroutineScope(newSingleThreadContext("SocketEventSerialScope"))

  private val connectionEventListeners: MutableSet<SocketConnectionEventListener> = mutableSetOf()

  private val messageEventListeners:
    MutableMap<InboundMeetingEventType, MutableSet<SocketMessageEventListener>> =
    HashMap()

  fun addConnectionEventListener(listener: SocketConnectionEventListener) {
    eventSerialScope.launch { connectionEventListeners.add(listener) }
  }

  fun removeConnectionEventListener(listener: SocketConnectionEventListener) {
    eventSerialScope.launch { connectionEventListeners.remove(listener) }
  }

  fun submitConnectionEvent(socketIoConnectionEvent: SocketConnectionEvent) {
    eventSerialScope.launch {
      connectionEventListeners.forEach { it.onConnectionEvent(socketIoConnectionEvent) }
    }
  }

  fun addMessageEventListener(
    event: InboundMeetingEventType,
    listener: SocketMessageEventListener,
  ) {
    eventSerialScope.launch {
      val existingEventListeners = messageEventListeners[event]
      if (existingEventListeners == null) {
        val newEventListenersSet = LinkedHashSet<SocketMessageEventListener>()
        newEventListenersSet.add(listener)
        messageEventListeners[event] = newEventListenersSet
      } else {
        existingEventListeners.add(listener)
      }
    }
  }

  fun removeMessageEventListener(
    event: InboundMeetingEventType,
    listener: SocketMessageEventListener,
  ) {
    eventSerialScope.launch { messageEventListeners[event]?.remove(listener) }
  }

  fun submitMessageEvent(messageEvent: InboundMeetingEvent) {
    eventSerialScope.launch { notifyMessageEventListeners(messageEvent) }
  }

  private suspend fun notifyMessageEventListeners(messageEvent: InboundMeetingEvent) {
    if (messageEventListeners.isEmpty()) return

    val eventListeners = messageEventListeners[messageEvent.eventType]
    if (!eventListeners.isNullOrEmpty()) {
      for (listener in eventListeners) {
        listener.onMessageEvent(messageEvent)
      }
    }
  }

  fun clear() {
    connectionEventListeners.clear()
    messageEventListeners.clear()
  }
}
