package io.dyte.core.room

import io.dyte.core.controllers.DyteStageStatus
import io.dyte.core.observability.DyteLogger
import io.dyte.core.socket.socketservice.ISockratesSocketService
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import io.dyte.core.socket.socketservice.SocketServiceUtils
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.channels.ProducerScope
import kotlinx.coroutines.channels.awaitClose
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.callbackFlow
import kotlinx.coroutines.flow.flowOn
import socket.room.GetPeerInfoRequest
import socket.room.GetRoomStageStateResponse
import socket.room.Peer
import socket.room.PeerFlags as SocketPeerFlags
import socket.room.PeerInfoResponse
import socket.room.RemoveParticipantsRequest
import socket.room.RoomPeersInfoResponse
import socket.room.StageType

// TODO(swapnil): add 'getAllAddedParticipants' method
internal interface RoomSocketHandler {
  val roomEvents: Flow<RoomSocketEvent>

  suspend fun getRoomStageState(): RoomStageState?

  suspend fun getPeerInfo(peerId: String): RoomPeerInfo?

  suspend fun getRoomPeersInfo(): List<RoomPeerInfo>

  suspend fun kickPeer(peerId: String)

  suspend fun kickAll()
}

internal data class RoomStageState(
  val onStagePeerIds: Set<String>,
  val approvedStagePeerIds: Set<String>,
  val requestStagePeerIds: Set<String>,
)

internal data class RoomPeerInfo(
  val peerId: String,
  val userId: String,
  val displayName: String,
  val stageStatus: DyteStageStatus?,
  val customParticipantId: String?,
  val presetId: String?,
  val presetName: String?,
  val displayPictureUrl: String?,
  val waitlisted: Boolean,
  val flags: PeerFlags?,
) {
  internal data class PeerFlags(val recorderType: String, val hiddenParticipant: Boolean)
}

internal sealed interface RoomSocketEvent {
  data class JoinRoom(val peerInfo: RoomPeerInfo) : RoomSocketEvent
}

internal class SocketServiceRoomSocketHandler(private val socketService: ISockratesSocketService) :
  RoomSocketHandler {
  private val logger = DyteLogger

  override val roomEvents: Flow<RoomSocketEvent>
    get() =
      callbackFlow {
          val socketServiceEventListener =
            object : SocketServiceEventListener {
              override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
                handleRoomSocketEvent(event, payload, this@callbackFlow)
              }
            }

          subscribeRoomSocketEvents(socketServiceEventListener)

          awaitClose { unsubscribeRoomSocketEvents(socketServiceEventListener) }
        }
        .flowOn(Dispatchers.Default)

  override suspend fun getRoomStageState(): RoomStageState? {
    val socketResponse =
      socketService.requestResponse(
        event = SocketServiceUtils.RoomEvent.GET_ROOM_STAGE_STATE.id,
        payload = null,
      )
        ?: kotlin.run {
          logger.warn("RoomSocketHandler::getRoomStageState::socket response is null")
          return null
        }

    val getRoomStageStateResponse =
      try {
        GetRoomStageStateResponse.ADAPTER.decode(socketResponse)
      } catch (e: Exception) {
        logger.warn(
          "RoomSocketHandler::getRoomStageState::failed to decode GetRoomStageStateResponse"
        )
        return null
      }

    return getRoomStageStateResponse.toRoomStageState()
  }

  override suspend fun getPeerInfo(peerId: String): RoomPeerInfo? {
    val getPeerInfoRequest = GetPeerInfoRequest(peerId)
    val socketResponse =
      socketService.requestResponse(
        event = SocketServiceUtils.RoomEvent.GET_PEER_INFO.id,
        payload = GetPeerInfoRequest.ADAPTER.encode(getPeerInfoRequest),
      )
        ?: kotlin.run {
          logger.warn("RoomSocketHandler::getPeerInfo::socket response is null")
          return null
        }

    return parsePeerInfoResponse(socketResponse)
  }

  override suspend fun getRoomPeersInfo(): List<RoomPeerInfo> {
    val socketResponse =
      socketService.requestResponse(
        event = SocketServiceUtils.RoomEvent.GET_ROOM_PEERS_INFO.id,
        payload = null,
      )
        ?: kotlin.run {
          logger.warn("RoomSocketHandler::getRoomPeersInfo::socket response is null")
          return emptyList()
        }

    val roomPeersInfoResponse =
      try {
        RoomPeersInfoResponse.ADAPTER.decode(socketResponse)
      } catch (e: Exception) {
        logger.warn("RoomSocketHandler::getRoomPeersInfo::failed to decode RoomPeersInfoResponse")
        return emptyList()
      }

    return roomPeersInfoResponse.peers.map { it.toRoomPeerInfo() }
  }

  override suspend fun kickPeer(peerId: String) {
    val removeParticipantsRequest = RemoveParticipantsRequest(listOf(peerId))
    socketService.send(
      event = SocketServiceUtils.RoomEvent.KICK.id,
      payload = RemoveParticipantsRequest.ADAPTER.encode(removeParticipantsRequest),
    )
  }

  override suspend fun kickAll() {
    socketService.send(event = SocketServiceUtils.RoomEvent.KICK_ALL.id, payload = null)
  }

  private fun subscribeRoomSocketEvents(socketServiceEventListener: SocketServiceEventListener) {
    socketService.subscribe(SocketServiceUtils.RoomEvent.JOIN_ROOM.id, socketServiceEventListener)
  }

  private fun unsubscribeRoomSocketEvents(socketServiceEventListener: SocketServiceEventListener) {
    socketService.unsubscribe(SocketServiceUtils.RoomEvent.JOIN_ROOM.id, socketServiceEventListener)
  }

  private fun handleRoomSocketEvent(
    event: Int,
    payload: ByteArray?,
    roomEventsFlow: ProducerScope<RoomSocketEvent>,
  ) {
    when (event) {
      SocketServiceUtils.RoomEvent.JOIN_ROOM.id -> {
        if (payload == null) {
          return
        }
        val roomPeerInfo = parsePeerInfoResponse(payload) ?: return
        roomEventsFlow.trySend(RoomSocketEvent.JoinRoom(roomPeerInfo))
      }
      else -> {}
    }
  }

  private fun parsePeerInfoResponse(socketPayload: ByteArray): RoomPeerInfo? {
    val peerInfoResponse =
      try {
        PeerInfoResponse.ADAPTER.decode(socketPayload)
      } catch (e: Exception) {
        logger.warn("RoomSocketHandler::parsePeerInfoResponse::failed to decode PeerInfoResponse")
        return null
      }

    val peer =
      peerInfoResponse.peer
        ?: kotlin.run {
          logger.warn("RoomSocketHandler::parsePeerInfoResponse::received peer is null")
          return null
        }

    return peer.toRoomPeerInfo()
  }

  companion object {
    private fun GetRoomStageStateResponse.toRoomStageState() =
      RoomStageState(
        on_stage_peers.toSet(),
        approved_stage_peers.toSet(),
        requested_stage_peers.toSet(),
      )

    private fun Peer.toRoomPeerInfo() =
      RoomPeerInfo(
        peer_id,
        user_id,
        display_name,
        stage_type?.toDyteStageStatus(),
        custom_participant_id,
        preset_id,
        flags?.preset_name,
        display_picture_url,
        waitlisted,
        flags?.toRoomPeerFlags(),
      )

    private fun SocketPeerFlags.toRoomPeerFlags() =
      RoomPeerInfo.PeerFlags(recorder_type, hidden_participant)

    // TODO: Move to a common place (mappers) so that it can be used by other components
    private fun StageType.toDyteStageStatus(): DyteStageStatus {
      return when (this) {
        StageType.STAGE_TYPE_ON_STAGE -> DyteStageStatus.ON_STAGE
        StageType.STAGE_TYPE_APPROVED_STAGE -> DyteStageStatus.ACCEPTED_TO_JOIN_STAGE
        StageType.STAGE_TYPE_REQUESTED_STAGE -> DyteStageStatus.REQUESTED_TO_JOIN_STAGE
        else -> DyteStageStatus.OFF_STAGE
      }
    }
  }
}
