package io.dyte.core.network.info.mapper

import io.dyte.core.models.DyteMediaPermission
import io.dyte.core.network.info.MiscellaneousPermissions
import io.dyte.core.network.info.RecorderType
import io.dyte.core.network.models.ParticipantMediaPreset
import io.dyte.core.network.models.ParticipantPreset

internal class V2PresetMapper(private val v2Preset: ParticipantPreset) {
  fun getMiscellaneousPermissions(): MiscellaneousPermissions {
    val stageEnabled =
      (v2Preset.permissions.stageEnabled == true) ||
        v2Preset.config.viewType == "WEBINAR" ||
        v2Preset.config.viewType == "LIVESTREAM"

    val stageAccessFromPreset =
      if (v2Preset.permissions.stageAccess != null) {
        getStageAccessFromString(v2Preset.permissions.stageAccess)
      } else {
        null
      }

    val stageAccess =
      stageAccessFromPreset ?: evaluateStageAccessUsingMediaPermissions(v2Preset.permissions.media)

    return MiscellaneousPermissions(
      v2Preset.permissions.canEditDisplayName,
      v2Preset.permissions.hiddenParticipant,
      v2Preset.permissions.isRecorder ?: false,
      recorderType = getRecorderType(v2Preset.permissions.recorderType),
      canSpotLight = v2Preset.permissions.canSpotlight ?: false,
      stageEnabled = stageEnabled,
      stageAccess = stageAccess,
    )
  }

  private fun evaluateStageAccessUsingMediaPermissions(
    mediaPermissions: ParticipantMediaPreset
  ): DyteMediaPermission {
    val videoPermission = getDyteMediaPermission(mediaPermissions.video.canProduce)
    val audioPermission = getDyteMediaPermission(mediaPermissions.audio.canProduce)
    val screenSharePermission = getDyteMediaPermission(mediaPermissions.screenshare.canProduce)
    return if (
      videoPermission == DyteMediaPermission.ALLOWED ||
        audioPermission == DyteMediaPermission.ALLOWED ||
        screenSharePermission === DyteMediaPermission.ALLOWED
    ) {
      DyteMediaPermission.ALLOWED
    } else if (
      videoPermission == DyteMediaPermission.CAN_REQUEST ||
        audioPermission == DyteMediaPermission.CAN_REQUEST ||
        screenSharePermission === DyteMediaPermission.CAN_REQUEST
    ) {
      DyteMediaPermission.CAN_REQUEST
    } else {
      DyteMediaPermission.NOT_ALLOWED
    }
  }

  private fun getDyteMediaPermission(mediaPermissionString: String): DyteMediaPermission {
    return try {
      DyteMediaPermission.valueOf(mediaPermissionString)
    } catch (e: Exception) {
      DyteMediaPermission.NOT_ALLOWED
    }
  }

  private fun getStageAccessFromString(stageAccessString: String): DyteMediaPermission? {
    if (stageAccessString.isBlank()) {
      return null
    }

    return try {
      DyteMediaPermission.valueOf(stageAccessString)
    } catch (e: Exception) {
      return null
    }
  }

  private fun getRecorderType(recorderTypeString: String): RecorderType {
    return try {
      RecorderType.valueOf(recorderTypeString)
    } catch (e: Exception) {
      RecorderType.NONE
    }
  }
}
