package io.dyte.core.controllers

import io.dyte.core.Result
import io.dyte.core.events.InternalEvents
import io.dyte.core.hive.HiveNodeSocketHandler
import io.dyte.core.media.IDyteHive
import io.dyte.core.media.ProducerType
import io.dyte.core.models.DyteMeetingType
import io.dyte.core.models.DyteParticipants
import io.dyte.core.models.WaitListStatus
import io.dyte.core.network.models.IceServerData
import io.dyte.core.observability.DyteLogger
import io.dyte.core.room.RoomSocketHandler
import io.dyte.core.room.RoomStageState
import io.dyte.core.socket.events.payloadmodel.inbound.MeetingPeerFlags
import io.dyte.core.socket.events.payloadmodel.inbound.MeetingPeerMetadata
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketMeetingPeerUser
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerLeftModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerMuteModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketSelectedPeersModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketStageStatus
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketJoinRoomModel
import io.dyte.core.socket.socketservice.ISockratesSocketService
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import io.dyte.core.socket.socketservice.SocketServiceUtils
import io.dyte.core.utils.JsonUtils.decodeFromByteString
import io.dyte.core.utils.JsonUtils.getPrimitiveValue
import io.dyte.media.hive.HiveConsumerOptions
import io.dyte.media.hive.HiveTransport
import io.dyte.webrtc.MediaStreamTrackKind
import io.dyte.webrtc.RtpTransceiverDirection
import io.dyte.webrtc.VideoStreamTrack
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.DelicateCoroutinesApi
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.MainScope
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.launch
import kotlinx.coroutines.newSingleThreadContext
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import kotlinx.datetime.Clock
import kotlinx.datetime.Instant
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive
import kotlinx.serialization.json.jsonObject
import media.ProducerKind
import media.ProducerState
import media.edge.*
import socket.room.BroadcastMessage
import socket.room.JoinRoomRequest
import socket.room.Peer
import socket.room.PeerInfoResponse
import socket.room.RoomInfoResponse

internal class HiveController(
  controllerContainer: IControllerContainer,
  private val socketHandler: HiveNodeSocketHandler,
  private val json: Json,
  private val socketService:
    ISockratesSocketService, // TODO: Remove direct passing of socketService
  private val roomSocketHandler: RoomSocketHandler,
) : IRoomNodeController, BaseController(controllerContainer), InternalEvents {

  lateinit var data: DyteParticipants

  private val logger = DyteLogger

  private lateinit var iceServers: List<IceServerData>

  private val hiveUtils by lazy { controllerContainer.sfuUtils as IDyteHive }

  // FIXME(itzmanish): hardcoding room's maxPreferredStreams to 6 for now.
  // Though it should not be use after completely shifting to hive node
  private val maxPreferredStreams = 6

  private var activatedProducingPeerIds = mutableSetOf<String>()

  private var roomJoined: Boolean = false

  private var peerProducers = mutableListOf<PeerProducerMeta>()

  private val peerId: String
    get() = controllerContainer.metaController.getPeerId()

  private var roomJoiningInProgress = false

  @OptIn(DelicateCoroutinesApi::class, ExperimentalCoroutinesApi::class)
  private val coroutineScope = CoroutineScope(newSingleThreadContext("HiveController"))

  // Cached callbacks coming from clients that use the callback version of joinRoom method.
  private var onRoomJoined: (() -> Unit)? = null
  private var onRoomJoinFailed: (() -> Unit)? = null

  private var onceJoined = false
  private var totalRejoinCount = 0
  private var lastRejoinTime: Instant? = null

  private val transportMutex = Mutex()

  override fun init() {
    coroutineScope.launch { handleHiveInternalEvents() }
    setupMediaListeners()
    handleSocketEvents()
  }

  private fun setupMediaListeners() {
    controllerContainer.internalEventEmitter.addListener(object : InternalEvents {})
  }

  override suspend fun joinRoom(onRoomJoined: (() -> Unit)?, onRoomJoinFailed: (() -> Unit)?) {
    // When joinRoom is called internally from WaitlistController, we do not want to clear the
    // cached callbacks.
    if (this.onRoomJoined == null) {
      this.onRoomJoined = onRoomJoined
    }
    if (this.onRoomJoinFailed == null) {
      this.onRoomJoinFailed = onRoomJoinFailed
    }

    val peerId = controllerContainer.metaController.getPeerId()
    val displayName = controllerContainer.selfController.getSelf().name
    val userId = controllerContainer.selfController.getSelf().userId

    try {
      val roomUuid = ""
      controllerContainer.metaController.setRoomUuid(roomUuid)
      val joinRoomRequest =
        JoinRoomRequest(
          peer = Peer(peer_id = peerId, display_name = displayName, user_id = userId),
          room_uuid = controllerContainer.metaController.getRoomUuid(),
        )
      val joinRoomResponsePayload =
        socketService.requestResponse(
          event = SocketServiceUtils.RoomEvent.JOIN_ROOM.id,
          payload = JoinRoomRequest.ADAPTER.encode(joinRoomRequest),
        )
          ?: kotlin.run {
            logger.error("HiveController::joinRoom::socket response is null")
            throw Exception("Failed to join room")
          }

      val peerInfoResponse = PeerInfoResponse.ADAPTER.decode(joinRoomResponsePayload)

      val isSelfWaitlisted = peerInfoResponse.peer?.waitlisted ?: false

      if (isSelfWaitlisted) {
        logger.info("HiveController::joinRoom::self is waitlisted")
        controllerContainer.selfController.getSelf()._waitListStatus = WaitListStatus.WAITING
        controllerContainer.selfController.emitEvent {
          it.onWaitListStatusUpdate(WaitListStatus.WAITING)
        }
        return
      }

      // TODO: Do not load timestamp when rejoining meeting
      val loadTimestampResult = loadMeetingStartedTimestamp()
      if (loadTimestampResult is Result.Failure) {
        DyteLogger.error(
          "HiveController::joinRoom::failed to load meeting started timestamp",
          loadTimestampResult.value,
        )
      }

      controllerContainer.participantController.handleSocketServiceRoomJoined()

      // /* TODO: Use suspend function to pause the flow until component completes onRoomJoined.
      // We want to load data after successful JOIN_ROOM message and then connect to media
      // and then finally notify client that join-room is complete */
      // controllerContainer.internalEventEmitter.emitEvent {
      //   it.onRoomJoined(WebSocketJoinRoomModel())
      // }

      transportMutex.withLock { joinMediaRoom() }
    } catch (e: Exception) {
      logger.error("HiveController::joinRoom", e)
      controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRoomJoinFailed(e))
    }
  }

  private suspend fun joinMediaRoom(rejoining: Boolean = false) {
    hiveUtils.cleanupTransport()
    this.reset()

    this.roomJoiningInProgress = true

    DyteLogger.info("HiveController: Join room started")
    controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRoomJoinStarted)

    try {
      socketHandler.joinRoom(
        displayName = controllerContainer.selfController.getSelf().name,
        preJoined = rejoining,
      )

      hiveUtils.setupTransports()

      completeJoinRoom()
    } catch (e: Exception) {
      logger.error("HiveController: Error on sending join room request: $e")
      controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRoomJoinFailed(e))
    }
  }

  private suspend fun completeJoinRoom() {
    try {
      val roomStageState =
        try {
          /* surrounding with try-catch as we don't want to stop join-room operation if this step fails */
          roomSocketHandler.getRoomStageState()
        } catch (e: Exception) {
          logger.warn("HiveController::completeJoinRoom::getRoomStageState failed -> ${e.message}")
          null
        }

      val peerJoinResponse = socketHandler.completeJoinRoom()!!

      // TODO: update proto files
      // val is_recorder = peerJoinResponse.room_state.recorder_participant_id == it.peer_id
      // TODO : fix MeetingPeerMetadata

      val oldSocketJoinRoomModel = createOldJoinRoomModel(peerJoinResponse, roomStageState)

      // populate remote peers joined before self in joinedParticipants list
      controllerContainer.participantController.handleRoomJoined(oldSocketJoinRoomModel)
      controllerContainer.participantController.onPinPeer(
        peerJoinResponse.room_state?.pinned_peer_ids?.get(0)
      )

      /* TODO: Use suspend function to pause the flow until component completes onRoomJoined.
      We want to load data after successful JOIN_ROOM message and then connect to media
      and then finally notify client that join-room is complete */
      controllerContainer.internalEventEmitter.emitEvent { it.onRoomJoined(oldSocketJoinRoomModel) }

      this.roomJoiningInProgress = false
      this.roomJoined = true
      this.onceJoined = true

      val meetingCondig = controllerContainer.metaController.getMeetingConfig()

      if (
        controllerContainer.selfController.canPublishAudio() &&
          meetingCondig.enableAudio &&
          controllerContainer.selfController.getSelf().audioEnabled
      ) {
        DyteLogger.info("HiveController: shareMic completeJoinRoom")
        hiveUtils.shareMic(
          controllerContainer.platformUtilsProvider.getMediaUtils().createAudioTrack()
        )
      }

      if (
        controllerContainer.selfController.canPublishVideo() &&
          controllerContainer.selfController.getSelf()._videoEnabled &&
          meetingCondig.enableVideo &&
          controllerContainer.selfController.getSelf().stageStatus == StageStatus.ON_STAGE
      ) {
        DyteLogger.info("HiveController: shareCam completeJoinRoom")
        hiveUtils.shareCam(
          controllerContainer.platformUtilsProvider.getMediaUtils().getVideoTrack()
        )
      }

      val selfTracks = hiveUtils.getSelfTrack()
      val selfParticipant = controllerContainer.selfController.getSelf()
      selfParticipant._videoTrack = selfTracks.second

      controllerContainer.selfController._roomJoined = true

      controllerContainer.participantController.onSelectedPeers(
        WebSocketSelectedPeersModel(
          peerIds = peerJoinResponse.selected_peers?.audio_peers ?: emptyList(),
          compulsoryPeers = peerJoinResponse.selected_peers?.compulsory_peers,
        )
      )

      if (controllerContainer.metaController.getMeetingType() == DyteMeetingType.LIVESTREAM) {
        controllerContainer.liveStreamController.loadData()
      }

      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnMeetingRoomJoined(controllerContainer.selfController.getSelf())
      )

      peerJoinResponse.participants.forEach { participant ->
        participant.producer_states.forEach { state ->
          val producerMeta =
            PeerProducerMeta(
              producerId = state.producer_id,
              kind = if (state.kind == ProducerKind.AUDIO) "audio" else "video",
              paused = state.pause,
              screenShare = state.screen_share,
              peerId = participant.peer_id,
            )

          createConsumer(producerMeta)
        }
      }

      controllerContainer.participantController.refreshGridParticipantsHive()

      controllerContainer.pollsController.loadPolls()
      controllerContainer.chatController.loadChatMessages()
      controllerContainer.platformUtilsProvider.getMediaUtils().routeAudioFromSpeakerphone()

      logger.info("HiveController: Join room completed")
      this.onRoomJoined?.let { onRoomJoinedCallback ->
        MainScope().launch { onRoomJoinedCallback.invoke() }
        this.onRoomJoined = null
      }
    } catch (e: Exception) {
      DyteLogger.error("HiveController: join room failed", e)
      controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRoomJoinFailed(e))
      this.onRoomJoinFailed?.let { onRoomJoinedFailedCallback ->
        MainScope().launch { onRoomJoinedFailedCallback.invoke() }
        this.onRoomJoinFailed = null
      }
    }
  }

  private suspend fun reJoinRoom() {
    val now = Clock.System.now()

    if (lastRejoinTime != null) {
      val duration = now - lastRejoinTime!!

      if (duration.inWholeMilliseconds >= 5000) {
        this.roomJoiningInProgress = true
        this.reset()
        this.lastRejoinTime = now

        transportMutex.withLock { joinMediaRoom(true) }
      }
    }
  }

  private fun createConsumer(producerMeta: PeerProducerMeta) {
    peerProducers.add(producerMeta)

    if (
      producerMeta.kind == "audio" ||
        producerMeta.screenShare ||
        controllerContainer.participantController.currentActiveGridParticipantsHive.contains(
          producerMeta.peerId
        )
    ) {
      logger.info("HiveDebug: handling new consumer")
      coroutineScope.launch {
        hiveUtils.handleNewConsumer(
          HiveConsumerOptions(
            id = null,
            paused = producerMeta.paused,
            producerId = producerMeta.producerId,
            kind =
              if (producerMeta.kind == "audio") MediaStreamTrackKind.Audio
              else MediaStreamTrackKind.Video,
            producingPeerId = producerMeta.peerId,
            appData = mutableMapOf("screenShare" to producerMeta.screenShare),
          )
        )
      }
    }
  }

  private fun handleSocketEvents() {
    val onPeerJoinedBroadcast =
      object : SocketServiceEventListener {
        override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
          coroutineScope.launch {
            logger.info("HiveDebug: Peer Joined Broadcast")
            if (!roomJoined) return@launch

            try {
              val participant =
                PeerJoinBroadcastResponse.ADAPTER.decode(payload!!).participant ?: return@launch
              logger.info("HiveController: Got peer joined with Id: ${participant.peer_id}")

              if (peerId == participant.peer_id) return@launch

              // TODO : fix MeetingPeerMetadata
              val dyteParticipant =
                WebSocketMeetingPeerUser(
                  id = participant.peer_id,
                  isHost = false,
                  name = participant.display_name,
                  flags = MeetingPeerFlags(),
                  audioMuted = true,
                  videoEnabled = false,
                  userId = participant.user_id ?: "", // Setting peer id as userid for now
                  participantMetadata = MeetingPeerMetadata("", ""),
                )

              controllerContainer.participantController.onPeerJoined(dyteParticipant)
              controllerContainer.participantController.refreshGridParticipantsHive()
            } catch (e: Exception) {
              logger.info("HiveController: Error in peer joined broadcast")
            }
          }
        }
      }

    val onPeerProducerCreateBroadcast =
      object : SocketServiceEventListener {
        override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
          if (!roomJoined) return

          try {
            val response = PeerProducerCreateBroadcastResponse.ADAPTER.decode(payload!!)

            if (response.participant_id == peerId) return

            val producerState = response.producer_state

            logger.info(
              "HiveController: Producer created broadcast: " +
                response.participant_id +
                "Producer id: ${producerState?.producer_id}"
            )

            val participant = data.joined.find { it.id == response.participant_id }

            if (participant == null) {
              logger.info("Peer not found for producer ${producerState?.producer_id} ")
            }

            val producerMeta =
              PeerProducerMeta(
                producerId = producerState!!.producer_id,
                kind = if (producerState.kind == ProducerKind.AUDIO) "audio" else "video",
                paused = producerState.pause,
                screenShare = producerState.screen_share,
                peerId = response.participant_id,
              )

            logger.info("HiveDebug: Producer meta ${producerMeta.kind}")

            createConsumer(producerMeta)
          } catch (e: Exception) {
            logger.info("HiveController: Error in peer producer create broadcast")
          }
        }
      }

    val onPeerProducerToggleBroadcast =
      object : SocketServiceEventListener {
        override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
          if (!roomJoined) return

          try {
            val response = PeerProducerToggleBroadcastResponse.ADAPTER.decode(payload!!)

            val producerState = response.producer_state

            val convertedKind: MediaStreamTrackKind =
              if (producerState?.kind == ProducerKind.AUDIO) MediaStreamTrackKind.Audio
              else MediaStreamTrackKind.Video

            if (response.participant_id == peerId && producerState!!.pause) {
              if (
                convertedKind == MediaStreamTrackKind.Audio &&
                  controllerContainer.selfController.getSelf()._audioEnabled
              ) {
                coroutineScope.launch { controllerContainer.selfController.disableAudio() }
              } else if (
                convertedKind == MediaStreamTrackKind.Video &&
                  controllerContainer.selfController.getSelf()._videoEnabled
              ) {
                coroutineScope.launch { controllerContainer.selfController.disableVideo() }
              }

              return
            }

            logger.info(
              "HiveController: Producer toggle broadcast: " +
                "${response.participant_id} producer id: " +
                "${producerState?.producer_id}"
            )
            val participant = data.joined.find { it.id == response.participant_id } ?: return

            if (producerState!!.kind == ProducerKind.AUDIO) {
              if (producerState.pause) {
                controllerContainer.participantController.onPeerAudioMuted(
                  WebSocketPeerMuteModel(participant.id)
                )
              } else {
                controllerContainer.participantController.onPeerAudioUnmuted(
                  WebSocketPeerMuteModel(participant.id)
                )
              }
            }

            val producer = peerProducers.find { it.peerId == response.participant_id }
            if (producer != null) {
              producer.paused = producerState.pause
            }

            val consumersForThisProducer =
              hiveUtils.getConsumers().values.filter {
                it.getProducerId() == producerState.producer_id && it.getKind() == convertedKind
              }

            consumersForThisProducer.forEach {
              if (it.getPaused() != producerState.pause) {
                logger.info("HiveController: Consumer state is mismatched for $")

                CoroutineScope(Dispatchers.Default).launch {
                  if (producerState.pause) {
                    it.pause()
                    hiveUtils.handlePauseConsumer(it.getId())
                  } else {
                    it.resume()
                    hiveUtils.handleResumeConsumer(it.getId())
                  }
                }
              }
            }
          } catch (e: Exception) {
            logger.info("HiveController: Error in producer toggle broadcast handler")
          }
        }
      }

    val onSelectedPeer =
      object : SocketServiceEventListener {
        override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
          if (!roomJoined) return

          try {
            val response = SelectedPeersResponse.ADAPTER.decode(payload!!)
            logger.info(
              "HiveController: Selected peers: ${response.audio_peers} " +
                "${response.compulsory_peers}"
            )

            controllerContainer.participantController.onSelectedPeers(
              WebSocketSelectedPeersModel(
                peerIds = response.audio_peers,
                compulsoryPeers = response.compulsory_peers,
              )
            )
          } catch (e: Exception) {
            logger.info("HiveController: Error in selected peer $e")
          }
        }
      }

    val onSelectedPeerDiff =
      object : SocketServiceEventListener {
        override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
          if (!roomJoined) return

          try {
            val entries = SelectedPeersDiffResponse.ADAPTER.decode(payload!!).entries

            val peerIdsWithPriorities =
              entries.map { PeerWithPriority(peerId = it.peer_id, priority = it.priority) }

            controllerContainer.participantController.onSelectedPeersDiff(peerIdsWithPriorities)
          } catch (e: Exception) {
            logger.info("HiveController: Error in selected peer diff $e")
          }
        }
      }

    val onPeerLeaveBroadcast =
      object : SocketServiceEventListener {
        override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
          logger.info("HiveController::onPeerLeaveBroadcast")
          if (!roomJoined) return

          try {
            val participantId = PeerLeaveBroadcastResponse.ADAPTER.decode(payload!!).participant_id

            if (participantId == peerId) {
              /*
               * NOTE(@madhugb): we are handling kicked event from socket
               * (refer to `RoomEvents.KICK` event in SelfController)
               * Ignore hive event here.
               * */

              // controllerContainer.selfController.onRemovedFromMeeting()
              return
            }

            logger.info("HiveController: Peer left broadcast: $participantId")

            runBlocking { hiveUtils.cleanupConsumers(participantId) }

            controllerContainer.participantController.onPeerLeft(
              WebSocketPeerLeftModel(id = participantId)
            )
          } catch (e: Exception) {
            logger.info("HiveController: Error in peer left broadcast $e")
          }
        }
      }

    val onPeerProducerCloseBroadcast =
      object : SocketServiceEventListener {
        override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
          if (!roomJoined) return

          try {
            val response = PeerProducerCloseBroadcastResponse.ADAPTER.decode(payload!!)

            if (response.participant_id == peerId) return

            val producerState = response.producer_state

            logger.info("HiveController: Producer closed broadcast ${response.participant_id}")

            peerProducers =
              peerProducers.filter { it.producerId != producerState!!.producer_id }.toMutableList()

            val producerId = producerState!!.producer_id

            val consumerId = hiveUtils.producerIdToConsumerIdMap[producerId]

            if (consumerId == null) {
              logger.info("HiveController: No consumer found for producer: $producerId")
              // return
            }

            logger.info("HiveController: Closing consumer $consumerId for $producerId")

            runBlocking {
              consumerId?.let { hiveUtils.closeConsumer(it, null) }
              // hiveUtils.closeConsumer(consumerId, null)
            }

            if (producerState.screen_share) {
              val participant = data.joined.find { it.id == response.participant_id }

              participant?.let {
                val screenShareParticipant =
                  data.joined.find { ss -> ss.id == response.participant_id }

                screenShareParticipant?.let {
                  controllerContainer.participantController.onPeerScreenSharedEnded(
                    screenShareParticipant
                  )
                }
              }
            }

            logger.info("HiveController: Closed consumer $consumerId")

            hiveUtils.producerIdToConsumerIdMap.remove(producerId)
            // todo hive

            //          CoroutineScope(Dispatchers.Default).async {
            //            hiveUtils.closeConsumer(consumerId)
            //          }
          } catch (e: Exception) {
            logger.info("HiveController: Error on Producer close broadcast $e")
          }
        }
      }

    val onGlobalPeerPinBroadcast =
      object : SocketServiceEventListener {
        override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
          if (!roomJoined) return

          try {
            if (payload != null) {
              val peerId = GlobalPeerPinningBroadcastResponse.ADAPTER.decode(payload).participant_id
              logger.info("HiveController: Pinning peerId: $peerId")
              controllerContainer.participantController.onPinPeer(peerId)
            } else {
              controllerContainer.participantController.onPinPeer(null)
            }
          } catch (e: Exception) {
            logger.info("HiveController: Error on global peer pin broadcast $e")
          }
        }
      }

    /*
     * TODO: Include timestamp in the emitted roomMessage.
     *  Reference: https://github.com/dyte-in/web-core/blob/df7c518b091d1cde51f959fed60f21cfc65f5ead/src/socketService/RoomSocketHandler.ts#L202C5-L206
     * */
    val onBroadcastMessage =
      object : SocketServiceEventListener {
        override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
          try {
            payload?.let {
              val res = BroadcastMessage.ADAPTER.decode(payload)
              val payloadJsonElement = json.decodeFromByteString(res.payload)
              val payloadJsonObject = payloadJsonElement.jsonObject
              val broadcastPayloadMap = hashMapOf<String, Any>()
              for (entry in payloadJsonObject) {
                val jsonPrimitive = entry.value as? JsonPrimitive ?: continue
                val primitiveValue = jsonPrimitive.getPrimitiveValue() ?: continue
                broadcastPayloadMap[entry.key] = primitiveValue
              }
              controllerContainer.selfController.emitEvent {
                it.onRoomMessage(res.type, broadcastPayloadMap)
              }
            }
          } catch (e: Exception) {
            logger.error("HiveController: Error in broadcast message $e")
          }
        }
      }

    val onMediaRoomTerminationBroadcast =
      object : SocketServiceEventListener {
        override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
          if (!roomJoined && !onceJoined) return

          try {
            val reason = MediaRoomTerminationBroadcastResponse.ADAPTER.decode(payload!!).reason
            logger.info("HiveController: media hub terminated with reason: $reason")

            coroutineScope.launch { reJoinRoom() }
          } catch (e: Exception) {
            logger.info(
              "HiveController: failed to handle MediaRoomTerminationBroadcastResponse: ${e.message}"
            )
          }
        }
      }

    socketHandler.subscribeToBroadcasts(
      onPeerJoinedBroadcast = onPeerJoinedBroadcast,
      onPeerProducerCreateBroadcast = onPeerProducerCreateBroadcast,
      onPeerProducerToggleBroadcast = onPeerProducerToggleBroadcast,
      onSelectedPeer = onSelectedPeer,
      onSelectedPeerDiff = onSelectedPeerDiff,
      onPeerLeaveBroadcast = onPeerLeaveBroadcast,
      onPeerProducerCloseBroadcast = onPeerProducerCloseBroadcast,
      onGlobalPeerPinBroadcast = onGlobalPeerPinBroadcast,
      onBroadcastMessage = onBroadcastMessage,
      onMediaRoomTerminationBroadcast = onMediaRoomTerminationBroadcast,
    )
  }

  override suspend fun closeConsumers(peers: List<String>) {
    val consumerIds =
      hiveUtils
        .getConsumers()
        .filter { peers.contains(it.value.getPeerId()) }
        .map { itt -> itt.value.getId() }

    peers.forEach { activatedProducingPeerIds.remove(it) }

    return hiveUtils.closeConsumers(consumerIds)
  }

  override suspend fun consumePeer(producingPeerId: String, force: Boolean?) {
    if (!activatedProducingPeerIds.contains(producingPeerId) || force!!) {
      try {
        hiveUtils.consumePeer(producingPeerId)
        this.activatedProducingPeerIds.add(producingPeerId)
      } catch (e: Exception) {
        logger.info("HiveController: consumePeer failed $e")
      }
    } else {
      logger.info("HiveController: Not creating consumer for peerId = $producingPeerId")
    }
  }

  override fun leaveRoom() {
    roomJoined = false
  }

  override suspend fun kickPeer(peerId: String): Boolean {
    return hiveUtils.sfuSocketHandler.kickPeer(peerId)
  }

  override suspend fun kickAll(): Boolean {
    return hiveUtils.sfuSocketHandler.kickAll()
  }

  suspend fun broadcastMessage(type: String, payload: JsonObject) =
    socketHandler.broadcastMessage(type, payload)

  suspend fun activatePeer(peerId: String, force: Boolean? = false) {
    logger.info("HiveDebug: in activate peer")
    val producers = peerProducers.filter { it.peerId == peerId }
    if (producers.isEmpty()) {
      logger.info("HiveUtils: activatePeer $peerId: Not an active peer")

      return
    }

    if (force == true) {
      this.hiveUtils.consumePeer(peerId)
      return
    }

    val consumersToResume = arrayListOf<String>()

    val allConsumersExist =
      producers.all { producer ->
        val consumerId =
          this.hiveUtils.producerIdToConsumerIdMap.get(producer.producerId) ?: return@all false
        val consumer = this.hiveUtils.getConsumers()[consumerId]

        if (consumer != null && !producer.paused && consumer.getPaused()) {
          consumersToResume.add(consumerId)
        }

        return@all true
      }

    if (!allConsumersExist) {
      this.hiveUtils.consumePeer(peerId)
      return
    }

    consumersToResume.forEach { this.hiveUtils.handleResumeConsumer(it) }
  }

  suspend fun deactivatePeer(peerId: String) {
    val producers = peerProducers.filter { it.peerId == peerId }
    logger.info("HiveUtils: deactivatePeer $peerId")

    producers.forEach {
      if (it.kind == "audio" || (it.kind == "video" && it.screenShare)) {
        logger.info("HiveUtils: We don't want to pause audio or screenshare")

        return
      }

      val consumerId = this.hiveUtils.producerIdToConsumerIdMap[it.producerId]

      if (consumerId == null) {
        logger.info("HiveUtils: Consumer not found in deactivate producers ${it.producerId}")

        return
      }

      this.hiveUtils.closeConsumer(consumerId)
      this.activatedProducingPeerIds.remove(peerId)
    }
  }

  override suspend fun reconnectTransport(transport: HiveTransport) {
    // TODO: Socket is connected or not
    if (!roomJoined) {
      if (roomJoiningInProgress) return
    }

    val iceServerData = controllerContainer.apiClient.getICEServers().iceServers

    transport.close()

    if (transport.getDirection() == RtpTransceiverDirection.SendOnly) {
      this.hiveUtils.createWebRtcTransportProd(null, iceServerData)
      logger.info("HiveUtils: Transport connected now need to start creating producers")

      val camTrack: VideoStreamTrack? =
        if (controllerContainer.selfController.getSelf()._videoEnabled)
          controllerContainer.platformUtilsProvider.getMediaUtils().getVideoTrack()
        else null

      // TODO(anunaym14): Disbaling it temporarily because resetting screenshare track will require
      // refactor with the new code flow
      //      val screenShareTrack: VideoStreamTrack? =
      //        if (controllerContainer.selfController.getSelf().screenShareTrack != null)
      //          controllerContainer.selfController.getSelf().screenShareTrack
      //        else null

      this.hiveUtils.resetVideoProducers(camTrack, null)

      this.hiveUtils.removeProducer(ProducerType.MIC.toString(), false)
      this.hiveUtils.shareMic(
        controllerContainer.platformUtilsProvider.getMediaUtils().createAudioTrack()
      )
      if (!controllerContainer.selfController.getSelf()._audioEnabled) this.hiveUtils.pauseMic()

      return
    }

    this.hiveUtils.cleanupConsumers(null)

    this.hiveUtils.createWebRtcTransportRecv(null, iceServerData)

    //    val lastActivePeers = this.activatedProducingPeerIds
    this.activatedProducingPeerIds = mutableSetOf()

    //    lastActivePeers.forEach {
    //      this.consumePeer(it, true)
    //    }
    controllerContainer.participantController.refreshGridParticipantsHive(true)
  }

  private suspend fun handleHiveInternalEvents() {
    this.hiveUtils
      .getObserverFlow()
      .onEach {
        when (it.eventName) {
          "reconnect_transport" -> {
            val transport = it.data as HiveTransport

            try {
              transportMutex.withLock { this.reconnectTransport(transport) }
              logger.info("HiveController: Transport reconnected ${transport.getId()}")
            } catch (e: Exception) {
              logger.info("HiveController: Error on reconnecting transports: $e")
            }
          }
          "rejoin" -> {
            val force = it.data as Boolean? ?: false

            if (this.totalRejoinCount > 3 && !force) {
              DyteLogger.warn("HiveController: cannot rejoin more already rejoined 3 times")
              return@onEach
            }
            if (this.roomJoiningInProgress && !force) {
              DyteLogger.warn("HiveController: room joining in progress, cannot start rejoining")
              return@onEach
            }

            this.totalRejoinCount++
            DyteLogger.warn(
              "HiveController: rejoining the room because transports are failing or media hub is closed",
              mapOf("rejoinCount" to this.totalRejoinCount.toString(), "force" to force.toString()),
            )

            reJoinRoom()
          }
        }
      }
      .launchIn(coroutineScope)
  }

  private fun reset() {
    this.peerProducers = mutableListOf()
    this.activatedProducingPeerIds = mutableSetOf()
  }

  override suspend fun connectToMediaProduction() {}

  /* Internal events start */
  override fun connectMedia() {
    if (controllerContainer.selfController._roomJoined) {
      coroutineScope.launch {
        controllerContainer.sfuUtils.refreshTracks()
        joinRoom()
      }
    }
  }

  override fun disconnectMedia() {
    if (controllerContainer.selfController._roomJoined) {
      reset()
      coroutineScope.launch {
        controllerContainer.sfuUtils.cleanupConsumers()
        controllerContainer.sfuUtils.cleanupProducers()
        controllerContainer.sfuUtils.cleanupTransport()
      }
      controllerContainer.platformUtilsProvider.getVideoUtils().destroyAll()
    }
  }

  override fun connectToRoomNode() {
    super.connectToRoomNode()
    coroutineScope.launch { joinRoom() }
  }

  override fun createProducers() {
    // TODO: Check if we really need the initial meeting config for this subsequent operation
    val meetingConfig = controllerContainer.metaController.getMeetingConfig()

    coroutineScope.launch {
      if (
        controllerContainer.selfController.canPublishAudio() &&
          meetingConfig.enableAudio &&
          controllerContainer.selfController.getSelf().audioEnabled
      ) {
        DyteLogger.info("HiveController: shareMic")
        val audioTrack =
          controllerContainer.platformUtilsProvider.getMediaUtils().createAudioTrack()
        audioTrack!!.enabled = true

        hiveUtils.shareMic(audioTrack)
      }

      if (
        controllerContainer.selfController.canPublishVideo() &&
          controllerContainer.selfController.getSelf()._videoEnabled &&
          meetingConfig.enableVideo /*&&
          controllerContainer.selfController.getSelf().stageStatus == StageStatus.ON_STAGE*/
      ) {
        DyteLogger.info("HiveController: shareCam")
        val videoTrack = controllerContainer.platformUtilsProvider.getMediaUtils().getVideoTrack()
        videoTrack!!.enabled = true

        hiveUtils.shareCam(videoTrack)
      }
    }
  }

  override fun closeProducers() {
    super.closeProducers()
    coroutineScope.launch { hiveUtils.cleanupProducers("leave_stage") }
  }
  /* Internal events end */

  private suspend fun loadMeetingStartedTimestamp(): Result<Unit, Exception> {
    try {
      val roomInfoResponse =
        socketService.requestResponse(SocketServiceUtils.RoomEvent.GET_ROOM_INFO.id, null)
          ?: throw Exception("GET_ROOM_INFO response is null")
      val parsedRoomInfo = RoomInfoResponse.ADAPTER.decode(roomInfoResponse)

      val roomCreatedAtSeconds =
        parsedRoomInfo.room?.created_at ?: throw Exception("GET_ROOM_INFO room field is null")
      val timeInEpochMillis = roomCreatedAtSeconds * 1000

      val timeInISO8601 =
        Instant.fromEpochMilliseconds(timeInEpochMillis).toString().replace("Z", ".000Z")
      controllerContainer.metaController.setMeetingStartedTimestamp(timeInISO8601.toString())
      return Result.Success(Unit)
    } catch (e: Exception) {
      return Result.Failure(e)
    }
  }

  private fun createOldJoinRoomModel(
    peerJoinCompleteResponse: PeerJoinCompleteResponse,
    roomStageState: RoomStageState?,
  ): WebSocketJoinRoomModel {
    val selfStageStatus =
      roomStageState?.getParticipantWebSocketStageStatus(
        participantUserId = controllerContainer.selfController.getSelf().userId
      ) ?: WebSocketStageStatus.OFF_STAGE

    val oldSocketPeerModels =
      peerJoinCompleteResponse.participants.map { it.toWebSocketMeetingPeerUser(roomStageState) }

    /*
     * No need to initialize WebSocketJoinRoomModel.requestToJoinPeersList, StageController fetches
     * the list from socket-service
     * */

    return WebSocketJoinRoomModel(peers = oldSocketPeerModels, stageStatus = selfStageStatus)
  }

  companion object {
    private fun RoomParticipants.toWebSocketMeetingPeerUser(
      roomStageState: RoomStageState?
    ): WebSocketMeetingPeerUser {
      val participantUserId = user_id
      val participantStageStatus =
        if (roomStageState != null && participantUserId != null) {
          roomStageState.getParticipantWebSocketStageStatus(participantUserId)
        } else {
          WebSocketStageStatus.OFF_STAGE
        }

      return WebSocketMeetingPeerUser(
        id = peer_id,
        isHost = false,
        name = display_name,
        flags = MeetingPeerFlags(recorder = false),
        userId = user_id ?: "",
        audioMuted = true,
        videoEnabled = false,
        participantMetadata = MeetingPeerMetadata("", ""),
        stageStatus = participantStageStatus,
      )
    }

    /** Evaluates the [WebSocketStageStatus] of a participant from the given [RoomStageState]. */
    private fun RoomStageState.getParticipantWebSocketStageStatus(
      participantUserId: String
    ): WebSocketStageStatus {
      return if (onStagePeerIds.contains(participantUserId)) {
        WebSocketStageStatus.ON_STAGE
      } else if (approvedStagePeerIds.contains(participantUserId)) {
        WebSocketStageStatus.ACCEPTED_TO_JOIN_STAGE
      } else if (requestStagePeerIds.contains(participantUserId)) {
        WebSocketStageStatus.REQUESTED_TO_JOIN_STAGE
      } else {
        WebSocketStageStatus.OFF_STAGE
      }
    }
  }
}

interface IRoomNodeController {
  fun init() {}

  suspend fun joinRoom(onRoomJoined: (() -> Unit)? = null, onRoomJoinFailed: (() -> Unit)? = null)

  suspend fun connectToMediaProduction()

  fun connectProducerTransport() {}

  // Hive specific methods
  suspend fun reconnectTransport(transport: HiveTransport) {}

  suspend fun activatePeer(peerId: String, producers: List<ProducerState>, force: Boolean?) {}

  suspend fun deactivatePeer(peerId: String, producers: List<ProducerState>) {}

  suspend fun closeConsumers(peers: List<String>) {}

  suspend fun consumePeer(producingPeerId: String, force: Boolean? = null) {}

  fun leaveRoom()

  suspend fun kickPeer(peerId: String): Boolean = true

  suspend fun kickAll(): Boolean = true
}

data class PeerProducerMeta(
  var producerId: String,
  var kind: String,
  var paused: Boolean,
  var screenShare: Boolean,
  var peerId: String,
)
