package io.dyte.core.waitingroom.hive

import io.dyte.core.Result
import io.dyte.core.observability.DyteLogger
import io.dyte.core.socket.socketservice.ISockratesSocketService
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import io.dyte.core.socket.socketservice.SocketServiceUtils
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.channels.ProducerScope
import kotlinx.coroutines.channels.awaitClose
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.callbackFlow
import kotlinx.coroutines.flow.flowOn
import media.edge.PeerJoinBroadcastResponse
import socket.room.AcceptWaitingRoomRequests
import socket.room.DenyWaitingRoomRequests
import socket.room.GetWaitingRoomRequests
import socket.room.WaitingRoomRequest

// TODO: Remove redundant WaitlistedParticipant class and make WaitlistHostSocketHandler return
// DyteWaitlistedParticipant
internal interface WaitlistHostSocketHandler {
  val hostEvents: Flow<WaitlistHostEvent>

  suspend fun getWaitlistedParticipants(): List<WaitlistedParticipant>

  suspend fun acceptWaitlistedParticipant(waitlistedUserIds: List<String>): Result<Unit, Exception>

  suspend fun rejectWaitlistedParticipant(waitlistedUserIds: List<String>): Result<Unit, Exception>
}

internal data class WaitlistedParticipant(
  val peerId: String,
  val userId: String,
  val displayName: String
)

internal sealed interface WaitlistHostEvent {
  data class WaitlistUpdated(val waitlist: List<WaitlistedParticipant>) : WaitlistHostEvent

  data class PeerJoinedMeeting(val peerId: String) : WaitlistHostEvent
}

internal class DefaultWaitlistHostSocketHandler(
  private val socketService: ISockratesSocketService
) : WaitlistHostSocketHandler {
  override val hostEvents: Flow<WaitlistHostEvent>
    get() =
      callbackFlow {
          val socketServiceEventListener =
            object : SocketServiceEventListener {
              override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
                handleWaitlistHostEvent(event, payload, this@callbackFlow)
              }
            }

          subscribeWaitlistHostSocketEvents(socketServiceEventListener)

          awaitClose { unsubscribeWaitlistHostSocketEvents(socketServiceEventListener) }
        }
        .flowOn(Dispatchers.Default)

  override suspend fun getWaitlistedParticipants(): List<WaitlistedParticipant> {
    val socketResponse =
      socketService.requestResponse(SocketServiceUtils.RoomEvent.GET_WAITING_ROOM_REQUESTS.id, null)
        ?: return emptyList() // currently null response means empty request list

    val getWaitingRoomRequestsResponse =
      try {
        GetWaitingRoomRequests.ADAPTER.decode(socketResponse)
      } catch (e: Exception) {
        DyteLogger.warn(
          "WaitlistHostSocketHandler::getWaitlistedParticipants::failed to decode GetWaitingRoomRequests"
        )
        return emptyList()
      }

    return getWaitingRoomRequestsResponse.requests.map { it.toWaitlistedParticipant() }
  }

  override suspend fun acceptWaitlistedParticipant(
    waitlistedUserIds: List<String>
  ): Result<Unit, Exception> {
    val message = AcceptWaitingRoomRequests(waitlistedUserIds)
    return try {
      socketService.send(
        event = SocketServiceUtils.RoomEvent.ACCEPT_WAITING_ROOM_REQUESTS.id,
        payload = AcceptWaitingRoomRequests.ADAPTER.encode(message)
      )
      Result.Success(Unit)
    } catch (e: Exception) {
      Result.Failure(e)
    }
  }

  override suspend fun rejectWaitlistedParticipant(
    waitlistedUserIds: List<String>
  ): Result<Unit, Exception> {
    val message = DenyWaitingRoomRequests(waitlistedUserIds)
    return try {
      socketService.send(
        event = SocketServiceUtils.RoomEvent.DENY_WAITING_ROOM_REQUESTS.id,
        payload = DenyWaitingRoomRequests.ADAPTER.encode(message)
      )
      Result.Success(Unit)
    } catch (e: Exception) {
      Result.Failure(e)
    }
  }

  private fun subscribeWaitlistHostSocketEvents(
    socketServiceEventListener: SocketServiceEventListener
  ) {
    socketService.subscribe(
      SocketServiceUtils.RoomEvent.GET_WAITING_ROOM_REQUESTS.id,
      socketServiceEventListener
    )
    socketService.subscribe(
      SocketServiceUtils.MediaEvent.PEER_JOINED_BROADCAST.id,
      socketServiceEventListener
    )
  }

  private fun unsubscribeWaitlistHostSocketEvents(
    socketServiceEventListener: SocketServiceEventListener
  ) {
    socketService.unsubscribe(
      SocketServiceUtils.RoomEvent.GET_WAITING_ROOM_REQUESTS.id,
      socketServiceEventListener
    )
    socketService.unsubscribe(
      SocketServiceUtils.MediaEvent.PEER_JOINED_BROADCAST.id,
      socketServiceEventListener
    )
  }

  private fun handleWaitlistHostEvent(
    event: Int,
    payload: ByteArray?,
    hostEventsFlow: ProducerScope<WaitlistHostEvent>
  ) {
    when (event) {
      SocketServiceUtils.RoomEvent.GET_WAITING_ROOM_REQUESTS.id -> {
        if (payload == null) {
          return
        }
        try {
          val updatedWaitlist =
            GetWaitingRoomRequests.ADAPTER.decode(payload).requests.map {
              it.toWaitlistedParticipant()
            }
          hostEventsFlow.trySend(WaitlistHostEvent.WaitlistUpdated(updatedWaitlist))
        } catch (e: Exception) {
          // no-op
        }
      }
      SocketServiceUtils.MediaEvent.PEER_JOINED_BROADCAST.id -> {
        if (payload == null) {
          return
        }
        try {
          val peerId =
            PeerJoinBroadcastResponse.ADAPTER.decode(payload).participant?.peer_id ?: return
          hostEventsFlow.trySend(WaitlistHostEvent.PeerJoinedMeeting(peerId))
        } catch (e: Exception) {
          // no-op
        }
      }
      else -> {}
    }
  }

  companion object {
    internal fun WaitingRoomRequest.toWaitlistedParticipant(): WaitlistedParticipant {
      return WaitlistedParticipant(peer_id, user_id, display_name)
    }
  }
}
