package io.dyte.core.plugins.factory

import io.dyte.core.controllers.IEventController
import io.dyte.core.events.EventEmitter
import io.dyte.core.events.InternalEvents
import io.dyte.core.feat.DytePlugin
import io.dyte.core.feat.DytePlugins
import io.dyte.core.models.DyteParticipants
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.network.IApiClient
import io.dyte.core.network.info.PluginPermissions
import io.dyte.core.network.models.PluginDataResponse
import io.dyte.core.platform.IDytePlatformUtilsProvider
import io.dyte.core.plugins.PluginChatSender
import io.dyte.core.plugins.PluginsController
import io.dyte.core.plugins.PluginsInternalRoomJoinEventAdapter
import io.dyte.core.plugins.roomnode.RoomNodePluginsController
import io.dyte.core.plugins.socketservice.DefaultPluginsSocketHandler
import io.dyte.core.plugins.socketservice.SocketServicePluginsController
import io.dyte.core.socket.IRoomNodeSocketService
import io.dyte.core.socket.ISocketMessageResponseParser
import io.dyte.core.socket.socketservice.ISockratesSocketService
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.DelicateCoroutinesApi
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.newSingleThreadContext
import kotlinx.serialization.json.Json

/**
 * Factory to create [DytePlugins] instance based on **plugin_socket_server** feature flag. The
 * feature flag tells whether Plugins in the current meeting use RoomNode or SocketService.
 */
internal class DytePluginsFactory(
  private val meetingTitle: String,
  private val roomName: String,
  private val organizationId: String,
  private val selfPluginsPermissions: PluginPermissions,
  private val roomNodeSocket: IRoomNodeSocketService,
  private val socketService: ISockratesSocketService,
  private val pluginsApi: IApiClient,
  private val dyteBaseUrl: String,
  private val webViewProvider: IDytePlatformUtilsProvider,
  private val internalEventsEmitter: EventEmitter<InternalEvents>,
  private val socketIoMessageResponseParser: ISocketMessageResponseParser,
  private val pluginChatSender: PluginChatSender,
  private val dyteParticipants: DyteParticipants,
  private val dyteSelfParticipant: DyteSelfParticipant,
  private val flutterNotifier: IEventController
) {
  @OptIn(ExperimentalCoroutinesApi::class, DelicateCoroutinesApi::class)
  suspend fun create(
    pluginSocketServer: String,
    v1PluginsConfig: String,
  ): DytePlugins {
    val json = Json {
      encodeDefaults = true
      isLenient = true
    }
    val scope = CoroutineScope(newSingleThreadContext("DytePlugins"))
    val v1Plugins = v1PluginsConfig.split(',').toSet()
    val getPluginsResponse = pluginsApi.getPlugins()
    val getPluginsResponseData: List<PluginDataResponse> = getPluginsResponse.data.plugins
    val dytePluginFactory = DytePluginFactory(pluginsApi, dyteBaseUrl, webViewProvider)
    val pluginsInternalRoomJoinEventAdapter = PluginsInternalRoomJoinEventAdapter()
    val pluginsController: PluginsController =
      when (pluginSocketServer) {
        "socket-service" -> {
          val pluginsSocketHandler = DefaultPluginsSocketHandler(socketService, json)
          val allPlugins =
            getSocketServicePluginsFromApiResponse(
              getPluginsResponseData,
              v1Plugins,
              dytePluginFactory,
              pluginSocketServer
            )
          SocketServicePluginsController(
            meetingTitle,
            roomName,
            organizationId,
            pluginsSocketHandler,
            pluginChatSender,
            dyteParticipants,
            dyteSelfParticipant,
            scope,
            selfPluginsPermissions,
            pluginsInternalRoomJoinEventAdapter,
            flutterNotifier,
            allPlugins
          )
        }
        else -> {
          val allPlugins =
            getRoomNodePluginsFromApiResponse(
              getPluginsResponseData,
              v1Plugins,
              dytePluginFactory,
              pluginSocketServer
            )
          RoomNodePluginsController(
            json,
            roomNodeSocket,
            socketIoMessageResponseParser,
            dyteSelfParticipant.id,
            scope,
            selfPluginsPermissions,
            pluginsInternalRoomJoinEventAdapter,
            flutterNotifier,
            allPlugins
          )
        }
      }

    internalEventsEmitter.addListener(pluginsInternalRoomJoinEventAdapter)
    return DytePlugins(pluginsController, scope)
  }

  private fun getSocketServicePluginsFromApiResponse(
    pluginsResponseData: List<PluginDataResponse>,
    v1Plugins: Set<String>,
    pluginFactory: DytePluginFactory,
    pluginSocketServer: String
  ): List<DytePlugin> {
    val socketServicePluginsResponse = pluginsResponseData.filter { !v1Plugins.contains(it.id) }
    return getDytePluginsFromApiResponse(
      socketServicePluginsResponse,
      pluginFactory,
      pluginSocketServer
    )
  }

  private fun getRoomNodePluginsFromApiResponse(
    pluginsResponseData: List<PluginDataResponse>,
    v1Plugins: Set<String>,
    pluginFactory: DytePluginFactory,
    pluginSocketServer: String,
  ): List<DytePlugin> {
    val roomNodePluginsResponse = pluginsResponseData.filter { v1Plugins.contains(it.id) }
    return getDytePluginsFromApiResponse(roomNodePluginsResponse, pluginFactory, pluginSocketServer)
  }

  private fun getDytePluginsFromApiResponse(
    getPluginsApiResponse: List<PluginDataResponse>,
    pluginFactory: DytePluginFactory,
    pluginSocketServer: String
  ): List<DytePlugin> {
    val dytePlugins = mutableListOf<DytePlugin>()
    for (pluginData in getPluginsApiResponse) {
      val plugin =
        pluginFactory.create(
          pluginData.id,
          pluginData.name,
          pluginData.description,
          pluginData.picture,
          pluginData.private,
          pluginData.staggered,
          pluginData.baseURL,
          pluginSocketServer
        )
      dytePlugins.add(plugin)
    }
    return dytePlugins
  }
}
