package io.dyte.core.controllers

import io.dyte.core.events.InternalEvents
import io.dyte.core.models.ActiveTabType
import io.dyte.core.models.ActiveTabType.PLUGIN
import io.dyte.core.models.ActiveTabType.SCREENSHARE
import io.dyte.core.models.DyteMeetingState
import io.dyte.core.models.DyteMeetingType
import io.dyte.core.network.info.DyteDesignToken
import io.dyte.core.network.info.ParticipantInfo
import io.dyte.core.observability.DyteLogger
import io.dyte.core.socket.SocketMessageEventListener
import io.dyte.core.socket.events.InboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.payloadmodel.InboundMeetingEvent
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketRoomMessage
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketJoinRoomModel
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.jsonPrimitive
import kotlinx.serialization.json.put

/**
 * Meta controller
 *
 * Contains meta details about the meeting
 *
 * @constructor Create empty Meta controller
 */
internal class MetaController(
  private val meetingInfo: MeetingInfo,
  private val controllerContainer: IControllerContainer,
  private val scope: CoroutineScope
) : IMetaController {

  internal var _meetingState: DyteMeetingState = DyteMeetingState.NotInitialised
  override val meetingState: DyteMeetingState
    get() = _meetingState

  private lateinit var participantInfo: ParticipantInfo
  private var meetingUserPeerId: String =
    controllerContainer.platformUtilsProvider.getPlatformUtils().getUuid()
  private lateinit var roomUuid: String
  private var useHive: Boolean? = null
  private lateinit var designToken: DyteDesignToken

  fun init() {
    meetingInfo.meetingId =
      controllerContainer.platformUtilsProvider
        .getPlatformUtils()
        .decodeAuthToken(meetingInfo.authToken)
    meetingInfo.isV2Meeting = meetingInfo.meetingId != null
    setupRoomJoinedListeners()
  }

  override fun getDisplayName(): String {
    return participantInfo.name
  }

  override fun getRoomName(): String {
    return meetingInfo.getRoomName()
  }

  override fun getAuthToken(): String {
    return meetingInfo.authToken
  }

  override fun getPeerId(): String {
    return meetingUserPeerId
  }

  override fun getOrgId(): String {
    return if (::participantInfo.isInitialized) participantInfo.organizationId else ""
  }

  override fun getMeetingTitle(): String {
    return meetingInfo.meetingTitle ?: ""
  }

  override fun getMeetingStatedTimestamp(): String {
    return meetingInfo.meetingStartedAt ?: ""
  }

  override fun setMeetingTitle(title: String) {
    meetingInfo.meetingTitle = title
  }

  override fun setMeetingStartedTimestamp(timestamp: String) {
    meetingInfo.meetingStartedAt = timestamp
  }

  override fun getMeetingId(): String {
    return meetingInfo.getRoomName()
  }

  override fun isV2AuthToken(): Boolean {
    return meetingInfo.isV2Meeting
  }

  override fun setParticipantInfo(participantInfo: ParticipantInfo) {
    this.participantInfo = participantInfo
  }

  override fun isWebinar(): Boolean {
    return participantInfo.presetInfo.viewType == "WEBINAR"
  }

  override fun isGroupCall(): Boolean {
    return participantInfo.presetInfo.viewType == "GROUP_CALL"
  }

  override fun getMeetingConfig(): MeetingConfig {
    return this.meetingInfo.config
  }

  override fun setRoomUuid(roomUuid: String) {
    this.roomUuid = roomUuid
  }

  override fun getRoomUuid(): String {
    return roomUuid
  }

  override fun getIsHive(): Boolean {
    return useHive!!
  }

  override fun setIsHive(isHive: Boolean) {
    useHive = isHive
  }

  override fun getMeetingType(): DyteMeetingType {
    return DyteMeetingType.fromViewTypeString(participantInfo.presetInfo.viewType)
  }

  override fun getDesignToken(): DyteDesignToken {
    return designToken
  }

  override fun setDesignToken(designToken: DyteDesignToken) {
    this.designToken = designToken
  }

  override fun refreshPeerId(): String {
    val newPeerId = controllerContainer.platformUtilsProvider.getPlatformUtils().getUuid()
    meetingUserPeerId = newPeerId
    return newPeerId
  }

  private fun setupRoomJoinedListeners() {
    controllerContainer.internalEventEmitter.addListener(
      object : InternalEvents {
        override fun onRoomJoined(webSocketJoinRoomModel: WebSocketJoinRoomModel) {
          super.onRoomJoined(webSocketJoinRoomModel)
          setUpSpotlightSocketListner()
        }
      }
    )
  }

  override fun syncTab(id: String, tabType: ActiveTabType) {
    scope.launch {
      if (
        controllerContainer.selfController.selfPermissions.miscellaneous.canSpotLight &&
          !getIsHive()
      ) {
        val activeTab = buildJsonObject {
          put("id", id)
          put("type", tabType.getValue())
        }

        val payload = buildJsonObject {
          put("roomMessageType", "spotlight")
          put("userId", controllerContainer.selfController.getSelf().userId)
          put("currentTab", activeTab)
        }
        controllerContainer.roomNodeSocketService.sendMessage(
          OutboundMeetingEventType.ROOM_MESSAGE,
          payload
        )
      } else {
        DyteLogger.error(
          "MetaController::syncTabSpotlight:: self not allowed to sync active tab. If you still want then please allow from Preset Editor and rejoin"
        )
      }
    }
  }

  private fun setUpSpotlightSocketListner() {
    controllerContainer.roomNodeSocketService.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_ROOM_MESSAGE,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          val roomMessagePayload = (event.payload as WebSocketRoomMessage).payload.jsonObject
          val roomMessageType = roomMessagePayload?.get("roomMessageType")?.jsonPrimitive?.content
          if (roomMessageType == "spotlight") {
            val currentTab = roomMessagePayload?.get("currentTab")?.jsonObject
            val tabId = currentTab?.get("id")?.jsonPrimitive?.content
            val tabTypeString = currentTab?.get("type")?.jsonPrimitive?.content
            if (tabId != null && tabTypeString != null) {
              val activeTab = getActiveTab(tabTypeString)
              if (activeTab != null) {
                controllerContainer.eventController.triggerEvent(
                  DyteEventType.OnActiveTabUpdate(tabId, activeTab)
                )
              }
            } else {
              DyteLogger.error("MetaController::setUpSpotlightSocketListner::packet is malformed")
            }
          }
        }
      }
    )
  }

  companion object {
    private fun ActiveTabType.getValue(): String {
      return when (this) {
        SCREENSHARE -> "screenshare"
        PLUGIN -> "plugin"
      }
    }

    private fun getActiveTab(value: String): ActiveTabType? {
      return when (value) {
        "screenshare" -> SCREENSHARE
        "plugin" -> PLUGIN
        else -> null
      }
    }
  }
}

internal data class MeetingInfo(
  val authToken: String,
  val config: MeetingConfig,
  val baseUrl: String
) {
  internal var isV2Meeting: Boolean = false
  internal var meetingId: String? = null
  internal var roomName: String? = null
  internal var viewType: String = "GROUP_CALL"
  internal var meetingTitle: String? = null
  internal var meetingStartedAt: String? = null

  fun getRoomName(): String {
    return (if (isV2Meeting) meetingId else this.roomName) ?: ""
  }
}

data class MeetingConfig(
  val enableAudio: Boolean,
  val enableVideo: Boolean,
)

interface IMetaController {
  val meetingState: DyteMeetingState

  fun getPeerId(): String

  fun getDisplayName(): String

  fun getRoomName(): String

  fun getAuthToken(): String

  fun getOrgId(): String

  fun setMeetingTitle(title: String)

  fun setMeetingStartedTimestamp(timestamp: String)

  fun getMeetingTitle(): String

  fun getMeetingStatedTimestamp(): String

  fun getMeetingId(): String

  fun getMeetingConfig(): MeetingConfig

  fun isWebinar(): Boolean

  fun isGroupCall(): Boolean

  fun setParticipantInfo(participantInfo: ParticipantInfo)

  fun isV2AuthToken(): Boolean

  fun setRoomUuid(roomUuid: String)

  fun getRoomUuid(): String

  fun getIsHive(): Boolean

  fun setIsHive(isHive: Boolean)

  fun getMeetingType(): DyteMeetingType

  fun getDesignToken(): DyteDesignToken

  fun setDesignToken(designToken: DyteDesignToken)

  /**
   * Generates a new peerId for local user.
   *
   * **NOTE**: This is needed as a part of a workaround that we have done to make the socket-service
   * work properly after the reconnection.
   *
   * @return new peerId
   */
  fun refreshPeerId(): String

  fun syncTab(id: String, tabType: ActiveTabType)
}
