package io.dyte.core.chat.roomnode

import io.dyte.core.Result
import io.dyte.core.chat.BaseChatController
import io.dyte.core.chat.ChatFileUploader
import io.dyte.core.chat.MessageType
import io.dyte.core.chat.models.GetMessagesResult
import io.dyte.core.feat.DyteChatMessage
import io.dyte.core.feat.DyteFileMessage
import io.dyte.core.feat.DyteImageMessage
import io.dyte.core.feat.DyteTextMessage
import io.dyte.core.observability.DyteLogger
import io.dyte.core.platform.IDytePlatformUtilsProvider
import io.dyte.core.socket.ISocketMessageResponseParser
import io.dyte.core.socket.RoomNodeSocketService
import io.dyte.core.socket.SocketMessageEventListener
import io.dyte.core.socket.events.InboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType.GET_CHAT_MESSAGES
import io.dyte.core.socket.events.OutboundMeetingEventType.SEND_CHAT_MESSAGE
import io.dyte.core.socket.events.payloadmodel.InboundMeetingEvent
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketChatMessage
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketChatMessagesModel
import kotlinx.coroutines.CoroutineScope
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.put

internal class ChatRoomNodeController(
  private val platformUtilsProvider: IDytePlatformUtilsProvider,
  canSendChatText: Boolean,
  canSendChatFile: Boolean,
  canSendPrivateChatText: Boolean,
  canSendPrivateChatFile: Boolean,
  private val peerID: String,
  private val displayName: String,
  private val roomNodeSocket: RoomNodeSocketService,
  private val socketIoMessageResponseParser: ISocketMessageResponseParser,
  chatFileUploader: ChatFileUploader,
  scope: CoroutineScope
) :
  BaseChatController(
    platformUtilsProvider,
    canSendChatText,
    canSendChatFile,
    canSendPrivateChatText,
    canSendPrivateChatFile,
    chatFileUploader,
    scope
  ) {
  init {
    setupEvents()
  }

  private fun setupEvents() {
    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_ON_CHAT_MESSAGE,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          DyteLogger.info("DyteChat::onMessageEvent::on chat message")
          val messageModel = event.payload as WebSocketChatMessage
          try {
            val convertedMessage = fromSocketIoChatMessage(messageModel)
            handleMessage(convertedMessage)
          } catch (e: Exception) {
            e.printStackTrace()
          }
        }
      }
    )
    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_ON_CHAT_MESSAGES,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          DyteLogger.info("DyteChat::onMessageEvent::on chat messages")
          val messagesModel = event.payload as WebSocketChatMessagesModel
          val convertedMessages = fromSocketIoChatMessageMultiple(messagesModel.messages)
          handleMultipleMessages(convertedMessages)
        }
      }
    )
  }

  override suspend fun sendFileMessageSocket(fileUrl: String, fileName: String, fileSize: Long) {
    sendFileMessageSocket(fileUrl, fileName, fileSize, listOf())
  }

  override suspend fun sendFileMessageSocket(
    fileUrl: String,
    fileName: String,
    fileSize: Long,
    peerIds: List<String>
  ) {
    if (peerIds.isNotEmpty()) {
      DyteLogger.error("DyteChat::sendFileMessageSocket::private_chat_unsupported")
      return
    }
    val messagePayload = buildJsonObject {
      put(PAYLOAD_KEY_USER_ID, peerID)
      put(PAYLOAD_KEY_DISPLAY_NAME, displayName)
      put(PAYLOAD_KEY_MESSAGE_TYPE, MessageType.FILE.type)
      put(PAYLOAD_KEY_LINK, fileUrl)
      put(PAYLOAD_KEY_FILE_NAME, fileName)
      put(PAYLOAD_KEY_FILE_SIZE, fileSize)
      put(PAYLOAD_KEY_TIME, platformUtilsProvider.getPlatformUtils().getCurrentTime())
    }
    sendMessageToRoom(messagePayload)
  }

  override suspend fun getMessagesPaginated(
    timestamp: Long,
    size: Int,
    offset: Int,
    reversed: Boolean,
    channelId: String?
  ): Result<GetMessagesResult, String> {
    DyteLogger.warn("DyteChat::getMessagesPaginated::not supported on room-node")
    return Result.Success(GetMessagesResult(emptyList(), false))
  }

  override suspend fun getChatMessages(): List<DyteChatMessage> {
    DyteLogger.info("DyteChat::getChatMessages::")
    val chatMessages =
      roomNodeSocket.sendMessageParsed<WebSocketChatMessagesModel>(GET_CHAT_MESSAGES, null)
    return fromSocketIoChatMessageMultiple(chatMessages.messages)
  }

  override suspend fun sendTextMessageSocket(message: String, peerIds: List<String>) {
    DyteLogger.info("DyteChat::sendTextMessageToSocket::")
    if (peerIds.isNotEmpty()) {
      DyteLogger.error("DyteChat::sendTextMessage::private_chat_unsupported")
      return
    }
    val messagePayload = buildJsonObject {
      put(PAYLOAD_KEY_USER_ID, peerID)
      put(PAYLOAD_KEY_DISPLAY_NAME, displayName)
      put(PAYLOAD_KEY_MESSAGE_TYPE, MessageType.TEXT.type)
      put(PAYLOAD_KEY_MESSAGE, message)
      put(PAYLOAD_KEY_TIME, platformUtilsProvider.getPlatformUtils().getCurrentTime())
    }
    sendMessageToRoom(messagePayload)
  }

  override suspend fun sendImageMessageSocket(path: String, peerIds: List<String>) {
    DyteLogger.info("DyteChat::sendImageMessageSocket::")
    if (peerIds.isNotEmpty()) {
      DyteLogger.error("sendImageMessageSocket::private_chat_unsupported")
      return
    }
    val messagePayload = buildJsonObject {
      put(PAYLOAD_KEY_USER_ID, peerID)
      put(PAYLOAD_KEY_DISPLAY_NAME, displayName)
      put(PAYLOAD_KEY_MESSAGE_TYPE, MessageType.IMAGE.type)
      put(PAYLOAD_KEY_LINK, path)
      put(PAYLOAD_KEY_TIME, platformUtilsProvider.getPlatformUtils().getCurrentTime())
    }
    sendMessageToRoom(messagePayload)
  }

  private suspend fun sendMessageToRoom(messagePayload: JsonObject) {
    roomNodeSocket.sendMessageAsync(SEND_CHAT_MESSAGE, messagePayload)
  }

  private fun fromSocketIoChatMessageMultiple(
    socketMessages: List<WebSocketChatMessage>?
  ): List<DyteChatMessage> {
    val convertedMessages = mutableListOf<DyteChatMessage>()
    if (socketMessages != null) {
      for (socketIoChatMessage in socketMessages) {
        try {
          val socketChatMessage = fromSocketIoChatMessage(socketIoChatMessage)
          convertedMessages.add(socketChatMessage)
        } catch (e: Exception) {
          DyteLogger.error("ChatSocketHandler::getChatMessagesSocketIO::error", e)
        }
      }
    }
    return convertedMessages
  }

  private fun fromSocketIoChatMessage(socketIoChatMessage: WebSocketChatMessage): DyteChatMessage {
    when (socketIoChatMessage.type) {
      0 -> {
        return DyteTextMessage(
          message = socketIoChatMessage.message ?: error("Text message is null"),
          userId = socketIoChatMessage.userId,
          displayName = socketIoChatMessage.displayName,
          time = parseTime(socketIoChatMessage.time),
          pluginId = null,
          read = false
        )
      }
      1 -> {
        return DyteImageMessage(
          link = socketIoChatMessage.link ?: "",
          userId = socketIoChatMessage.userId,
          displayName = socketIoChatMessage.displayName,
          time = parseTime(socketIoChatMessage.time),
          pluginId = null,
          read = false
        )
      }
      2 -> {
        return DyteFileMessage(
          name = socketIoChatMessage.name ?: "",
          size = socketIoChatMessage.size ?: 0L,
          link = socketIoChatMessage.link ?: "",
          userId = socketIoChatMessage.userId,
          displayName = socketIoChatMessage.displayName,
          time = parseTime(socketIoChatMessage.time),
          pluginId = null,
          read = false,
        )
      }
      else -> {
        throw UnsupportedOperationException(
          "Message type ${socketIoChatMessage.type} not supported"
        )
      }
    }
  }

  companion object {

    private const val PAYLOAD_KEY_USER_ID = "userId"
    private const val PAYLOAD_KEY_DISPLAY_NAME = "displayName"
    private const val PAYLOAD_KEY_MESSAGE_TYPE = "type"
    private const val PAYLOAD_KEY_LINK = "link"
    private const val PAYLOAD_KEY_FILE_NAME = "name"
    private const val PAYLOAD_KEY_FILE_SIZE = "size"
    private const val PAYLOAD_KEY_TIME = "time"
    private const val PAYLOAD_KEY_MESSAGE = "message"
  }
}
