package io.dyte.core.host

import io.dyte.core.DyteError
import io.dyte.core.HostControlErrorCodes.Kick_All
import io.dyte.core.HostControlErrorCodes.Kick_Peer
import io.dyte.core.HostControlErrorCodes.Mute_All_Audio
import io.dyte.core.HostControlErrorCodes.Mute_All_Video
import io.dyte.core.HostControlErrorCodes.Mute_Audio
import io.dyte.core.HostControlErrorCodes.Mute_Video
import io.dyte.core.HostControlErrorCodes.Pin_Peer
import io.dyte.core.HostControlErrorCodes.Unpin_Peer
import io.dyte.core.Result
import io.dyte.core.controllers.IRoomNodeController
import io.dyte.core.network.info.HostPermissions
import io.dyte.core.observability.DyteLogger
import io.dyte.core.socket.socketservice.ISockratesSocketService
import io.dyte.core.socket.socketservice.SocketServiceUtils
import io.dyte.webrtc.MediaStreamTrackKind
import media.edge.GlobalPeerPinningRequest
import media.edge.HostMediaControlForAllPeerRequest
import media.edge.HostMediaControlForAllPeerResponse
import media.edge.HostMediaControlForPeerRequest
import media.edge.HostMediaControlForPeerResponse
import socket.room.RemoveParticipantsRequest

internal class HostSocketServiceController(
  private val selfHostPermissions: HostPermissions,
  private val socketService: ISockratesSocketService,
  private val mediaRoomController: IRoomNodeController
) : IHostController {
  private val logger = DyteLogger

  override suspend fun kickPeer(peerId: String): Result<Unit, DyteError> {
    logger.info("DyteHost::kickPeer::$peerId")
    try {
      val kickedFromMediaRoom = mediaRoomController.kickPeer(peerId)
      if (!kickedFromMediaRoom) {
        logger.error("HostController::kickPeer::failed to kick peer $peerId from media room")
        return Result.Failure(DyteError(Kick_Peer, "failed to kick peer $peerId"))
      }
      // Kick peer from socketService room
      val req = RemoveParticipantsRequest(peer_ids = listOf(peerId))
      socketService.send(
        event = SocketServiceUtils.RoomEvent.KICK.id,
        payload = RemoveParticipantsRequest.ADAPTER.encode(req)
      )
      return Result.Success(Unit)
    } catch (e: Exception) {
      DyteLogger.error("DyteParticipant::kick_peer_failed::", e)
      return Result.Failure(
        DyteError(Kick_Peer, "HostSocketService::kick_peer_failed: and raised Exception")
      )
    }
  }

  override suspend fun kickAll(): Result<Unit, DyteError> {
    logger.info("DyteHost::kickAll")
    if (!selfHostPermissions.canKickParticipant) {
      logger.error("HostController::kickAll::self not permitted")
      return Result.Failure(
        DyteError(Kick_All, "Unauthorized: User does not have permission to kick peers.")
      )
    }

    return try {
      // Kick all from media room
      val kickedFromMediaRoom = mediaRoomController.kickAll()
      if (!kickedFromMediaRoom) {
        logger.error("HostController::kickAll::failed to kick all from media room")
        return Result.Failure(DyteError(Kick_All, "failed to kick all participants"))
      }
      // Kick all from socketService room
      socketService.send(event = SocketServiceUtils.RoomEvent.KICK_ALL.id, payload = null)
      return Result.Success(Unit)
    } catch (e: Exception) {
      logger.error("DyteParticipants::kickAll", e)
      Result.Failure(DyteError(Kick_All, "HostSocketService::kick_all_failed:: with Exception"))
    }
  }

  override suspend fun muteAllAudio(allowUnmute: Boolean?): Result<Unit, DyteError> {
    DyteLogger.info("DyteHost::muteAllAudio::")
    if (hostControlForAll(MediaStreamTrackKind.Audio)) {
      return Result.Success(Unit)
    }
    return Result.Failure(DyteError(Mute_All_Audio))
  }

  override suspend fun muteAllVideo(): Result<Unit, DyteError> {
    DyteLogger.info("DyteHost::muteAllVideo::")
    if (hostControlForAll(MediaStreamTrackKind.Video)) {
      return Result.Success(Unit)
    }
    return Result.Failure(DyteError(Mute_All_Video))
  }

  override suspend fun muteAudio(peerId: String): Result<Unit, DyteError> {
    DyteLogger.info("DyteHost::muteAudio::$peerId")
    if (hostControlForPeer(peerId, MediaStreamTrackKind.Audio)) {
      return Result.Success(Unit)
    }
    return Result.Failure(DyteError(Mute_Audio))
  }

  override suspend fun muteVideo(peerId: String): Result<Unit, DyteError> {
    DyteLogger.info("DyteHost::muteVideo::$peerId")
    if (hostControlForPeer(peerId, MediaStreamTrackKind.Video)) {
      return Result.Success(Unit)
    }
    return Result.Failure(DyteError(Mute_Video))
  }

  override suspend fun pinPeer(peerId: String): Result<Unit, DyteError> {
    DyteLogger.info("DyteHost::pinPeer::$peerId")
    if (pinPeerInternal(peerId)) {
      return Result.Success(Unit)
    }
    return Result.Failure(DyteError(Pin_Peer))
  }

  override suspend fun unpinPeer(): Result<Unit, DyteError> {
    DyteLogger.info("DyteHost::unpinPeer::")
    if (pinPeerInternal()) {
      return Result.Success(Unit)
    }
    return Result.Failure(DyteError(Unpin_Peer))
  }

  private suspend fun hostControlForAll(kind: MediaStreamTrackKind): Boolean {
    DyteLogger.info("DyteHost::hostControlForAll::kind::${kind.name}")
    return try {
      val req =
        HostMediaControlForAllPeerRequest(
          audio = kind == MediaStreamTrackKind.Audio,
          screen_share = false,
          video = kind == MediaStreamTrackKind.Video
        )

      val res =
        socketService.requestResponse(
          event = SocketServiceUtils.MediaEvent.HOST_CONTROL_ALL_PEERS.id,
          payload = HostMediaControlForAllPeerRequest.ADAPTER.encode(req)
        )

      !(res == null || HostMediaControlForAllPeerResponse.ADAPTER.decode(res).status != "success")
    } catch (e: Exception) {
      DyteLogger.error("HostSocketService::host_control_for_all_failed::", e)
      false
    }
  }

  private suspend fun hostControlForPeer(peerId: String, kind: MediaStreamTrackKind): Boolean {
    DyteLogger.info("DyteHost::hostControlForPeer::kind::${kind.name}::$peerId")
    return try {
      val req =
        HostMediaControlForPeerRequest(
          audio = kind == MediaStreamTrackKind.Audio,
          scree_share = false,
          video = kind == MediaStreamTrackKind.Video,
          participant_id = peerId
        )

      val res =
        socketService.requestResponse(
          event = SocketServiceUtils.MediaEvent.HOST_CONTROL_PEER.id,
          payload = HostMediaControlForPeerRequest.ADAPTER.encode(req)
        )

      !(res == null || HostMediaControlForPeerResponse.ADAPTER.decode(res).status != "success")
    } catch (e: Exception) {
      DyteLogger.error("HostSocketService::host_control_for_peer_failed::", e)
      false
    }
  }

  private suspend fun pinPeerInternal(peerId: String = ""): Boolean {
    return try {
      val req =
        GlobalPeerPinningRequest(
          participant_id = peerId,
        )

      socketService.send(
        event = SocketServiceUtils.MediaEvent.GLOBAL_PIN_PEER.id,
        payload = GlobalPeerPinningRequest.ADAPTER.encode(req)
      )

      true
    } catch (e: Exception) {
      DyteLogger.error("HostSocketService::pin_peer_failed::", e)
      false
    }
  }
}
