package io.dyte.core.chat.channel

import io.dyte.core.Result
import io.dyte.core.chat.ChatFileUploader
import io.dyte.core.chat.ChatFileUploader.UploadFileResult
import io.dyte.core.chat.ChatMessageMapper
import io.dyte.core.chat.MessageType
import io.dyte.core.chat.models.ChatChannel
import io.dyte.core.chat.models.ChatChannel.ChannelVisibility
import io.dyte.core.chat.models.ChatChannelUpdateParams
import io.dyte.core.chat.models.SocketServiceFileMessage
import io.dyte.core.chat.socketservice.ChatSocketEvent
import io.dyte.core.events.EventEmitter
import io.dyte.core.listeners.DyteChatChannelEventsListener
import io.dyte.core.models.DyteParticipant
import io.dyte.core.network.info.ChatChannelPermissions
import io.dyte.core.network.info.ChatChannelPermissions.ChatChannelPermission
import io.dyte.core.observability.DyteLogger
import io.dyte.core.platform.IDytePlatformUtils
import io.dyte.core.platform.Uri
import io.dyte.core.socket.socketservice.ISockratesSocketService
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import io.dyte.core.socket.socketservice.SocketServiceUtils
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import socket.chat.ChannelMember as SocketChannelMember
import socket.chat.ChatChannel as SocketChatChannel
import socket.chat.CreateChatChannelRequest
import socket.chat.GetChatChannelMembersResponse
import socket.chat.GetChatChannelRequest
import socket.chat.GetChatChannelResponse
import socket.chat.SendChatMessageToChannelRequest
import socket.chat.SendChatMessageToRoomResponse
import socket.chat.UpdateChatChannelRequest

internal interface ChatChannelController {
  val channels: List<ChatChannel>

  suspend fun loadAllChatChannels()

  suspend fun createChannel(
    name: String,
    memberIds: List<String>,
    displayPictureUrl: String?,
    visibility: ChannelVisibility,
    isDirectMessage: Boolean
  ): Result<ChatChannel, String>

  suspend fun updateChannel(
    channelId: String,
    channelUpdateParams: ChatChannelUpdateParams
  ): Result<ChatChannel, String>

  suspend fun getChannelMembers(channelId: String): Result<List<DyteParticipant>, String>

  suspend fun sendTextMessageToChannel(channelId: String, message: String)

  suspend fun sendImageMessageToChannel(channelId: String, imageUri: Uri)

  suspend fun sendFileMessageToChannel(channelId: String, fileUri: Uri)
}

internal class DefaultChatChannelController(
  private val selfUserId: String,
  private val chatChannelPermissions: ChatChannelPermissions,
  private val chatFileUploader: ChatFileUploader,
  private val platformFileProvider: IDytePlatformUtils,
  private val socketService: ISockratesSocketService,
  private val chatMessageMapper: ChatMessageMapper,
  private val chatChannelEventEmitter: EventEmitter<DyteChatChannelEventsListener>,
  private val scope: CoroutineScope
) : ChatChannelController {
  private val chatChannelSocketEventListener =
    object : SocketServiceEventListener {
      override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
        scope.launch { handleChatChannelSocketEvent(event, payload) }
      }
    }

  private val logger = DyteLogger

  init {
    /* No need to unsubscribe explicitly. We clear all the SocketService listeners while leaving call. */
    socketService.subscribe(
      event = ChatChannelSocketEvent.CREATE_CHAT_CHANNEL.id,
      listener = chatChannelSocketEventListener
    )
    socketService.subscribe(
      event = ChatChannelSocketEvent.UPDATE_CHAT_CHANNEL.id,
      listener = chatChannelSocketEventListener
    )
    socketService.subscribe(
      event = ChatSocketEvent.SEND_MESSAGE_TO_ROOM.id,
      listener = chatChannelSocketEventListener
    )
  }

  override val channels: MutableList<ChatChannel> = mutableListOf()

  override suspend fun loadAllChatChannels() {
    try {
      val response =
        socketService.requestResponse(
          event = ChatSocketEvent.GET_ALL_CHAT_CHANNELS.id,
          payload = null
        )
          ?: kotlin.run {
            logger.warn("ChatChannelController::loadChatChannels::socket response is null")
            return
          }

      val getAllChatChannelResponse =
        try {
          GetChatChannelResponse.ADAPTER.decode(response)
        } catch (e: Exception) {
          logger.warn(
            "ChatChannelController::loadChatChannels::failed to decode getAllChatChannelResponse"
          )
          return
        }

      val chatChannels = mutableListOf<ChatChannel>()
      for (socketChatChannel in getAllChatChannelResponse.chat_channels) {
        try {
          val chatChannel = socketChatChannel.toDyteChatChannel(chatMessageMapper)
          chatChannels.add(chatChannel)
        } catch (e: Exception) {
          logger.warn(
            "ChatChannelController::loadChatChannels::socketChatChannel parsing failed",
            mapOf("error" to (e.message ?: ""))
          )
        }
      }

      channels.clear()
      channels.addAll(chatChannels)
      if (chatChannels.isNotEmpty()) {
        chatChannelEventEmitter.emitEvent { it.onChatChannelUpdates(channels) }
      }
    } catch (e: Exception) {
      logger.error("ChatChannelController::loadChatChannels::error", e)
    }
  }

  // TODO: move all the socket request and response parsing part to a socket handler class
  override suspend fun createChannel(
    name: String,
    memberIds: List<String>,
    displayPictureUrl: String?,
    visibility: ChannelVisibility,
    isDirectMessage: Boolean
  ): Result<ChatChannel, String> {
    logger.info("ChatChannelController.createChannel")

    if (chatChannelPermissions.createPermission == ChatChannelPermission.NONE) {
      logger.warn("ChatChannelController::createChannel::self not allowed to create channel")
      return Result.Failure("Not allowed to create channel")
    }

    if (
      visibility == ChannelVisibility.PUBLIC &&
        !(chatChannelPermissions.createPermission == ChatChannelPermission.PUBLIC ||
          chatChannelPermissions.createPermission == ChatChannelPermission.ALL)
    ) {
      logger.warn(
        "ChatChannelController::createChannel::self not allowed to create public channels"
      )
      return Result.Failure("Not allowed to create public channel")
    }

    if (
      visibility == ChannelVisibility.PRIVATE &&
        !(chatChannelPermissions.createPermission == ChatChannelPermission.PRIVATE ||
          chatChannelPermissions.createPermission == ChatChannelPermission.ALL)
    ) {
      logger.warn(
        "ChatChannelController::createChannel::self not allowed to create private channels"
      )
      return Result.Failure("Not allowed to create private channel")
    }

    if (name.isBlank()) {
      logger.warn("ChatChannelController::createChannel::channel name is blank")
      return Result.Failure("Channel name must not be blank")
    }

    val channelName = name.trim()
    val uniqueMemberIds =
      mutableSetOf<String>().apply {
        addAll(memberIds)
        add(selfUserId)
      }
    val createChatChannelRequest =
      CreateChatChannelRequest(
        channelName,
        uniqueMemberIds.toList(),
        displayPictureUrl,
        visibility.toSocketString(),
        isDirectMessage
      )
    val socketResponse =
      try {
        socketService.requestResponse(
          event = ChatChannelSocketEvent.CREATE_CHAT_CHANNEL.id,
          payload = CreateChatChannelRequest.ADAPTER.encode(createChatChannelRequest)
        )
      } catch (e: Exception) {
        logger.error("ChatChannelController::createChannel::socket request failed ${e.message}")
        return Result.Failure(e.message ?: "Failed to send create channel request")
      }
        ?: kotlin.run {
          logger.warn("ChatChannelController::createChannel::socket response is null")
          // This will happen only if there is an issue on server side
          return Result.Failure("Something went wrong!")
        }

    val createChatChannelPayload =
      try {
        GetChatChannelResponse.ADAPTER.decode(socketResponse)
      } catch (e: Exception) {
        logger.warn("ChatChannelController::createChannel::failed to decode GetChatChannelResponse")
        return Result.Failure("Invalid response from sever")
      }

    if (createChatChannelPayload.chat_channels.isEmpty()) {
      logger.info("ChatChannelController::createChannel::response missing created channel")
      return Result.Failure("Invalid response from sever")
    }

    val createdChatChannel =
      createChatChannelPayload.chat_channels[0].toDyteChatChannel(chatMessageMapper)
    return Result.Success(createdChatChannel)
  }

  // TODO: move all the socket request and response parsing part to a socket handler class
  override suspend fun updateChannel(
    channelId: String,
    channelUpdateParams: ChatChannelUpdateParams
  ): Result<ChatChannel, String> {
    logger.info("ChatChannelController.updateChannel")

    val channelToUpdate =
      channels.find { it.id == channelId } ?: return Result.Failure("Channel not found: $channelId")

    if (chatChannelPermissions.updatePermission == ChatChannelPermission.NONE) {
      logger.warn("ChatChannelController::updateChannel::self not allowed to update channel")
      return Result.Failure("Not allowed to update channel")
    }

    if (
      channelToUpdate.visibility == ChannelVisibility.PUBLIC &&
        !(chatChannelPermissions.updatePermission == ChatChannelPermission.PUBLIC ||
          chatChannelPermissions.updatePermission == ChatChannelPermission.ALL)
    ) {
      logger.warn(
        "ChatChannelController::updateChannel::self not allowed to update public channels"
      )
      return Result.Failure("Not allowed to update public channel")
    }

    if (
      channelToUpdate.visibility == ChannelVisibility.PRIVATE &&
        !(chatChannelPermissions.updatePermission == ChatChannelPermission.PRIVATE ||
          chatChannelPermissions.updatePermission == ChatChannelPermission.ALL)
    ) {
      logger.warn(
        "ChatChannelController::updateChannel::self not allowed to update private channels"
      )
      return Result.Failure("Not allowed to update private channel")
    }

    val updateChatChannelRequest =
      UpdateChatChannelRequest(
        chat_channel_id = channelToUpdate.id,
        display_name = channelUpdateParams.displayName ?: channelToUpdate.displayName,
        target_user_ids = channelUpdateParams.memberIds ?: channelToUpdate.memberIds,
        display_picture_url = channelUpdateParams.displayPictureUrl
            ?: channelToUpdate.displayPictureUrl,
        visibility = channelUpdateParams.visibility?.toSocketString()
            ?: channelToUpdate.visibility.toSocketString()
      )
    val socketResponse =
      try {
        socketService.requestResponse(
          event = ChatChannelSocketEvent.UPDATE_CHAT_CHANNEL.id,
          payload = UpdateChatChannelRequest.ADAPTER.encode(updateChatChannelRequest)
        )
      } catch (e: Exception) {
        logger.error("ChatChannelController::updateChannel::socket request failed ${e.message}")
        return Result.Failure(e.message ?: "Failed to send update channel request")
      }
        ?: kotlin.run {
          logger.warn("ChatChannelController::updateChannel::socket response is null")
          // This will happen only if there is an issue on server side
          return Result.Failure("Something went wrong!")
        }

    val updateChatChannelPayload =
      try {
        GetChatChannelResponse.ADAPTER.decode(socketResponse)
      } catch (e: Exception) {
        logger.warn("ChatChannelController::updateChannel::failed to decode GetChatChannelResponse")
        return Result.Failure("Invalid response from sever")
      }

    if (updateChatChannelPayload.chat_channels.isEmpty()) {
      logger.info("ChatChannelController::updateChannel::response missing updated channel")
      return Result.Failure("Invalid response from sever")
    }

    val updatedChatChannel =
      updateChatChannelPayload.chat_channels[0].toDyteChatChannel(chatMessageMapper)
    return Result.Success(updatedChatChannel)
  }

  override suspend fun getChannelMembers(channelId: String): Result<List<DyteParticipant>, String> {
    logger.info("ChatChannelController.getChannelMembers")

    if (channelId.isBlank()) {
      logger.warn("ChatChannelController::getChannelMembers::channel id is blank")
      return Result.Failure("channel id is blank")
    }

    // TODO: Confirm if we should check any permission here or not

    val getChannelMembersRequest = GetChatChannelRequest(channelId)
    val socketResponse =
      try {
        socketService.requestResponse(
          event = ChatChannelSocketEvent.GET_CHANNEL_MEMBERS.id,
          payload = GetChatChannelRequest.ADAPTER.encode(getChannelMembersRequest)
        )
      } catch (e: Exception) {
        logger.error("ChatChannelController::getChannelMembers::socket request failed ${e.message}")
        return Result.Failure(e.message ?: "Failed to request channel members from server")
      }
        ?: kotlin.run {
          logger.warn("ChatChannelController::getChannelMembers::socket response is null")
          return Result.Success(emptyList())
        }

    val getChatChannelMembersResponse =
      try {
        GetChatChannelMembersResponse.ADAPTER.decode(socketResponse)
      } catch (e: Exception) {
        logger.warn(
          "ChatChannelController::getChannelMembers::failed to decode GetChatChannelMembersResponse"
        )
        return Result.Success(emptyList())
      }

    val channelMembers =
      getChatChannelMembersResponse.channel_members.map { it.toDyteParticipant() }
    return Result.Success(channelMembers)
  }

  override suspend fun sendTextMessageToChannel(channelId: String, message: String) {
    DyteLogger.info("ChatChannelController.sendTextMessageToChannel")

    // TODO: Add permission check for sending message to channels

    if (channelId.isBlank()) {
      logger.info("ChatChannelController::sendTextMessageToChannel::channelId is blank")
      return
    }

    if (message.isBlank()) {
      logger.info("ChatChannelController::sendTextMessageToChannel::message is blank")
      return
    }

    try {
      sendMessageToChannelViaSocketService(channelId, MessageType.TEXT, message)
    } catch (e: Exception) {
      logger.error(
        "ChatChannelController::sendTextMessageToChannel::socket request failed ${e.message}"
      )
    }
  }

  override suspend fun sendImageMessageToChannel(channelId: String, imageUri: Uri) {
    DyteLogger.info("ChatChannelController.sendImageMessageToChannel")

    // TODO: Add permission check for sending message to channels

    if (channelId.isBlank()) {
      logger.info("ChatChannelController::sendImageMessageToChannel::channelId is blank")
      return
    }

    val imageFile =
      platformFileProvider.getPlatformFile(imageUri)
        ?: run {
          DyteLogger.error(
            "ChatChannelController::sendImageMessageToChannel::failed to read image content from given URI"
          )
          // throw Exception("Requested image doesn't exist or is missing read permission")
          return
        }

    if (imageFile.mimeType !in ACCEPTED_IMAGE_TYPES) {
      DyteLogger.error(
        "ChatChannelController::sendImageMessageToChannel::unsupported image type ${imageFile.mimeType}"
      )
      // throw UnsupportedOperationException("Unsupported image file found ->
      // ${imageFile.mimeType}")
      return
    }

    val imageGetLocation =
      when (val uploadImageResult = chatFileUploader.uploadFile(imageFile)) {
        is UploadFileResult.Uploaded -> {
          uploadImageResult.getLocation
        }
        is UploadFileResult.UploadFailed -> {
          DyteLogger.error(
            "ChatChannelController::sendImageMessageToChannel::${uploadImageResult.reason}"
          )
          // throw Exception("Failed to upload image")
          return
        }
      }

    try {
      sendMessageToChannelViaSocketService(channelId, MessageType.IMAGE, imageGetLocation)
    } catch (e: Exception) {
      logger.error(
        "ChatChannelController::sendImageMessageToChannel::socket request failed ${e.message}"
      )
    }
  }

  override suspend fun sendFileMessageToChannel(channelId: String, fileUri: Uri) {
    DyteLogger.info("ChatChannelController.sendFileMessageToChannel")

    // TODO: Add permission check for sending message to channels

    if (channelId.isBlank()) {
      logger.info("ChatChannelController::sendFileMessageToChannel::channelId is blank")
      return
    }

    val file =
      platformFileProvider.getPlatformFile(fileUri)
        ?: run {
          DyteLogger.error(
            "ChatChannelController::sendFileMessageToChannel::failed to read file content from given URI"
          )
          // throw Exception("Requested file doesn't exist or is missing read permission")
          return
        }

    val fileGetLocation =
      when (val uploadFileResult = chatFileUploader.uploadFile(file)) {
        is UploadFileResult.Uploaded -> {
          uploadFileResult.getLocation
        }
        is UploadFileResult.UploadFailed -> {
          DyteLogger.error(
            "ChatChannelController::sendFileMessageToChannel::${uploadFileResult.reason}"
          )
          // throw Exception("Failed to upload file")
          return
        }
      }

    try {
      val messagePayload = SocketServiceFileMessage(fileGetLocation, file.name, file.size)
      sendMessageToChannelViaSocketService(
        channelId,
        MessageType.FILE,
        Json.encodeToString(messagePayload)
      )
    } catch (e: Exception) {
      logger.error(
        "ChatChannelController::sendFileMessageToChannel::socket request failed ${e.message}"
      )
    }
  }

  private suspend fun sendMessageToChannelViaSocketService(
    channelId: String,
    messageType: MessageType,
    messagePayload: String
  ) {
    val sendChatMessageToChannelRequest =
      SendChatMessageToChannelRequest(channelId, messageType.type, messagePayload)
    socketService.send(
      event = ChatSocketEvent.SEND_MESSAGE_TO_CHANNEL.id,
      payload = SendChatMessageToChannelRequest.ADAPTER.encode(sendChatMessageToChannelRequest)
    )
  }

  private fun handleChatChannelSocketEvent(event: Int, payload: ByteArray?) {
    when (event) {
      ChatChannelSocketEvent.CREATE_CHAT_CHANNEL.id -> {
        if (payload == null) {
          return
        }
        val createChatChannelPayload =
          try {
            GetChatChannelResponse.ADAPTER.decode(payload)
          } catch (e: Exception) {
            logger.warn(
              "ChatChannelController::handle CREATE_CHAT_CHANNEL event::failed to decode GetChatChannelResponse"
            )
            return
          }
        if (createChatChannelPayload.chat_channels.isEmpty()) {
          logger.info(
            "ChatChannelController::handle CREATE_CHAT_CHANNEL event::response missing created channel"
          )
          return
        }

        val createdChatChannel =
          createChatChannelPayload.chat_channels[0].toDyteChatChannel(chatMessageMapper)
        channels.add(createdChatChannel)
        chatChannelEventEmitter.emitEvent {
          it.onChatChannelCreated(createdChatChannel)
          it.onChatChannelUpdates(channels)
        }
      }
      ChatChannelSocketEvent.UPDATE_CHAT_CHANNEL.id -> {
        if (payload == null) {
          return
        }
        val updateChatChannelPayload =
          try {
            GetChatChannelResponse.ADAPTER.decode(payload)
          } catch (e: Exception) {
            logger.warn(
              "ChatChannelController::handle UPDATE_CHAT_CHANNEL event::failed to decode GetChatChannelResponse"
            )
            return
          }
        if (updateChatChannelPayload.chat_channels.isEmpty()) {
          logger.info(
            "ChatChannelController::handle UPDATE_CHAT_CHANNEL event::response missing updated channel"
          )
          return
        }

        val updatedChatChannel =
          updateChatChannelPayload.chat_channels[0].toDyteChatChannel(chatMessageMapper)

        val updatedChannelIndex = channels.indexOfFirst { it.id == updatedChatChannel.id }
        if (updatedChannelIndex == -1) {
          return
        }
        channels[updatedChannelIndex] = updatedChatChannel
        chatChannelEventEmitter.emitEvent {
          it.onChatChannelUpdated(updatedChatChannel)
          it.onChatChannelUpdates(channels)
        }
      }
      ChatSocketEvent.SEND_MESSAGE_TO_ROOM.id -> {
        if (payload == null) {
          return
        }

        val sendChatMessageToRoomResponse =
          try {
            SendChatMessageToRoomResponse.ADAPTER.decode(payload)
          } catch (e: Exception) {
            DyteLogger.warn(
              "ChatChannelController::handle SEND_MESSAGE_TO_ROOM event::failed to decode SendChatMessageToRoomResponse"
            )
            return
          }
        val message = sendChatMessageToRoomResponse.message ?: return
        if (message.channel_id.isNullOrBlank()) {
          return
        }

        val dyteChatMessage = chatMessageMapper.toDyteChatMessage(message)
        val channelIndex = channels.indexOfFirst { it.id == dyteChatMessage.channelId }
        if (channelIndex == -1) {
          return
        }

        val channel = channels[channelIndex]
        val updatedChannel =
          channel.copy(
            latestMessage = dyteChatMessage,
            unreadMessageCount = channel.unreadMessageCount + 1
          )
        channels[channelIndex] = updatedChannel

        chatChannelEventEmitter.emitEvent {
          it.onNewChannelMessage(updatedChannel.id, dyteChatMessage)
          it.onChatChannelUpdated(updatedChannel)
          it.onChatChannelUpdates(channels)
        }
      }
      else -> {
        logger.info("ChatChannelController::handleChatChannelSocketEvent::invalid event $event")
      }
    }
  }

  companion object {
    /*
     * TODO: check if these kind of fields and methods which apply to whole Chat functionality can
     *  be moved to a ChatUtils object
     * */
    private val ACCEPTED_IMAGE_TYPES = listOf("image/jpeg", "image/png")

    private fun parseChannelVisibility(visibilityString: String): ChannelVisibility {
      if (visibilityString == "public") {
        return ChannelVisibility.PUBLIC
      }

      return ChannelVisibility.PRIVATE
    }

    private fun ChannelVisibility.toSocketString(): String {
      return when (this) {
        ChannelVisibility.PUBLIC -> "public"
        ChannelVisibility.PRIVATE -> "private"
      }
    }

    private fun SocketChatChannel.toDyteChatChannel(
      chatMessageMapper: ChatMessageMapper
    ): ChatChannel {
      val latestMessage =
        latest_message_and_unread_count?.message?.let { chatMessageMapper.toDyteChatMessage(it) }
      return ChatChannel(
        id = chat_channel_id,
        displayName = display_name,
        displayPictureUrl = display_picture_url,
        visibility = parseChannelVisibility(visibility),
        isDirectMessage = is_direct_message,
        latestMessage = latestMessage,
        unreadMessageCount = latest_message_and_unread_count?.unread_count ?: 0L,
        memberIds = target_user_ids
      )
    }

    private fun SocketChannelMember.toDyteParticipant(): DyteParticipant {
      return DyteParticipant(
        userId = id,
        name = name,
        picture = picture,
        customParticipantId = custom_participant_id
      )
    }
  }
}

internal enum class ChatChannelSocketEvent(slug: Int) {
  CREATE_CHAT_CHANNEL(0),
  GET_CHAT_CHANNEL(1),
  DEPRECATED_GET_ALL_CHAT_CHANNELS(2),
  GET_CHANNEL_MEMBERS(3),
  UPDATE_CHAT_CHANNEL(4);

  private val base: Int = 4
  val id: Int = SocketServiceUtils.generateSocketEventId(base, slug)
}
