package io.dyte.core.hive

import io.dyte.core.observability.DyteLogger
import io.dyte.core.platform.IDytePlatformUtils
import io.dyte.core.socket.socketservice.ISockratesSocketService
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import io.dyte.core.socket.socketservice.SocketServiceUtils
import io.dyte.core.utils.JsonUtils.encodeToByteString
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonObject
import media.edge.PeerJoinCompleteRequest
import media.edge.PeerJoinCompleteResponse
import media.edge.PeerJoinRequest
import socket.room.BroadcastMessage

internal interface HiveNodeSocketHandler {
  fun subscribeToBroadcasts(
    onPeerJoinedBroadcast: SocketServiceEventListener,
    onPeerProducerCreateBroadcast: SocketServiceEventListener,
    onPeerProducerToggleBroadcast: SocketServiceEventListener,
    onSelectedPeer: SocketServiceEventListener,
    onSelectedPeerDiff: SocketServiceEventListener,
    onPeerLeaveBroadcast: SocketServiceEventListener,
    onPeerProducerCloseBroadcast: SocketServiceEventListener,
    onGlobalPeerPinBroadcast: SocketServiceEventListener,
    onBroadcastMessage: SocketServiceEventListener
  )

  suspend fun joinRoom(peerJoinRequest: PeerJoinRequest)

  suspend fun completeJoinRoom(): PeerJoinCompleteResponse?

  suspend fun broadcastMessage(type: String, payload: JsonObject)
}

internal class DefaultHiveNodeSocketHandler(
  private val socketService: ISockratesSocketService,
  private val json: Json,
  private val timeProvider: IDytePlatformUtils
) : HiveNodeSocketHandler {
  override fun subscribeToBroadcasts(
    onPeerJoinedBroadcast: SocketServiceEventListener,
    onPeerProducerCreateBroadcast: SocketServiceEventListener,
    onPeerProducerToggleBroadcast: SocketServiceEventListener,
    onSelectedPeer: SocketServiceEventListener,
    onSelectedPeerDiff: SocketServiceEventListener,
    onPeerLeaveBroadcast: SocketServiceEventListener,
    onPeerProducerCloseBroadcast: SocketServiceEventListener,
    onGlobalPeerPinBroadcast: SocketServiceEventListener,
    onBroadcastMessage: SocketServiceEventListener
  ) {
    socketService.subscribe(
      SocketServiceUtils.MediaEvent.PEER_JOINED_BROADCAST.id,
      onPeerJoinedBroadcast
    )

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.PEER_PRODUCER_CREATE_BROADCAST.id,
      onPeerProducerCreateBroadcast
    )

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.PEER_PRODUCER_TOGGLE_BROADCAST.id,
      onPeerProducerToggleBroadcast
    )

    socketService.subscribe(SocketServiceUtils.MediaEvent.SELECTED_PEER.id, onSelectedPeer)

    socketService.subscribe(SocketServiceUtils.MediaEvent.SELECTED_PEER_DIFF.id, onSelectedPeerDiff)

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.PEER_LEAVE_BROADCAST.id,
      onPeerLeaveBroadcast
    )

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.PEER_PRODUCER_CLOSE_BROADCAST.id,
      onPeerProducerCloseBroadcast
    )

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.GLOBAL_PEER_PIN_BROADCAST.id,
      onGlobalPeerPinBroadcast
    )

    socketService.subscribe(SocketServiceUtils.RoomEvent.BROADCAST_MESSAGE.id, onBroadcastMessage)
  }

  override suspend fun joinRoom(peerJoinRequest: PeerJoinRequest) {
    try {
      socketService.send(
        event = SocketServiceUtils.MediaEvent.JOIN_ROOM.id,
        payload = PeerJoinRequest.ADAPTER.encode(peerJoinRequest)
      )
    } catch (e: Exception) {
      DyteLogger.error("HiveSocketHandler: Error on join room: ${e.message}")
    }
  }

  override suspend fun completeJoinRoom(): PeerJoinCompleteResponse? {
    return try {
      val req = PeerJoinCompleteRequest()

      PeerJoinCompleteResponse.ADAPTER.decode(
        socketService.requestResponse(
          event = SocketServiceUtils.MediaEvent.SELF_JOIN_COMPLETE.id,
          payload = PeerJoinCompleteRequest.ADAPTER.encode(req)
        )!!
      )
    } catch (e: Exception) {
      DyteLogger.error("HiveSocketHandler: Error on complete join room: ${e.message}")
      null
    }
  }

  override suspend fun broadcastMessage(type: String, payload: JsonObject) {
    try {
      val message =
        BroadcastMessage(
          type = type,
          payload = json.encodeToByteString(payload),
          timestamp = timeProvider.getCurrentTime()
        )
      socketService.send(
        event = SocketServiceUtils.RoomEvent.BROADCAST_MESSAGE.id,
        payload = BroadcastMessage.ADAPTER.encode(message)
      )
    } catch (e: Exception) {
      DyteLogger.error("HiveSocketHandler: Error on broadcast message: ${e.message}")
    }
  }
}
