package io.dyte.core.controllers

import defaults.BuildKonfig
import io.dyte.callstats.CallStats
import io.dyte.callstats.events.DeviceInfo
import io.dyte.callstats.events.PeerMetaData
import io.dyte.callstats.models.Environment
import io.dyte.core.ControllerScopeProvider
import io.dyte.core.Result.Failure
import io.dyte.core.Result.Success
import io.dyte.core.chat.BaseChatController
import io.dyte.core.chat.DyteChatFactory
import io.dyte.core.chat.IChatController
import io.dyte.core.chat.channel.ChatChannelController
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomInitFailed
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomInitStarted
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomLeave
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomLeaveStarted
import io.dyte.core.controllers.polls.BasePollsController
import io.dyte.core.controllers.polls.IPollsController
import io.dyte.core.controllers.polls.PollsRoomNodeController
import io.dyte.core.controllers.polls.PollsSocketServiceController
import io.dyte.core.controllers.stage.BaseStageController
import io.dyte.core.controllers.stage.StageRoomNodeController
import io.dyte.core.controllers.stage.StageSocketServiceController
import io.dyte.core.events.EventEmitter
import io.dyte.core.events.InternalEvents
import io.dyte.core.feat.DyteChat
import io.dyte.core.feat.DyteLiveStream
import io.dyte.core.feat.DyteMeta
import io.dyte.core.feat.DytePlugins
import io.dyte.core.feat.DytePoll
import io.dyte.core.feat.DyteRecording
import io.dyte.core.feat.DyteStage
import io.dyte.core.featureflag.DyteFeatureFlagEntity
import io.dyte.core.featureflag.DyteFeatureFlags
import io.dyte.core.featureflag.FeatureFlagService
import io.dyte.core.featureflag.FeatureFlagService.UserAttributeKeys
import io.dyte.core.featureflag.FlagsmithService
import io.dyte.core.hive.DefaultHiveNodeSocketHandler
import io.dyte.core.hive.DefaultHiveSFUSocketHandler
import io.dyte.core.host.HostRoomNodeController
import io.dyte.core.host.HostSocketServiceController
import io.dyte.core.host.IHostController
import io.dyte.core.listeners.DyteChatChannelEventsListener
import io.dyte.core.listeners.DyteChatEventsListener
import io.dyte.core.listeners.DyteParticipantEventsListener
import io.dyte.core.listeners.DytePluginEventsListener
import io.dyte.core.listeners.DytePollEventsListener
import io.dyte.core.listeners.DyteRecordingEventsListener
import io.dyte.core.listeners.DyteSelfEventsListener
import io.dyte.core.listeners.DyteStageEventListener
import io.dyte.core.listeners.DyteWaitlistEventsListener
import io.dyte.core.media.DyteCallStatsObserver
import io.dyte.core.media.DyteHive
import io.dyte.core.media.DyteMediaSoup
import io.dyte.core.media.IDyteHive
import io.dyte.core.media.IDyteMediaSoupUtils
import io.dyte.core.media.IDyteSFUUtils
import io.dyte.core.models.*
import io.dyte.core.network.ApiClient
import io.dyte.core.network.BaseApiService
import io.dyte.core.network.IApiClient
import io.dyte.core.network.info.DyteMediaRoomType
import io.dyte.core.network.info.MeetingSessionInfo
import io.dyte.core.network.info.ParticipantInfo
import io.dyte.core.network.info.PluginPermissions
import io.dyte.core.observability.DyteLogger
import io.dyte.core.observability.DyteTracer
import io.dyte.core.observability.IDyteTracer
import io.dyte.core.platform.IDytePlatformUtilsProvider
import io.dyte.core.plugins.PluginChatSender
import io.dyte.core.plugins.PluginsController
import io.dyte.core.plugins.factory.DytePluginsFactory
import io.dyte.core.room.SocketServiceRoomSocketHandler
import io.dyte.core.socket.IRoomNodeSocketService
import io.dyte.core.socket.ISocketMessageResponseParser
import io.dyte.core.socket.RoomNodeSocketService
import io.dyte.core.socket.SocketMessageResponseParser
import io.dyte.core.socket.events.OutboundMeetingEventType.GET_ROOM_STATE
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketRoomStateModel
import io.dyte.core.socket.socketservice.SockratesSocketService
import io.dyte.core.waitingroom.BaseWaitlistController
import io.dyte.core.waitingroom.IWaitlistController
import io.dyte.core.waitingroom.hive.DefaultWaitlistHostSocketHandler
import io.dyte.core.waitingroom.hive.HiveWaitlistController
import io.dyte.core.waitingroom.hive.HiveWaitlistSubscription
import io.dyte.core.waitingroom.roomnode.RoomNodeWaitlistController
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.DelicateCoroutinesApi
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.newSingleThreadContext
import kotlinx.coroutines.withContext

internal class Controller(private val _platformUtilsProvider: IDytePlatformUtilsProvider) :
  IController, IControllerContainer, ControllerScopeProvider() {

  private lateinit var _tracer: DyteTracer
  override val tracer: IDyteTracer
    get() = _tracer

  override val platformUtilsProvider: IDytePlatformUtilsProvider
    get() = _platformUtilsProvider

  @Deprecated("please use DyteSelfParticipant.mediaRoomType")
  // TODO : delete this
  private var useHive: Boolean = false

  var chatEventListeners = EventEmitter<DyteChatEventsListener>()
  override lateinit var chatController: BaseChatController

  var participantEventListener = EventEmitter<DyteParticipantEventsListener>()
  private lateinit var _participantController: ParticipantController
  override val participantController: ParticipantController
    get() = this._participantController

  var pollsEventListener = EventEmitter<DytePollEventsListener>()
  override lateinit var pollsController: BasePollsController

  private lateinit var _metaController: MetaController
  override val metaController: MetaController
    get() = _metaController

  var selfEventListeners = EventEmitter<DyteSelfEventsListener>()
  private lateinit var _selfController: SelfController
  override val selfController: SelfController
    get() = this._selfController

  private lateinit var _roomNodeController: IRoomNodeController
  override val roomNodeController: IRoomNodeController
    get() = this._roomNodeController

  private lateinit var _hostController: IHostController
  override val hostController: IHostController
    get() = this._hostController

  private lateinit var _apiClient: ApiClient
  override val apiClient: IApiClient
    get() = this._apiClient

  private lateinit var _roomNodeSocketService: RoomNodeSocketService
  override val roomNodeSocketService: IRoomNodeSocketService
    get() = this._roomNodeSocketService

  private var _eventController = EventController(this)
  override val eventController: IEventController
    get() = this._eventController

  private var _connectionController = ConnectionController(this)
  override val connectionController: IConnectionController
    get() = this._connectionController

  private lateinit var _socketMessageResponseParser: SocketMessageResponseParser
  override val socketMessageResponseParser: ISocketMessageResponseParser
    get() = this._socketMessageResponseParser

  var recordingEventListeners = EventEmitter<DyteRecordingEventsListener>()
  override lateinit var recordingController: RecordingController

  private lateinit var _permissionController: PermissionController
  override val permissionController: IPermissionController
    get() = this._permissionController

  private lateinit var _mediaSoupController: MediaSoupController
  override val mediaSoupController: IMediaSoupController
    get() = this._mediaSoupController

  private lateinit var _liveStreamController: LiveStreamController
  override val liveStreamController: ILiveStreamController
    get() = this._liveStreamController

  var waitlistEventsListener = EventEmitter<DyteWaitlistEventsListener>()
  private lateinit var _waitlistController: BaseWaitlistController
  override val waitlistController: BaseWaitlistController
    get() = this._waitlistController

  private lateinit var _sfuUtils: IDyteSFUUtils
  override val sfuUtils: IDyteSFUUtils
    get() = _sfuUtils

  var pluginsEventsListener = EventEmitter<DytePluginEventsListener>()
  override lateinit var pluginsController: PluginsController

  var stageEventListeners = EventEmitter<DyteStageEventListener>()
  override lateinit var stageController: BaseStageController

  private lateinit var recording: DyteRecording
  private lateinit var poll: DytePoll
  private lateinit var meta: DyteMeta
  private lateinit var plugins: DytePlugins
  private lateinit var _stage: DyteStage
  private lateinit var liveStream: DyteLiveStream
  private lateinit var chat: DyteChat
  private lateinit var _participants: DyteParticipants

  override val participants: DyteParticipants
    get() = _participants

  override val stage: DyteStage
    get() = _stage

  private var _internalEventsEmitter = EventEmitter<InternalEvents>()
  override val internalEventEmitter: EventEmitter<InternalEvents>
    get() = _internalEventsEmitter

  override lateinit var sockratesSocketService: SockratesSocketService

  override val socketController = SocketController(this)

  override lateinit var featureFlagService: FeatureFlagService
    private set

  private lateinit var callStatsClient: CallStats

  private var initInProgress = false

  internal var chatChannelEventListeners: EventEmitter<DyteChatChannelEventsListener>? = null
  private var chatChannelController: ChatChannelController? = null

  override suspend fun init(dyteMeetingInfo: DyteMeetingInfo) {
    if (!initInProgress) {
      initInProgress = true
      initInternal(
        MeetingInfo(
            dyteMeetingInfo.authToken,
            MeetingConfig(dyteMeetingInfo.enableAudio, dyteMeetingInfo.enableVideo),
            dyteMeetingInfo.baseUrl
          )
          .apply { roomName = dyteMeetingInfo.roomName }
      )
    }
  }

  override suspend fun init(dyteMeetingInfoV2: DyteMeetingInfoV2) {
    if (!initInProgress) {
      initInProgress = true
      initInternal(
        MeetingInfo(
          dyteMeetingInfoV2.authToken,
          MeetingConfig(dyteMeetingInfoV2.enableAudio, dyteMeetingInfoV2.enableVideo),
          dyteMeetingInfoV2.baseUrl
        )
      )
    }
  }

  private suspend fun initInternal(meetingInfo: MeetingInfo) {
    try {
      _tracer = DyteTracer(this)

      val baseDomain = getBaseDomain(meetingInfo.baseUrl)

      // Initialising CallStats client
      callStatsClient = initialiseCallStatsClient(baseDomain)

      _platformUtilsProvider.init(this@Controller)

      _eventController.triggerEvent(OnMeetingRoomInitStarted)

      _metaController = MetaController(meetingInfo, this@Controller, serialScope)

      DyteLogger.init(this@Controller)

      DyteLogger.info("Controller.init")

      try {
        _metaController.init()
      } catch (e: Exception) {
        DyteLogger.error("Controller::meta_init::failed", e)
        eventController.triggerEvent(
          OnMeetingRoomInitFailed(IllegalArgumentException("Invalid auth token"))
        )
        return
      }
      _permissionController = getPermissionController(this@Controller)
      doInit(baseDomain)
    } catch (e: Exception) {
      DyteLogger.error("Controller::init::failed", e)
      eventController.triggerEvent(OnMeetingRoomInitFailed(e))
    }
  }

  private fun getPermissionController(parent: Controller): PermissionController {
    val permissionController = PermissionController(parent)
    permissionController.init()
    return permissionController
  }

  private suspend fun askDevicePermissions(meetingConfig: MeetingConfig) {
    val audioEnabled = meetingConfig.enableAudio
    val videoEnable = meetingConfig.enableVideo
    if (audioEnabled || videoEnable) {
      permissionController.askPermissions(
        audioEnabled,
        videoEnable,
        { DyteLogger.info("permissions granted") },
        { DyteLogger.info("------permissions denied") }
      )
    }
  }

  private suspend fun doInit(baseDomain: String) {
    tracer.startTrace("init")
    try {

      _connectionController = ConnectionController(this@Controller)
      _apiClient =
        ApiClient(
          baseDomain = baseDomain,
          authToken = _metaController.getAuthToken(),
          peerId = _metaController.getPeerId(),
          roomName = _metaController.getRoomName(),
          meetingId = _metaController.getMeetingId(),
          orgId = _metaController.getOrgId(),
          isV2AuthToken = _metaController.isV2AuthToken()
        )

      val participantInfo: ParticipantInfo
      try {
        participantInfo = apiClient.getParticipantInfo()
      } catch (e: Exception) {
        eventController.triggerEvent(OnMeetingRoomInitFailed(e))
        DyteLogger.error("DyteLog: Failed  to get ParticipantInfo ${e.message}", e)
        return
      }

      DyteLogger.info("Controller::init::preset::${participantInfo.presetInfo}")

      platformUtilsProvider.getPlatformUtils().listenForCrashes()

      /*
       * Initializing the Feature Flags.
       * IMP: The shared API client is sending our Authorization header because of the default headers set.
       * We will need to pass a new instance of HttpClient but an ideal way would be to use the
       * shared instance only. In most of the Dyte API calls > 90%, we have manually added the
       * Authorization header, so the default header block inside the current HttpClient initialisation
       * can be removed.
       * */
      featureFlagService =
        FlagsmithService(environmentKey = FEATURE_FLAG_KEY, httpClient = _apiClient.getClient())

      featureFlagService.initializeWithUserContext(
        identifier =
          getFeatureFlagIdentifier(
            featureFlagEntity = DyteFeatureFlagEntity.PEER,
            peerId = metaController.getPeerId()
          ),
        userAttributes =
          getFeatureFlagUserAttributes(
            featureFlagEntity = DyteFeatureFlagEntity.PEER,
            participantInfo = participantInfo,
            metaController = metaController,
            platformUtilsProvider = platformUtilsProvider
          )
      )

      DyteLogger.setPostInitInfo(participantInfo.organizationId, metaController.getRoomName())
      _metaController.setParticipantInfo(participantInfo)

      val meetingSessionData: MeetingSessionInfo
      try {
        meetingSessionData = apiClient.getRoomNodeData()
      } catch (e: Exception) {
        eventController.triggerEvent(OnMeetingRoomInitFailed(e))
        DyteLogger.error("Controller: init: Meeting Room Init Failed ${e.message}", e)
        return
      }
      participantInfo.presetInfo.permissions.mediaRoomType =
        if (meetingSessionData.useHive) {
          DyteMediaRoomType.HIVE
        } else {
          DyteMediaRoomType.MEDIA_SOUP
        }
      useHive = meetingSessionData.useHive

      DyteLogger.error("Controller: init: isHiveMeeting $useHive")

      metaController.setIsHive(useHive)
      metaController.setMeetingTitle(meetingSessionData.title)

      /*
       * Initializing both SocketIO and SocketService sockets.
       * Needed as dependency for initialisation of Chat and Plugin features.
       * */
      _roomNodeSocketService = RoomNodeSocketService(this@Controller)
      _socketMessageResponseParser = SocketMessageResponseParser()
      sockratesSocketService =
        SockratesSocketService(
          baseDomain = baseDomain,
          peerId = _metaController.getPeerId(),
          roomName = _metaController.getRoomName(),
          authToken = _metaController.getAuthToken(),
          useHive = useHive,
        )

      socketController.init(useHive)

      val meetingType = _metaController.getMeetingType()
      when (meetingType) {
        DyteMeetingType.CHAT -> {
          initChatMeeting(useHive, featureFlagService, participantInfo)
        }
        else -> {
          initVideoAudioMeeting(participantInfo)
        }
      }

      if (!useHive) {
        _roomNodeSocketService.init(meetingSessionData)
        _roomNodeSocketService.connect()
      }

      sockratesSocketService.connect()
      eventController.triggerEvent(DyteEventType.OnMetaUpdate)
      selfController.emitEvent { it.onUpdate(selfController.getSelf()) }
      eventController.triggerEvent(DyteEventType.OnMeetingRoomInitCompleted)
    } catch (e: Exception) {
      e.printStackTrace()
      eventController.triggerEvent(OnMeetingRoomInitFailed(e))
      DyteLogger.error("Controller::init::error", e)
    }
    initInProgress = false
    tracer.endTrace("init")
  }

  private suspend fun initVideoAudioMeeting(participantInfo: ParticipantInfo) {
    askDevicePermissions(metaController.getMeetingConfig())

    val roomSocketHandler = SocketServiceRoomSocketHandler(sockratesSocketService)

    // Initialise the Chat feature: DyteChat
    chat = initChatModule(useHive, featureFlagService, participantInfo)

    val pollSocketServerType =
      if (useHive) "socket-service"
      else featureFlagService.getConfigValue(DyteFeatureFlags.POLL_SOCKET_SERVER)
    val pollsPermissions = participantInfo.presetInfo.permissions.polls
    pollsController =
      if (pollSocketServerType == "socket-service") {
        PollsSocketServiceController(
          permissions = pollsPermissions,
          socket = sockratesSocketService,
        )
      } else {
        PollsRoomNodeController(
          permissions = pollsPermissions,
          socket = roomNodeSocketService,
        )
      }

    pollsController.copy(pollsEventListener)
    pollsEventListener = pollsController

    val tempHostController =
      createHostController(
        useHive = useHive,
        socketService = sockratesSocketService,
        socketController = roomNodeSocketService,
      )
    _hostController = tempHostController

    _mediaSoupController = MediaSoupController(this@Controller)
    _roomNodeController =
      if (useHive) {
        val hiveNodeSocketHandler =
          DefaultHiveNodeSocketHandler(
            socketService = sockratesSocketService,
            json = BaseApiService.json,
            timeProvider = _platformUtilsProvider.getPlatformUtils()
          )
        val hiveController =
          HiveController(
            this@Controller,
            hiveNodeSocketHandler,
            BaseApiService.json,
            sockratesSocketService,
            roomSocketHandler
          )
        internalEventEmitter.addListener(hiveController)
        hiveController
      } else {
        val roomNode =
          RoomNodeController(
            this@Controller,
          )
        internalEventEmitter.addListener(roomNode)
        roomNode
      }

    _participantController =
      ParticipantController(
        _mediaSoupController,
        useHive,
        participantInfo,
        platformUtilsProvider.getVideoUtils(),
        _roomNodeSocketService,
        hostController,
        roomNodeController,
        serialScope,
        _eventController,
        _metaController.getMeetingType(),
        null,
        roomSocketHandler
      )
    internalEventEmitter.addListener(_participantController)

    _participantController.copy(participantEventListener)
    participantEventListener = _participantController

    _selfController =
      SelfController(
        _permissionController,
        internalEventEmitter,
        metaController.getPeerId(),
        useHive,
        _mediaSoupController,
        _roomNodeSocketService,
        _platformUtilsProvider.getMediaUtils(),
        participantInfo,
        _participantController,
        metaController,
        hostController,
        sockratesSocketService,
        serialScope,
        platformUtilsProvider.getVideoUtils()
      )
    internalEventEmitter.addListener(_selfController)

    _selfController.copy(selfEventListeners)
    selfEventListeners = _selfController

    _participantController.selfParticipant = _selfController.selfParticipant

    _waitlistController = createWaitlistController(useHive)

    _waitlistController.copy(waitlistEventsListener)
    waitlistEventsListener = _waitlistController

    participantController.waitlistController = _waitlistController
    internalEventEmitter.addListener(_waitlistController)

    // _pluginsController = PluginsController(this@Controller)
    _liveStreamController = LiveStreamController(this@Controller, sockratesSocketService)

    _participants = DyteParticipants(_participantController, _hostController, _waitlistController)
    if (useHive && roomNodeController is HiveController) {
      (roomNodeController as HiveController).data = _participants
    }

    recording =
      initialiseRecording(useHive, _apiClient, participantInfo.presetInfo.permissions.canRecord())
    recordingController = recording.recordingController
    recordingController.copy(recordingEventListeners)
    recordingEventListeners = recordingController

    poll = DytePoll(pollsController)
    meta = DyteMeta(metaController = _metaController)

    // Initialise the Plugins feature: DytePlugins
    plugins =
      initPluginsFeature(
        useHive,
        featureFlagService,
        internalEventEmitter,
        participantInfo.presetInfo.permissions.plugins,
        chatController,
        _eventController
      )
    /*
     * This is just to follow the existing Controller pattern. It might not be required after refactor
     * of Controller class.
     * */
    pluginsController = plugins.pluginsController

    pluginsController.copy(pluginsEventsListener)
    pluginsEventsListener = pluginsController

    // Initialise the Stage feature: DyteStage
    _stage = initStageFeature(useHive, participantInfo)

    liveStream = DyteLiveStream(this@Controller)

    _sfuUtils =
      if (useHive) {
        val hiveSFUSocketHandler = DefaultHiveSFUSocketHandler(sockratesSocketService)
        DyteHive(_participants, this@Controller, hiveSFUSocketHandler, callStatsClient)
      } else
        DyteMediaSoup(callStatsClient, _participants, _roomNodeSocketService, internalEventEmitter)

    _selfController.sfuUtils = _sfuUtils

    _selfController.getSelf()._videoEnabled =
      metaController.getMeetingConfig().enableVideo && _selfController.canPublishVideo()
    _selfController.getSelf()._audioEnabled =
      metaController.getMeetingConfig().enableAudio && _selfController.canPublishAudio()

    withContext(Dispatchers.Main) {
      if (useHive) (_sfuUtils as IDyteHive).init()
      else (_sfuUtils as IDyteMediaSoupUtils).init(this@Controller)
    }

    _mediaSoupController.init()
    _permissionController.init()
    _connectionController.init()
    pollsController.init()
    recordingController.init()
    _liveStreamController.init()
    _roomNodeController.init()

    withContext(Dispatchers.Main) { platformUtilsProvider.getMediaUtils().init() }
  }

  private suspend fun initChatMeeting(
    useHive: Boolean,
    featureFlagService: FeatureFlagService,
    participantInfo: ParticipantInfo
  ) {
    chatChannelEventListeners = EventEmitter()

    val roomSocketHandler = SocketServiceRoomSocketHandler(sockratesSocketService)

    meta = DyteMeta(metaController = _metaController)

    // Initialise the Chat feature: DyteChat
    chat = initChatModule(useHive, featureFlagService, participantInfo)

    val tempHostController =
      createHostController(
        useHive = useHive,
        socketService = sockratesSocketService,
        socketController = roomNodeSocketService,
      )
    _hostController = tempHostController

    _participantController =
      ParticipantController(
        mediaSoupController = null,
        useHive,
        participantInfo,
        videoUtils = null,
        _roomNodeSocketService,
        hostController,
        roomNodeController = null,
        serialScope,
        _eventController,
        _metaController.getMeetingType(),
        sockratesSocketService,
        roomSocketHandler
      )
    internalEventEmitter.addListener(_participantController)

    _participantController.copy(participantEventListener)
    participantEventListener = _participantController

    _selfController =
      SelfController(
        _permissionController,
        internalEventEmitter,
        metaController.getPeerId(),
        useHive,
        mediaSoupController = null,
        _roomNodeSocketService,
        mediaUtils = null,
        participantInfo,
        _participantController,
        metaController,
        hostController,
        sockratesSocketService,
        serialScope,
        videoUtils = null
      )
    internalEventEmitter.addListener(_selfController)

    _selfController.copy(selfEventListeners)
    selfEventListeners = _selfController

    _participantController.selfParticipant = _selfController.selfParticipant

    /*
     * TODO: remove instantiation of WaitlistController for CHAT meetingType
     * Currently needed in DyteParticipants public class.
     * */
    _waitlistController = createWaitlistController(useHive)

    _waitlistController.copy(waitlistEventsListener)
    waitlistEventsListener = _waitlistController

    participantController.waitlistController = _waitlistController
    internalEventEmitter.addListener(_waitlistController)

    _participants = DyteParticipants(_participantController, _hostController, _waitlistController)
  }

  private fun initialiseRecording(
    useHive: Boolean,
    apiClient: ApiClient,
    canRecord: Boolean,
  ): DyteRecording {
    @OptIn(DelicateCoroutinesApi::class, ExperimentalCoroutinesApi::class)
    val serialScope = CoroutineScope(newSingleThreadContext("DyteRecording${hashCode()}"))

    if (useHive) {
      val recordingController =
        RecordingSocketServiceController(
          apiClient,
          canRecord = canRecord,
          sockratesSocketService,
          serialScope
        )
      return DyteRecording(recordingController = recordingController, serialScope)
    } else {
      val recordingController =
        RecordingRoomNodeController(
          apiClient,
          canRecord = canRecord,
          _roomNodeSocketService,
          scope = serialScope
        )
      internalEventEmitter.addListener(recordingController)
      return DyteRecording(recordingController = recordingController, serialScope)
    }
  }

  private fun createHostController(
    useHive: Boolean,
    socketService: SockratesSocketService,
    socketController: IRoomNodeSocketService,
  ): IHostController {
    return if (useHive) {
      HostSocketServiceController(socketService)
    } else {
      HostRoomNodeController(socketController)
    }
  }

  override suspend fun joinRoom() {
    tracer.startTrace("joinRoom")
    DyteLogger.info("Controller.joinRoom")

    if (metaController.getMeetingType() == DyteMeetingType.CHAT) {
      joinChatRoom()
    } else {
      /* Send JOIN_ROOM to socket service from Controller only in case of Mediasoup meeting/
      HiveController handles this itself. */
      if (!useHive) {
        socketController.joinRoomSockratesSocketService()
        this.eventController.triggerEvent(DyteEventType.OnMeetingRoomJoinStarted)
      }
      platformUtilsProvider.getMediaUtils().routeAudioFromSpeakerphone()
      roomNodeController.joinRoom()

      // Send call join begin event to callstats after room is joined
      sendCallJoinBeginCallStatEvent()
    }

    tracer.endTrace("joinRoom")
  }

  override suspend fun joinRoom(onRoomJoined: () -> Unit, onRoomJoinFailed: () -> Unit) {
    tracer.startTrace("joinRoom")
    DyteLogger.info("Controller.joinRoom")

    if (metaController.getMeetingType() == DyteMeetingType.CHAT) {
      joinChatRoom(onRoomJoined, onRoomJoinFailed)
    } else {
      /* Send JOIN_ROOM to socket service from Controller only in case of Mediasoup meeting/
      HiveController handles this itself. */
      if (!useHive) {
        socketController.joinRoomSockratesSocketService()
        this.eventController.triggerEvent(DyteEventType.OnMeetingRoomJoinStarted)
      }
      platformUtilsProvider.getMediaUtils().routeAudioFromSpeakerphone()
      roomNodeController.joinRoom(onRoomJoined, onRoomJoinFailed)

      // Send call join begin event to callstats after room is joined
      sendCallJoinBeginCallStatEvent()
    }

    tracer.endTrace("joinRoom")
  }

  /* TODO: Enable onRoomJoinFailed callback.
   * This would need joinRoomSockratesSocketService to return result. */
  private suspend fun joinChatRoom(
    onRoomJoined: (() -> Unit)? = null,
    onRoomJoinFailed: (() -> Unit)? = null
  ) {
    this.eventController.triggerEvent(DyteEventType.OnMeetingRoomJoinStarted)
    socketController.joinRoomSockratesSocketService()
    participantController.handleSocketServiceRoomJoined()
    chatChannelController?.loadAllChatChannels()
      ?: kotlin.run {
        eventController.triggerEvent(
          DyteEventType.OnMeetingRoomJoinFailed(Exception("Chat meeting not initialised"))
        )
        onRoomJoinFailed?.invoke()
        return
      }
    selfController._roomJoined = true
    eventController.triggerEvent(DyteEventType.OnMeetingRoomJoined(selfController.getSelf()))
    eventController.triggerEvent(DyteEventType.OnMetaUpdate)
    onRoomJoined?.invoke()
  }

  override suspend fun leaveRoom() {
    tracer.startTrace("leaveRoom")
    DyteLogger.info("Controller.leaveRoom")
    if (metaController.getMeetingType() == DyteMeetingType.CHAT) {
      leaveChatRoom()
    } else {
      eventController.triggerEvent(OnMeetingRoomLeaveStarted)
      platformUtilsProvider.getVideoUtils().destroyAll()
      _sfuUtils.leaveCall()
      plugins.clear()
      if (!useHive) roomNodeSocketService.disconnect()
      sockratesSocketService.disconnect()
      roomNodeController.leaveRoom()
      sockratesSocketService.clear() // Remove all socket event listeners as part of leave call
      eventController.triggerEvent(OnMeetingRoomLeave)
      // needs to be done after triggering on meeting room leave
      disposeListeners()
    }
    tracer.endTrace("leaveRoom")
  }

  private suspend fun leaveChatRoom() {
    eventController.triggerEvent(OnMeetingRoomLeaveStarted)
    if (!useHive) roomNodeSocketService.disconnect()
    sockratesSocketService.disconnect()
    sockratesSocketService.clear() // Remove all socket event listeners as part of leave call
    selfController._roomJoined = false

    eventController.triggerEvent(OnMeetingRoomLeave)

    internalEventEmitter.dispose()
    chatController.dispose()
    selfController.dispose()
    eventController.dispose()
    chatChannelEventListeners?.dispose()
  }

  private fun disposeListeners() {
    internalEventEmitter.dispose()
    chatController.dispose()
    pollsController.dispose()
    stageController.dispose()
    participantController.dispose()
    pluginsController.dispose()
    pollsController.dispose()
    recordingController.dispose()
    _waitlistController.dispose()
    selfController.dispose()
  }

  override suspend fun release() {
    try {
      if (metaController.getMeetingType() == DyteMeetingType.CHAT) {
        releaseForChatRoom()
      } else {
        platformUtilsProvider.getVideoUtils().destroyAll()
        _sfuUtils.leaveCall()
        plugins.clear()
        roomNodeSocketService.disconnect()
        sockratesSocketService.disconnect()
        eventController.dispose()
        disposeListeners()
      }
    } catch (e: Exception) {
      DyteLogger.error("Controller::release::failed", e)
    }
  }

  private suspend fun releaseForChatRoom() {
    roomNodeSocketService.disconnect()
    sockratesSocketService.disconnect()
    eventController.dispose()
    internalEventEmitter.dispose()
    chatController.dispose()
    selfController.dispose()
    eventController.dispose()
    chatChannelEventListeners?.dispose()
  }

  override fun getSelf(): DyteSelfParticipant {
    return this.selfController.getSelf()
  }

  override fun getChat(): DyteChat {
    return chat
  }

  override fun getRecording(): DyteRecording {
    return recording
  }

  override fun getPolls(): DytePoll {
    return poll
  }

  override fun getMeta(): DyteMeta {
    return meta
  }

  override fun getPlugins(): DytePlugins {
    return plugins
  }

  override fun getLiveStream(): DyteLiveStream {
    return liveStream
  }

  /** Fetches roomUuid from Room node. */
  suspend fun getRoomUuid(): String {
    when (val roomStateResponse = _roomNodeSocketService.sendPacket(GET_ROOM_STATE, null)) {
      is Success -> {
        val roomState =
          socketMessageResponseParser.parseResponse(roomStateResponse.value).payload
            as WebSocketRoomStateModel

        return roomState.roomState.roomUUID
      }
      is Failure -> {
        throw Exception("Invalid room state")
      }
    }
  }

  private fun getFeatureFlagIdentifier(featureFlagEntity: String, peerId: String): String {
    return "${featureFlagEntity}_${peerId}"
  }

  private fun getFeatureFlagUserAttributes(
    featureFlagEntity: String,
    participantInfo: ParticipantInfo,
    metaController: IMetaController,
    platformUtilsProvider: IDytePlatformUtilsProvider
  ): Map<String, Any> {
    return mapOf(
      UserAttributeKeys.ENTITY to featureFlagEntity,
      UserAttributeKeys.CLIENT_ID to participantInfo.organizationId,
      UserAttributeKeys.ROOM_NAME to metaController.getRoomName(),
      UserAttributeKeys.ANONYMOUS_USER to participantInfo.organizationId.isBlank(),
      UserAttributeKeys.SDK_VERSION to BuildKonfig.sdkVersion,
      UserAttributeKeys.IS_MOBILE to true,
      UserAttributeKeys.OS_NAME to platformUtilsProvider.getPlatformUtils().getOsName(),
      UserAttributeKeys.OS_VERSION_NAME to platformUtilsProvider.getPlatformUtils().getOsVersion()
    )
  }

  private fun initialiseCallStatsClient(baseDomain: String): CallStats {
    val environment =
      if (baseDomain == "dyte.io") {
        Environment.PRODUCTION
      } else {
        Environment.STAGING
      }

    val debugMode = environment != Environment.PRODUCTION

    /*
     * note(swapnil): Since DyteMediaSoup creates its own scope with only a single thread, in order to
     * run CallStats operations in DyteMediaSoup's scope safely, we will need to pass a multi-thread Context.
     *
     * For the initial phase, we are passing a separate scope to the CallStatsClient.
     * */
    @OptIn(DelicateCoroutinesApi::class, ExperimentalCoroutinesApi::class)
    val callStatsScope = CoroutineScope(newSingleThreadContext("CallStatsScopeScope"))

    return CallStats(
      debug = debugMode,
      callStatsObserver = DyteCallStatsObserver(),
      env = environment,
      coroutineScope = callStatsScope
    )
  }

  // TODO: Move this out of Controller?
  private fun sendCallJoinBeginCallStatEvent() {
    // cpus and memory field is populated by Callstats lib internally
    val deviceInfo =
      DeviceInfo(
        isMobile = true,
        osName = platformUtilsProvider.getPlatformUtils().getOsName(),
        osVersionName = platformUtilsProvider.getPlatformUtils().getOsVersion(),
        cpus = 0,
        memory = null
      )

    val roomViewType =
      when (metaController.getMeetingType()) {
        DyteMeetingType.GROUP_CALL -> {
          "groupCall"
        }
        else -> {
          metaController.getMeetingType().name
        }
      }

    val peerMetaData =
      PeerMetaData(
        metaData = emptyMap(),
        deviceInfo = deviceInfo,
        displayName = metaController.getDisplayName(),
        meetingEnv = null, // populated by Callstats lib internally
        peerId = selfController.getSelf().id,
        userId = selfController.getSelf().userId,
        clientSpecificId = selfController.getSelf().clientSpecificId,
        roomName = metaController.getRoomName(),
        roomUUID = metaController.getRoomUuid(),
        permissions = emptyMap(),
        participantRole = null,
        roomViewType = roomViewType,
        sdkName = "mobile-core",
        sdkVersion = BuildKonfig.sdkVersion
      )
    callStatsClient.sendCallJoinBeginEvent(peerMetaData)
  }

  /**
   * Initialises the Plugins feature in the SDK.
   *
   * @return [DytePlugins] instance
   */
  private suspend fun initPluginsFeature(
    useHive: Boolean,
    featureFlagService: FeatureFlagService,
    internalEventsEmitter: EventEmitter<InternalEvents>,
    selfPluginsPermissions: PluginPermissions,
    pluginChatSender: PluginChatSender,
    flutterNotifier: EventController
  ): DytePlugins {
    val pluginSocketServer =
      if (useHive) "socket-service"
      else featureFlagService.getConfigValue(DyteFeatureFlags.PLUGIN_SOCKET_SERVER)
    val v1PluginsConfig = featureFlagService.getConfigValue(DyteFeatureFlags.V1_PLUGINS)
    return DytePluginsFactory(
        metaController.getMeetingTitle(),
        metaController.getRoomName(),
        metaController.getOrgId(),
        selfPluginsPermissions,
        roomNodeSocketService,
        sockratesSocketService,
        apiClient,
        apiClient.getApiBaseUrl(),
        platformUtilsProvider,
        internalEventsEmitter,
        socketMessageResponseParser,
        pluginChatSender,
        participants,
        selfController.getSelf(),
        flutterNotifier
      )
      .create(pluginSocketServer, v1PluginsConfig)
  }

  /**
   * Initialises the Chat feature in the SDK.
   *
   * @return [DyteChat] instance
   */
  private fun initChatModule(
    useHive: Boolean,
    featureFlagService: FeatureFlagService,
    participantInfo: ParticipantInfo
  ): DyteChat {
    val chatSocketServer =
      if (useHive) "socket-service"
      else featureFlagService.getConfigValue(DyteFeatureFlags.CHAT_SOCKET_SERVER)

    val dyteChat =
      DyteChatFactory(
          selfPeerId = metaController.getPeerId(),
          selfUserId = participantInfo.id,
          metaController.getDisplayName(),
          participantInfo.presetInfo.permissions.chat,
          participantInfo.presetInfo.permissions.privateChat,
          participantInfo.presetInfo.permissions.chatChannel,
          participantInfo.presetInfo.permissions.chatMessage,
          _roomNodeSocketService,
          sockratesSocketService,
          apiClient.getApiBaseUrl(),
          platformUtilsProvider,
          socketMessageResponseParser,
          metaController.getMeetingType(),
          metaController.getMeetingId(),
          chatChannelEventListeners
        )
        .create(chatSocketServer)

    chatController = dyteChat.chatController
    chatChannelController = dyteChat.chatChannelController

    chatController.copy(chatEventListeners)
    chatEventListeners = chatController

    return dyteChat
  }

  @OptIn(ExperimentalCoroutinesApi::class, DelicateCoroutinesApi::class)
  private fun initStageFeature(useHive: Boolean, participantInfo: ParticipantInfo): DyteStage {
    val stageSocketServerType =
      if (useHive) "socket-service"
      else if (participantInfo.presetInfo.viewType == "LIVESTREAM") "socket-service"
      else "room-node"

    val serialScope = CoroutineScope(newSingleThreadContext("DyteStage"))

    stageController =
      if (stageSocketServerType == "socket-service") {
        StageSocketServiceController(
          selfController.getSelf(),
          sockratesSocketService,
          serialScope,
          participantController,
          _selfController,
          platformUtilsProvider.getMediaUtils(),
          internalEventEmitter,
          metaController.getMeetingType()
        )
      } else {
        StageRoomNodeController(
          selfController.getSelf(),
          _participantController,
          hostController,
          _selfController,
          participantInfo.presetInfo.permissions,
          roomNodeSocketService,
          serialScope
        )
      }
    stageController.copy(stageEventListeners)
    stageEventListeners = stageController
    stageController.init()

    internalEventEmitter.addListener(stageController)

    return DyteStage(stageController, serialScope)
  }

  private fun createWaitlistController(useHive: Boolean): BaseWaitlistController {
    return if (useHive) {
      HiveWaitlistController(
        waitlistHostSocketHandler = DefaultWaitlistHostSocketHandler(sockratesSocketService),
        waitlistSubscription = HiveWaitlistSubscription(sockratesSocketService),
        internalEventEmitter,
        _selfController.selfParticipant,
        serialScope,
        participantController,
        hostController,
      )
    } else {
      RoomNodeWaitlistController(
        _roomNodeSocketService,
        internalEventEmitter,
        _selfController.selfParticipant,
        serialScope,
        participantController,
        hostController
      )
    }
  }

  companion object {
    private const val FEATURE_FLAG_KEY = "hXgU8Wc8pwuGNq9ms5q9Hh"

    /**
     * Utility method to get base domain in a backward compatible way.
     *
     * If [meetingInfoBaseUrl] is an old Dyte URL (e.g. https://api.cluster.dyte.in/v2), this method
     * returns the new dyte domain 'dyte.io'.
     *
     * If [meetingInfoBaseUrl] is a devel('devel.dyte.io') or preprod('preprod.dyte.io') domain,
     * this method returns the domain as-it-is.
     *
     * If [meetingInfoBaseUrl] is a non-Dyte domain like 'acme.org', this method returns the domain
     * as-it-is. It's meant for white-label URLs.
     *
     * @param meetingInfoBaseUrl base URL from [MeetingInfo]
     * @return base domain
     */
    private fun getBaseDomain(meetingInfoBaseUrl: String): String {
      if (meetingInfoBaseUrl.endsWith("v2") || meetingInfoBaseUrl.endsWith("v1")) {
        return "dyte.io"
      }
      return meetingInfoBaseUrl
    }
  }
}

internal interface IController {
  suspend fun init(dyteMeetingInfoV2: DyteMeetingInfoV2)

  suspend fun init(dyteMeetingInfo: DyteMeetingInfo)

  suspend fun joinRoom()

  suspend fun joinRoom(onRoomJoined: () -> Unit, onRoomJoinFailed: () -> Unit)

  suspend fun leaveRoom()

  suspend fun release()

  fun getSelf(): DyteSelfParticipant

  fun getRecording(): DyteRecording

  fun getChat(): DyteChat

  fun getPolls(): DytePoll

  fun getMeta(): DyteMeta

  fun getPlugins(): DytePlugins

  fun getLiveStream(): DyteLiveStream
}

internal interface IControllerContainer {
  val participants: DyteParticipants
  val stage: DyteStage
  val internalEventEmitter: EventEmitter<InternalEvents>
  val chatController: IChatController
  val participantController: ParticipantController
  val pollsController: IPollsController
  val metaController: IMetaController
  val selfController: SelfController
  val roomNodeController: IRoomNodeController
  val hostController: IHostController
  val apiClient: IApiClient
  val socketController: SocketController
  val sockratesSocketService: SockratesSocketService
  val roomNodeSocketService: IRoomNodeSocketService
  val socketMessageResponseParser: ISocketMessageResponseParser
  val platformUtilsProvider: IDytePlatformUtilsProvider
  val recordingController: IRecordingController
  val eventController: IEventController
  val connectionController: IConnectionController
  val permissionController: IPermissionController
  val mediaSoupController: IMediaSoupController
  val liveStreamController: ILiveStreamController
  val waitlistController: IWaitlistController
  val sfuUtils: IDyteSFUUtils
  val tracer: IDyteTracer
  val pluginsController: PluginsController
  var stageController: BaseStageController
  val featureFlagService: FeatureFlagService
}
