package io.dyte.core.controllers

import io.dyte.core.events.EventEmitter
import io.dyte.core.events.InternalEvents
import io.dyte.core.listeners.DyteConnectionEventListener
import io.dyte.core.observability.DyteLogger
import io.dyte.core.platform.IDytePlatformUtilsProvider
import io.dyte.core.socket.IRoomNodeSocketService
import io.dyte.core.socket.SocketConnectionEvent
import io.dyte.core.socket.SocketConnectionEventListener
import io.dyte.core.socket.socketservice.ISockratesSocketService
import io.dyte.core.socket.socketservice.SocketServiceConnectionStateListener
import io.dyte.core.socket.socketservice.SocketServiceUtils
import io.dyte.sockrates.client.WebSocketConnectionState
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.DelicateCoroutinesApi
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.launch
import kotlinx.coroutines.newSingleThreadContext
import socket.room.JoinRoomRequest
import socket.room.Peer

internal class SocketController(private val controllerContainer: Controller) :
  EventEmitter<DyteConnectionEventListener>() {

  private lateinit var oldSocket: IRoomNodeSocketService
  private lateinit var newSocket: ISockratesSocketService

  private lateinit var internalEmitter: EventEmitter<InternalEvents>
  private var reconnectingToMeetingRoom = false
  private var useHive = false

  private var prevConnected = false
  // 0 , No AutoRetrying, Means client needs to start the rejoin process again
  // <0, Unlimited Retrying
  // >0, Time based Retrying.
  // NOTE: For now we have only supportd <0 (Unlimited Retrying)
  private var retryingTimeIntervalInMillis: Long = -1
  private var startRetrying: Boolean = false

  private lateinit var platformConnectionNotif: PlatformConnectionChangeListener

  @OptIn(DelicateCoroutinesApi::class, ExperimentalCoroutinesApi::class)
  private val serialScope = CoroutineScope(newSingleThreadContext("DyteScope${hashCode()}"))

  fun init(useHive: Boolean) {
    oldSocket = controllerContainer.roomNodeSocketService
    newSocket = controllerContainer.sockratesSocketService
    internalEmitter = controllerContainer.internalEventEmitter
    this.useHive = useHive
    prevConnected = false
    /**
     * The below listeners are for external event listeners, if it is a hive meeting room node
     * socket service will not be used and sockrates socket service is taken as source of truth in
     * room node meetings, both socket systems are active but room node socket service is used as a
     * source of truth
     */
    if (useHive) {
      newSocket.addConnectionStateListener(
        object : SocketServiceConnectionStateListener {
          override fun onConnectionStateChanged(newState: WebSocketConnectionState) {
            serialScope.launch { onNewSocketConnectionEvent(newState) }
          }
        }
      )
    } else {
      oldSocket.addConnectionListener(
        object : SocketConnectionEventListener {
          override fun onConnectionEvent(event: SocketConnectionEvent) {
            serialScope.launch { onOldSocketConnectionEvent(event) }
          }
        }
      )
    }

    /**
     * The below listeners are for controlling media flow the listeners need to be setup on both, if
     * one of the the socket disconnects we need to disconnect the other one to ensure consistent
     * state on the server
     */
    newSocket.addConnectionStateListener(
      object : SocketServiceConnectionStateListener {
        override fun onConnectionStateChanged(newState: WebSocketConnectionState) {
          serialScope.launch { onNewSocketConnectionMediaEvent(newState) }
        }
      }
    )

    oldSocket.addConnectionListener(
      object : SocketConnectionEventListener {
        override fun onConnectionEvent(event: SocketConnectionEvent) {
          serialScope.launch { onOldSocketConnectionMediaEvent(event) }
        }
      }
    )
    platformConnectionNotif =
      PlatformConnectionChangeListener(controllerContainer.platformUtilsProvider) {
        serialScope.launch { reconnectSocketsWithNewPeerId() }
      }
  }

  /*
  note(swapnil): This method or step can be moved to some other initializer class once
  the roomUUID retrieval is made cleaner.
   */
  suspend fun joinRoomSockratesSocketService() {
    var peerId = controllerContainer.metaController.getPeerId()
    var displayName = controllerContainer.selfController.getSelf().name
    var userId = controllerContainer.selfController.getSelf().userId

    try {
      val roomUuid = if (!useHive) controllerContainer.getRoomUuid() else ""
      /*
       * Note(swapnil): We are storing the roomUuid in the MetaController for using it inside
       * ChatController and PluginsController. Ideal way would be to fetch the roomUuid before
       * initialisation of the feature modules/controllers.
       *
       * WARNING: This is a temporary hack to make the roomUuid available to feature modules.
       * Doing this after connecting to sockets in doInit() was breaking on iOS.
       * TODO: Improve the initialization order.
       * */
      controllerContainer.metaController.setRoomUuid(roomUuid)
      val joinRoomRequest =
        JoinRoomRequest(
          peer = Peer(peer_id = peerId, display_name = displayName, user_id = userId),
          room_uuid = controllerContainer.metaController.getRoomUuid()
        )

      newSocket.requestResponse(
        event = SocketServiceUtils.RoomEvent.JOIN_ROOM.id,
        payload = JoinRoomRequest.ADAPTER.encode(joinRoomRequest)
      )
    } catch (e: Exception) {
      DyteLogger.error("Controller::sendJoinRoomSocketService::failed", e)
    }
  }

  private fun isAutoRetryEnabled(): Boolean {
    if (retryingTimeIntervalInMillis == 0L) {
      return false
    }
    return true
  }

  private fun onConnected() {
    reconnectingToMeetingRoom = false
    startRetrying = false
    if (!prevConnected) {
      emitEvent { it.onConnectedToMeetingRoom() }
    } else {
      emitEvent { it.onReconnectedToMeetingRoom() }
    }
    prevConnected = true
  }

  private fun onConnecting() {
    if (!prevConnected) {
      emitEvent { it.onConnectingToMeetingRoom() }
    } else {
      emitReconnectingIfNotEmittedBefore()
    }
  }

  private fun onConnectFailed() {
    if (!prevConnected) {
      emitEvent { it.onMeetingRoomConnectionFailed() }
    } else {
      // No need to generate reconnection Failed Event as this is only triggered when specified
      // schedule timer
      // time got elapsed else system will always retrying.
    }
  }

  private fun onDisconnected() {
    emitEvent { it.onDisconnectedFromMeetingRoom() }
  }

  private fun emitReconnectingIfNotEmittedBefore() {
    if (prevConnected) {
      if (isAutoRetryEnabled()) {
        if (startRetrying == false) {
          startRetrying = true
          emitEvent { it.onReconnectingToMeetingRoom() }
        }
      } else {
        // Client of SDK is responsible to initiate a rejoin mechanism. For this we need a method
        // for it. Will support in Next Release
        // For now we are not supporting it.
      }
    }
  }

  private fun onOldSocketConnectionEvent(event: SocketConnectionEvent) {
    when (event) {
      is SocketConnectionEvent.Connected -> {
        DyteLogger.info("SocketController Connection::SocketIO Connected")
        onConnected()
      }
      is SocketConnectionEvent.Connecting -> {
        DyteLogger.info("SocketController Connection::SocketIO Connecting")
        onConnecting()
      }
      is SocketConnectionEvent.ConnectionFailed -> {
        DyteLogger.info("SocketController Connection::SocketIO ConnectionFailed")
        onConnectFailed()
      }
      is SocketConnectionEvent.Disconnected -> {
        DyteLogger.info("SocketController Connection::SocketIO Disconnected")
        onDisconnected()
      }
      else -> {}
    }
  }

  private fun onNewSocketConnectionEvent(state: WebSocketConnectionState) {
    DyteLogger.info("SocketController::${state}")
    when (state) {
      is WebSocketConnectionState.Connected -> {
        DyteLogger.info("SocketController Connection::SocketService Connected")
        onConnected()
      }
      is WebSocketConnectionState.Connecting -> {
        DyteLogger.info("SocketController Connection::SocketService Connecting")
        onConnecting()
      }
      is WebSocketConnectionState.ConnectFailed -> {
        DyteLogger.info("SocketController Connection::SocketService ConnectFailed")
        onConnectFailed()
      }
      is WebSocketConnectionState.Disconnected -> {
        DyteLogger.info("SocketController Connection::SocketService Disconnected")
        onDisconnected()
      }
      else -> {}
    }
  }

  private suspend fun onOldSocketConnectionMediaEvent(event: SocketConnectionEvent) {
    DyteLogger.info("SocketController::Old::${event}")
    when (event) {
      is SocketConnectionEvent.Connected -> {
        if (!useHive) {
          internalEmitter.emitEvent { it.connectMedia() }
        }
      }
      is SocketConnectionEvent.Disconnected -> {
        // do not reconnect if manually disconnected
        // https://socket.io/docs/v4/client-socket-instance/#disconnect
        if (event.reason != "io client disconnect") {
          reconnectSocketsWithNewPeerId()
        }
      }
      else -> {}
    }
  }

  private suspend fun onNewSocketConnectionMediaEvent(state: WebSocketConnectionState) {
    DyteLogger.info("SocketController::New::${state}")
    when (state) {
      is WebSocketConnectionState.Connected -> {
        if (controllerContainer.selfController._roomJoined) {
          if (useHive) {
            // No need to send JOIN_ROOM to socket-service separately.
            // Hive controller internally does both join room and media connection
            internalEmitter.emitEvent { it.connectMedia() }
          } else {
            // For Mediasoup room-node meetings we just need to send JOIN_ROOM to socket-service.
            // Media connection relies on Old socket (SocketIO) connection.
            joinRoomSockratesSocketService()
          }
        }
      }
      is WebSocketConnectionState.Disconnected -> {
        // do not reconnect if manually disconnected
        // code == 1000
        if (state.code != 1000) {
          reconnectSocketsWithNewPeerId()
        }
      }
      else -> {}
    }
  }

  /**
   * Changes socket Urls of both SocketController and SocketService client to use the new peerId and
   * reconnects to both RoomNode socket and SocketService.
   */
  private suspend fun reconnectSocketsWithNewPeerId() {
    DyteLogger.info("SocketController: reconnectSocketsWithNewPeerId $reconnectingToMeetingRoom")
    if (!reconnectingToMeetingRoom) {
      reconnectingToMeetingRoom = true
      internalEmitter.emitEvent { it.disconnectMedia() }

      if (!useHive) {
        oldSocket.disconnect()
      }
      newSocket.disconnect()

      val peerId = controllerContainer.metaController.refreshPeerId()

      if (!useHive) {
        oldSocket.refreshUrl(peerId)
      }
      newSocket.refreshUrl(peerId)

      if (!useHive) {
        oldSocket.connect()
        emitEvent { it.onReconnectingToMeetingRoom() }
      }
      newSocket.connect()
    }
  }
}

internal expect class PlatformConnectionChangeListener(
  provider: IDytePlatformUtilsProvider,
  disconnectCallback: () -> Unit
) {}
