package io.dyte.core.room

import io.dyte.core.controllers.DyteStageStatus
import io.dyte.core.observability.DyteLogger
import io.dyte.core.socket.socketservice.ISockratesSocketService
import io.dyte.core.socket.socketservice.SocketServiceUtils
import socket.room.GetPeerInfoRequest
import socket.room.GetRoomStageStateResponse
import socket.room.Peer
import socket.room.PeerInfoResponse
import socket.room.StageType

internal interface RoomSocketHandler {
  suspend fun getRoomStageState(): RoomStageState?

  suspend fun getPeerInfo(peerId: String): RoomParticipantInfo?
}

internal data class RoomStageState(
  val onStagePeerIds: Set<String>,
  val approvedStagePeerIds: Set<String>,
  val requestStagePeerIds: Set<String>
)

internal data class RoomParticipantInfo(
  val peerId: String,
  val userId: String,
  val displayName: String,
  val stageStatus: DyteStageStatus?,
  val customParticipantId: String?,
  val presetId: String?,
  val displayPictureUrl: String?,
  val waitlisted: Boolean
)

internal class SocketServiceRoomSocketHandler(private val socketService: ISockratesSocketService) :
  RoomSocketHandler {
  private val logger = DyteLogger

  override suspend fun getRoomStageState(): RoomStageState? {
    val socketResponse =
      socketService.requestResponse(
        event = SocketServiceUtils.RoomEvent.GET_ROOM_STAGE_STATE.id,
        payload = null
      )
        ?: kotlin.run {
          logger.warn("RoomSocketHandler::getRoomStageState::socket response is null")
          return null
        }

    val getRoomStageStateResponse =
      try {
        GetRoomStageStateResponse.ADAPTER.decode(socketResponse)
      } catch (e: Exception) {
        logger.warn(
          "RoomSocketHandler::getRoomStageState::failed to decode GetRoomStageStateResponse"
        )
        return null
      }

    return getRoomStageStateResponse.toRoomStageState()
  }

  override suspend fun getPeerInfo(peerId: String): RoomParticipantInfo? {
    val getPeerInfoRequest = GetPeerInfoRequest(peerId)
    val socketResponse =
      socketService.requestResponse(
        event = SocketServiceUtils.RoomEvent.GET_PEER_INFO.id,
        payload = GetPeerInfoRequest.ADAPTER.encode(getPeerInfoRequest)
      )
        ?: kotlin.run {
          logger.warn("RoomSocketHandler::getPeerInfo::socket response is null")
          return null
        }

    val peerInfoResponse =
      try {
        PeerInfoResponse.ADAPTER.decode(socketResponse)
      } catch (e: Exception) {
        logger.warn("RoomSocketHandler::getPeerInfo::failed to decode PeerInfoResponse")
        return null
      }

    val peer =
      peerInfoResponse.peer
        ?: kotlin.run {
          logger.warn("RoomSocketHandler::getPeerInfo::received peer is null")
          return null
        }

    return peer.toRoomParticipantInfo()
  }

  companion object {
    private fun GetRoomStageStateResponse.toRoomStageState() =
      RoomStageState(
        on_stage_peers.toSet(),
        approved_stage_peers.toSet(),
        requested_stage_peers.toSet()
      )

    private fun Peer.toRoomParticipantInfo() =
      RoomParticipantInfo(
        peer_id,
        user_id,
        display_name,
        stage_type?.toDyteStageStatus(),
        custom_participant_id,
        preset_id,
        display_picture_url,
        waitlisted
      )

    // TODO: Move to a common place (mappers) so that it can be used by other components
    private fun StageType.toDyteStageStatus(): DyteStageStatus {
      return when (this) {
        StageType.STAGE_TYPE_ON_STAGE -> DyteStageStatus.ON_STAGE
        StageType.STAGE_TYPE_APPROVED_STAGE -> DyteStageStatus.ACCEPTED_TO_JOIN_STAGE
        StageType.STAGE_TYPE_REQUESTED_STAGE -> DyteStageStatus.REQUESTED_TO_JOIN_STAGE
        else -> DyteStageStatus.OFF_STAGE
      }
    }
  }
}
