package io.dyte.core.plugins.socketservice

import io.dyte.core.plugins.socketservice.PluginStore.PluginStoreItem
import io.dyte.core.plugins.socketservice.PluginsSocketHandler.ActivePlugin
import io.dyte.core.plugins.socketservice.PluginsSocketHandler.PluginStoreKeyEntry
import io.dyte.core.socket.socketservice.ISockratesSocketService
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.channels.ProducerScope
import kotlinx.coroutines.channels.awaitClose
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.callbackFlow
import kotlinx.coroutines.flow.flowOn
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import okio.ByteString
import okio.ByteString.Companion.encodeUtf8
import socket.plugin.AddPluginRequest
import socket.plugin.DisablePluginForPeersRequest
import socket.plugin.DisablePluginForRoomRequest
import socket.plugin.DisablePluginResponse
import socket.plugin.EnablePluginForPeersRequest
import socket.plugin.EnablePluginForRoomRequest
import socket.plugin.EnablePluginResponse
import socket.plugin.EnablePluginsResponse
import socket.plugin.PluginEventResponse
import socket.plugin.PluginEventToPeersRequest
import socket.plugin.PluginEventToRoomRequest
import socket.plugin.PluginStoreDeleteKeysRequest
import socket.plugin.PluginStoreDeleteRequest
import socket.plugin.PluginStoreGetKeysRequest
import socket.plugin.PluginStoreInsertKeysRequest
import socket.plugin.PluginStoreResponse
import socket.plugin.RemovePluginRequest
import socket.plugin.StoreKeys

internal class DefaultPluginsSocketHandler(
  private val socketService: ISockratesSocketService,
  private val json: Json
) : PluginsSocketHandler {

  override suspend fun addPlugin(pluginId: String, staggered: Boolean) {
    val addPluginRequest = AddPluginRequest(pluginId, staggered)
    socketService.send(
      event = PluginsSocketEvent.ADD_PLUGIN.id,
      payload = AddPluginRequest.ADAPTER.encode(addPluginRequest)
    )
  }

  override suspend fun removePlugin(pluginId: String) {
    val removePluginRequest = RemovePluginRequest(pluginId)
    socketService.send(
      event = PluginsSocketEvent.REMOVE_PLUGIN.id,
      payload = RemovePluginRequest.ADAPTER.encode(removePluginRequest)
    )
  }

  override suspend fun getActivePlugins(): List<ActivePlugin> {
    try {
      val response = socketService.requestResponse(PluginsSocketEvent.GET_PLUGINS.id, null)

      // In case there are no active plugins, a null response is returned by socket server :(
      if (response == null) {
        return emptyList()
      }

      val getActivePluginsResponse = EnablePluginsResponse.ADAPTER.decode(response)
      val activePlugins =
        getActivePluginsResponse.plugins.map { ActivePlugin(it.plugin_id, it.enabled_by) }
      return activePlugins
    } catch (e: Exception) {
      throw Exception("Failed to get active plugins from server")
    }
  }

  override suspend fun sendCustomPluginEventToRoom(
    pluginId: String,
    data: JsonElement,
    messageId: String?
  ) {
    val pluginEventToRoomRequest =
      PluginEventToRoomRequest(plugin_id = pluginId, plugin_data = json.encodeToByteString(data))
    socketService.send(
      event = PluginsSocketEvent.CUSTOM_PLUGIN_EVENT_TO_ROOM.id,
      payload = PluginEventToRoomRequest.ADAPTER.encode(pluginEventToRoomRequest),
      messageId = messageId
    )
  }

  override suspend fun sendCustomPluginEventToPeers(
    pluginId: String,
    peerIds: List<String>,
    data: JsonElement,
    messageId: String?
  ) {
    val pluginEventToPeersRequest =
      PluginEventToPeersRequest(
        plugin_id = pluginId,
        peer_ids = peerIds,
        plugin_data = json.encodeToByteString(data)
      )
    socketService.send(
      event = PluginsSocketEvent.CUSTOM_PLUGIN_EVENT_TO_PEERS.id,
      payload = PluginEventToPeersRequest.ADAPTER.encode(pluginEventToPeersRequest),
      messageId = messageId
    )
  }

  override suspend fun enablePluginForRoom(pluginId: String, messageId: String?) {
    val enablePluginForRoomRequest = EnablePluginForRoomRequest(pluginId)
    socketService.send(
      event = PluginsSocketEvent.ENABLE_PLUGIN_FOR_ROOM.id,
      payload = EnablePluginForRoomRequest.ADAPTER.encode(enablePluginForRoomRequest),
      messageId = messageId
    )
  }

  override suspend fun disablePluginForRoom(pluginId: String, messageId: String?) {
    val disablePluginForRoomRequest = DisablePluginForRoomRequest(pluginId)
    socketService.send(
      event = PluginsSocketEvent.DISABLE_PLUGIN_FOR_ROOM.id,
      payload = DisablePluginForRoomRequest.ADAPTER.encode(disablePluginForRoomRequest),
      messageId = messageId
    )
  }

  override suspend fun enablePluginForPeers(
    pluginId: String,
    peerIds: List<String>,
    messageId: String?
  ) {
    val enablePluginForPeersRequest = EnablePluginForPeersRequest(pluginId, peerIds)
    socketService.send(
      event = PluginsSocketEvent.ENABLE_PLUGIN_FOR_PEERS.id,
      payload = EnablePluginForPeersRequest.ADAPTER.encode(enablePluginForPeersRequest),
      messageId = messageId
    )
  }

  override suspend fun disablePluginForPeers(
    pluginId: String,
    peerIds: List<String>,
    messageId: String?
  ) {
    val disablePluginForPeersRequest = DisablePluginForPeersRequest(pluginId, peerIds)
    socketService.send(
      event = PluginsSocketEvent.DISABLE_PLUGIN_FOR_PEERS.id,
      payload = DisablePluginForPeersRequest.ADAPTER.encode(disablePluginForPeersRequest),
      messageId = messageId
    )
  }

  override suspend fun storeInsertKeys(
    pluginId: String,
    store: String,
    insertKeys: List<PluginStoreKeyEntry>,
    messageId: String?
  ) {
    val storeKeys =
      insertKeys.map { pluginStoreKeyEntry ->
        val storeKeyPayload = pluginStoreKeyEntry.payload?.let { json.encodeToByteString(it) }
        StoreKeys(pluginStoreKeyEntry.key, storeKeyPayload)
      }
    val pluginStoreInsertKeysRequest = PluginStoreInsertKeysRequest(pluginId, store, storeKeys)
    socketService.send(
      event = PluginsSocketEvent.STORE_INSERT_KEYS.id,
      payload = PluginStoreInsertKeysRequest.ADAPTER.encode(pluginStoreInsertKeysRequest),
      messageId = messageId
    )
  }

  override suspend fun storeGetKeys(
    pluginId: String,
    store: String,
    getKeys: List<String>,
    messageId: String?
  ) {
    val storeKeys = getKeys.map { key -> StoreKeys(key) }
    val pluginStoreGetKeysRequest = PluginStoreGetKeysRequest(pluginId, store, storeKeys)
    socketService.send(
      event = PluginsSocketEvent.STORE_GET_KEYS.id,
      payload = PluginStoreGetKeysRequest.ADAPTER.encode(pluginStoreGetKeysRequest),
      messageId = messageId
    )
  }

  override suspend fun storeDeleteKeys(
    pluginId: String,
    store: String,
    deleteKeys: List<String>,
    messageId: String?
  ) {
    val storeKeys = deleteKeys.map { key -> StoreKeys(key) }
    val pluginStoreDeleteKeysRequest = PluginStoreDeleteKeysRequest(pluginId, store, storeKeys)
    socketService.send(
      event = PluginsSocketEvent.STORE_DELETE_KEYS.id,
      payload = PluginStoreDeleteKeysRequest.ADAPTER.encode(pluginStoreDeleteKeysRequest),
      messageId = messageId
    )
  }

  override suspend fun deleteStore(pluginId: String, store: String, messageId: String?) {
    val pluginStoreDeleteRequest = PluginStoreDeleteRequest(pluginId, store)
    socketService.send(
      event = PluginsSocketEvent.STORE_DELETE.id,
      payload = PluginStoreDeleteRequest.ADAPTER.encode(pluginStoreDeleteRequest),
      messageId = messageId
    )
  }

  override fun listenToPluginSocketMessages(): Flow<PluginSocketMessage> =
    callbackFlow {
        val socketServiceEventListener =
          object : SocketServiceEventListener {
            override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
              handleSocketServiceEvent(event, eventId, payload, this@callbackFlow)
            }
          }

        subscribeToPluginsSocketServiceEvents(socketServiceEventListener)

        awaitClose { unsubscribeToPluginsSocketServiceEvents(socketServiceEventListener) }
      }
      .flowOn(Dispatchers.Default)

  private fun subscribeToPluginsSocketServiceEvents(listener: SocketServiceEventListener) {
    with(socketService) {
      subscribe(PluginsSocketEvent.ADD_PLUGIN.id, listener)
      subscribe(PluginsSocketEvent.ENABLE_PLUGIN_FOR_PEERS.id, listener)
      subscribe(PluginsSocketEvent.ENABLE_PLUGIN_FOR_ROOM.id, listener)
      subscribe(PluginsSocketEvent.REMOVE_PLUGIN.id, listener)
      subscribe(PluginsSocketEvent.DISABLE_PLUGIN_FOR_PEERS.id, listener)
      subscribe(PluginsSocketEvent.DISABLE_PLUGIN_FOR_ROOM.id, listener)
      subscribe(PluginsSocketEvent.CUSTOM_PLUGIN_EVENT_TO_PEERS.id, listener)
      subscribe(PluginsSocketEvent.CUSTOM_PLUGIN_EVENT_TO_ROOM.id, listener)
      subscribe(PluginsSocketEvent.STORE_INSERT_KEYS.id, listener)
      subscribe(PluginsSocketEvent.STORE_GET_KEYS.id, listener)
      subscribe(PluginsSocketEvent.STORE_DELETE_KEYS.id, listener)
      subscribe(PluginsSocketEvent.STORE_DELETE.id, listener)
    }
  }

  private fun unsubscribeToPluginsSocketServiceEvents(listener: SocketServiceEventListener) {
    with(socketService) {
      unsubscribe(PluginsSocketEvent.ADD_PLUGIN.id, listener)
      unsubscribe(PluginsSocketEvent.ENABLE_PLUGIN_FOR_PEERS.id, listener)
      unsubscribe(PluginsSocketEvent.ENABLE_PLUGIN_FOR_ROOM.id, listener)
      unsubscribe(PluginsSocketEvent.REMOVE_PLUGIN.id, listener)
      unsubscribe(PluginsSocketEvent.DISABLE_PLUGIN_FOR_PEERS.id, listener)
      unsubscribe(PluginsSocketEvent.DISABLE_PLUGIN_FOR_ROOM.id, listener)
      unsubscribe(PluginsSocketEvent.CUSTOM_PLUGIN_EVENT_TO_PEERS.id, listener)
      unsubscribe(PluginsSocketEvent.CUSTOM_PLUGIN_EVENT_TO_ROOM.id, listener)
      unsubscribe(PluginsSocketEvent.STORE_INSERT_KEYS.id, listener)
      unsubscribe(PluginsSocketEvent.STORE_GET_KEYS.id, listener)
      unsubscribe(PluginsSocketEvent.STORE_DELETE_KEYS.id, listener)
      unsubscribe(PluginsSocketEvent.STORE_DELETE.id, listener)
    }
  }

  private fun handleSocketServiceEvent(
    event: Int,
    id: String?,
    payload: ByteArray?,
    socketMessagesFlow: ProducerScope<PluginSocketMessage>
  ) {
    when (event) {
      PluginsSocketEvent.ADD_PLUGIN.id -> {
        try {
          if (payload == null) return

          val enablePluginResponse = EnablePluginResponse.ADAPTER.decode(payload)
          val addPlugin =
            PluginSocketMessage.AddPlugin(
              pluginId = enablePluginResponse.plugin_id,
              enabledBy = enablePluginResponse.enabled_by
            )
          socketMessagesFlow.trySend(addPlugin)
        } catch (e: Exception) {
          // TODO: Log plugin socket message parsing error - ADD_PLUGIN
        }
      }
      PluginsSocketEvent.ENABLE_PLUGIN_FOR_PEERS.id -> {
        try {
          if (payload == null) return

          val enablePluginResponse = EnablePluginResponse.ADAPTER.decode(payload)
          val enablePluginForPeers =
            PluginSocketMessage.EnablePluginForPeers(
              id = id,
              pluginId = enablePluginResponse.plugin_id,
              enabledBy = enablePluginResponse.enabled_by
            )
          socketMessagesFlow.trySend(enablePluginForPeers)
        } catch (e: Exception) {
          // TODO: Log plugin socket message parsing error - ENABLE_PLUGIN_FOR_PEERS
        }
      }
      PluginsSocketEvent.ENABLE_PLUGIN_FOR_ROOM.id -> {
        try {
          if (payload == null) return

          val enablePluginResponse = EnablePluginResponse.ADAPTER.decode(payload)
          val enablePluginForRoom =
            PluginSocketMessage.EnablePluginForRoom(
              id = id,
              pluginId = enablePluginResponse.plugin_id,
              enabledBy = enablePluginResponse.enabled_by
            )
          socketMessagesFlow.trySend(enablePluginForRoom)
        } catch (e: Exception) {
          // TODO: Log plugin socket message parsing error - ENABLE_PLUGIN_FOR_ROOM
        }
      }
      PluginsSocketEvent.REMOVE_PLUGIN.id -> {
        try {
          if (payload == null) return

          val disablePluginResponse = DisablePluginResponse.ADAPTER.decode(payload)
          val removePlugin =
            PluginSocketMessage.RemovePlugin(
              pluginId = disablePluginResponse.plugin_id,
              disabledBy = disablePluginResponse.disabled_by
            )
          socketMessagesFlow.trySend(removePlugin)
        } catch (e: Exception) {
          // TODO: Log plugin socket message parsing error - REMOVE_PLUGIN
        }
      }
      PluginsSocketEvent.DISABLE_PLUGIN_FOR_PEERS.id -> {
        try {
          if (payload == null) return

          val disablePluginResponse = DisablePluginResponse.ADAPTER.decode(payload)
          val disablePluginForPeers =
            PluginSocketMessage.DisablePluginForPeers(
              id = id,
              pluginId = disablePluginResponse.plugin_id,
              disabledBy = disablePluginResponse.disabled_by
            )
          socketMessagesFlow.trySend(disablePluginForPeers)
        } catch (e: Exception) {
          // TODO: Log plugin socket message parsing error - DISABLE_PLUGIN_FOR_PEERS
        }
      }
      PluginsSocketEvent.DISABLE_PLUGIN_FOR_ROOM.id -> {
        try {
          if (payload == null) return

          val disablePluginResponse = DisablePluginResponse.ADAPTER.decode(payload)
          val disablePluginForRoom =
            PluginSocketMessage.DisablePluginForRoom(
              id = id,
              pluginId = disablePluginResponse.plugin_id,
              disabledBy = disablePluginResponse.disabled_by
            )
          socketMessagesFlow.trySend(disablePluginForRoom)
        } catch (e: Exception) {
          // TODO: Log plugin socket message parsing error - DISABLE_PLUGIN_FOR_ROOM
        }
      }
      PluginsSocketEvent.CUSTOM_PLUGIN_EVENT_TO_PEERS.id -> {
        try {
          if (payload == null) return

          val pluginEventResponse = PluginEventResponse.ADAPTER.decode(payload)
          val customPluginEventToPeers =
            PluginSocketMessage.CustomPluginEventToPeers(
              id = id,
              pluginId = pluginEventResponse.plugin_id,
              pluginData = json.decodeFromByteString(pluginEventResponse.plugin_data)
            )
          socketMessagesFlow.trySend(customPluginEventToPeers)
        } catch (e: Exception) {
          // TODO: Log plugin socket message parsing error - CUSTOM_PLUGIN_EVENT_TO_PEERS
        }
      }
      PluginsSocketEvent.CUSTOM_PLUGIN_EVENT_TO_ROOM.id -> {
        try {
          if (payload == null) return

          val pluginEventResponse = PluginEventResponse.ADAPTER.decode(payload)
          val customPluginEventToRoom =
            PluginSocketMessage.CustomPluginEventToRoom(
              id = id,
              pluginId = pluginEventResponse.plugin_id,
              pluginData = json.decodeFromByteString(pluginEventResponse.plugin_data)
            )
          socketMessagesFlow.trySend(customPluginEventToRoom)
        } catch (e: Exception) {
          // TODO: Log plugin socket message parsing error - CUSTOM_PLUGIN_EVENT_TO_ROOM
        }
      }
      PluginsSocketEvent.STORE_INSERT_KEYS.id -> {
        try {
          if (payload == null) return

          val pluginStoreResponse = PluginStoreResponse.ADAPTER.decode(payload)
          val pluginStore = pluginStoreResponse.toPluginStore(json)
          val storeInsertKeys = PluginSocketMessage.StoreInsertKeys(id, pluginStore)
          socketMessagesFlow.trySend(storeInsertKeys)
        } catch (e: Exception) {
          // TODO: Log plugin socket message parsing error - STORE_INSERT_KEYS
        }
      }
      PluginsSocketEvent.STORE_GET_KEYS.id -> {
        try {
          if (payload == null) return

          val pluginStoreResponse = PluginStoreResponse.ADAPTER.decode(payload)
          val pluginStore = pluginStoreResponse.toPluginStore(json)
          val storeGetKeys = PluginSocketMessage.StoreGetKeys(id, pluginStore)
          socketMessagesFlow.trySend(storeGetKeys)
        } catch (e: Exception) {
          // TODO: Log plugin socket message parsing error - STORE_GET_KEYS
        }
      }
      PluginsSocketEvent.STORE_DELETE_KEYS.id -> {
        try {
          if (payload == null) return

          val pluginStoreResponse = PluginStoreResponse.ADAPTER.decode(payload)
          val pluginStore = pluginStoreResponse.toPluginStore(json)
          val storeDeleteKeys = PluginSocketMessage.StoreDeleteKeys(id, pluginStore)
          socketMessagesFlow.trySend(storeDeleteKeys)
        } catch (e: Exception) {
          // TODO: Log plugin socket message parsing error - STORE_DELETE_KEYS
        }
      }
      PluginsSocketEvent.STORE_DELETE.id -> {
        try {
          if (payload == null) return

          val pluginStoreResponse = PluginStoreResponse.ADAPTER.decode(payload)
          val storeDelete =
            PluginSocketMessage.StoreDelete(
              id = id,
              pluginId = pluginStoreResponse.plugin_id,
              storeName = pluginStoreResponse.store_name
            )
          socketMessagesFlow.trySend(storeDelete)
        } catch (e: Exception) {
          // TODO: Log plugin socket message parsing error - STORE_DELETE
        }
      }
      else -> {
        // TODO: log invalid plugin socket event
      }
    }
  }

  companion object {
    private fun Json.encodeToByteString(jsonElement: JsonElement): ByteString {
      return encodeToString(jsonElement).encodeUtf8()
    }

    private fun Json.decodeFromByteString(byteString: ByteString): JsonElement {
      return parseToJsonElement(byteString.utf8())
    }

    private fun PluginStoreResponse.toPluginStore(json: Json): PluginStore {
      val pluginStoreItems =
        store_items.map {
          val jsonPayload =
            try {
              json.decodeFromByteString(it.payload)
            } catch (e: Exception) {
              JsonObject(emptyMap())
            }
          PluginStoreItem(it.timestamp, it.peer_id, it.store_key, jsonPayload)
        }

      return PluginStore(plugin_id, store_name, pluginStoreItems)
    }
  }
}
