package io.dyte.core.observability

import defaults.BuildKonfig
import io.dyte.core.controllers.Controller
import io.dyte.core.network.BaseApiService
import io.dyte.core.observability.LogSeverity.debug
import io.dyte.core.observability.LogSeverity.error
import io.dyte.core.observability.LogSeverity.info
import io.dyte.core.observability.LogSeverity.warn
import io.ktor.utils.io.core.toByteArray
import kotlin.jvm.Synchronized
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable

object DyteLogger {
  private var controller: Controller? = null

  private var logs: ArrayList<DyteLog> = arrayListOf()

  private lateinit var meetingMetaData: MeetingMetaData

  private var logId = 0

  // TODO create util class timer
  private suspend fun startTimerContinuous(tickInMillis: Long) {
    while (true) {
      processLogs()
      delay(tickInMillis)
    }
  }

  private suspend fun processLogs(shouldPushAll: Boolean = false) {
    var logsToBeProcessed = logs.take(if (shouldPushAll) Int.MAX_VALUE else 500)
    if (logsToBeProcessed.isEmpty()) {
      return
    }

    try {
      val payload =
        PostDyteLogs(meetingMetaData, logsToBeProcessed.map { it.asMap() }, "mobile-core")
      val sizeInBytes =
        BaseApiService.json.encodeToString(PostDyteLogs.serializer(), payload).toByteArray().size
      val sizeInKB = sizeInBytes / 1024
      val newPayload =
        if (sizeInKB > 150) {
          logsToBeProcessed = logsToBeProcessed.filter { it.level == warn || it.level == error }
          PostDyteLogs(meetingMetaData, logsToBeProcessed.map { it.asMap() }, "mobile-core")
        } else {
          payload
        }
      controller?.apiClient?.uploadLogs(newPayload)
      logs.removeAll(logsToBeProcessed)
    } catch (e: Exception) {
      // TODO: Possibly internet failure
    }
  }

  internal fun init(c: Controller) {
    controller = c
    val platformUtils = c.platformUtilsProvider.getPlatformUtils()
    val meta = c.metaController
    c.serialScope.launch {
      meetingMetaData =
        MeetingMetaData(
          meta.getPeerId(),
          meta.getRoomName(),
          meta.getOrgId(),
          BuildKonfig.sdkVersion,
          true,
          platformUtils.getDeviceInfo(),
          platformUtils.getDeviceModel(),
          platformUtils.getDeviceType(),
          platformUtils.getSdkType(),
          platformUtils.getBatteryLevel(),
          platformUtils.getOsVersion()
        )
      startTimerContinuous(7000)
    }
  }

  fun info(message: String, attr: Map<String, String>? = null) {
    log(message, info, attr)
  }

  fun warn(message: String, attr: Map<String, String>? = null) {
    log(message, warn, attr)
  }

  fun error(message: String, attr: Map<String, String>? = null) {
    log(message, error, attr)
  }

  fun error(message: String, e: Throwable) {
    log(
      message,
      error,
      mapOf(
        "error.message" to e.message,
        "error.stack" to e.stackTraceToString(),
        "error.isException" to "true"
      )
    )
  }

  fun debug(message: String, attr: Map<String, String>? = null) {
    log(message, debug, attr)
  }

  @Synchronized
  private fun log(message: String, severity: LogSeverity, attr: Map<String, String?>? = null) {
    println("dytelog:: $severity $message")
    controller?.let {
      it.serialScope.launch {
        withContext(it.serialScope.coroutineContext) {
          logs.add(
            DyteLog(
              logId,
              message,
              it.platformUtilsProvider.getPlatformUtils().getUtcTimeNow(),
              severity,
              attr
            )
          )
          logId++
          if (severity == error) {
            pushAllNow()
          }
        }
      }
    }
  }

  fun pushAllNow() {
    controller?.serialScope?.launch { processLogs(true) }
  }

  fun setPostInitInfo(organizationId: String, roomName: String) {
    controller?.serialScope?.launch {
      meetingMetaData = meetingMetaData.copy(organizationId = organizationId, roomName = roomName)
    }
  }
}

@Serializable
data class DyteLog(
  val id: Int,
  val message: String,
  val loggedAt: String,
  val level: LogSeverity,
  val attrs: Map<String, String?>?
) {
  fun asMap(): Map<String, String?> {
    return mapOf(
        "id" to id.toString(),
        "message" to message,
        "loggedAt" to loggedAt,
        "level" to level.name,
      )
      .plus(if (attrs !== null) attrs else mapOf())
  }
}

@Serializable
data class PostDyteLogs(
  val meetingMetadata: MeetingMetaData,
  val logs: List<Map<String, String?>>,
  val serviceName: String
)

@Serializable
data class MeetingMetaData(
  @SerialName("meetingMetadata.peerId") val peerId: String,
  @SerialName("meetingMetadata.roomName") val roomName: String,
  @SerialName("meetingMetadata.organizationId") val organizationId: String,
  @SerialName("meetingMetadata.sdkVersion") val sdkVersion: String,
  @SerialName("meetingMetadata.deviceInfo.isMobile") val isMobile: Boolean,
  @SerialName("meetingMetadata.deviceInfo.deviceInfo") val deviceInfo: String,
  @SerialName("meetingMetadata.deviceInfo.deviceModel") val deviceModel: String,
  @SerialName("meetingMetadata.deviceInfo.deviceType") val deviceType: String,
  @SerialName("meetingMetadata.deviceInfo.sdkType") val sdkType: String,
  @SerialName("meetingMetadata.deviceInfo.batteryLevel") val batteryLevel: String,
  @SerialName("meetingMetadata.deviceInfo.osVersion") val osVersion: String,
)

enum class LogSeverity {
  info,
  error,
  debug,
  warn
}
