package io.dyte.core.models

// (Manish): The structure for differentiation of compulsory peer and active speaker peer
// in selected peers is explained below:
//
// The Priority line:
//
// The absolute refrence point is 0.
// The negative side of the X axis represents compulsory peers.
// The Positive side of the Y axis represents active speaker peers.
//
//
// -X axis   ---|---|---|---|---|---|---|---|---|---|---|---|---   +X axis
//              5   4   3   2   1   0   1   2   3   4   5   6
//              (Compulsory Peers)     (Active speaker Peers)
//
//                 (Priority line for selected peers)
//
//
// For compulsory peers:
// The more negative Int of the X axis from 0, the less priority it has for the compulsory peer.
//
// For active speaker peers:
// The more positive Int of the X axis from 0, the less priority it has for active speaker peer.
//
// (NOTE): 0 is the higeshest priority for active speaker peers and -1 is the highest priority
// for compulsory peers.
//
// Eg:
// let's suppose we have selected peers as y = [(a,-3),(b,-2),(c,-1),(d,0),(e,1),(f,2),(g,3)]
// Here we can seperate out active speakers and compulsory peers easily.
// According to our priority line, we can see that the active speaker peers are: [d,e,f,g]
// and the compulsory peers are: [a,b,c]
//
// Now if we want to get the order of active speaker then, it would be [d,e,f,g] as
// d=>0(0 is the higest priority), e=>1, f=>2, g=>3 (3 is the lowest priority)
//
// For compulsory peers we always get priority -1 since we have only
// screenshare peer as compulsory peer and we store in different array.

data class PeerPriorityObject(val peerId: String, val priority: Int)

class PrioritySet {
  val _orderedArray: MutableList<String> = mutableListOf()

  private val _map: MutableMap<String, PeerPriorityObject> = mutableMapOf()

  fun add(peerId: String, priority: Int): Int {
    if (!this._map.containsKey(peerId)) {
      this._map[peerId] = PeerPriorityObject(peerId, priority)
      this._orderedArray.add(maxOf(priority - 1, 0), peerId)
      return index(peerId)
    }
    val currentIdx = this.index(peerId)
    this.delete(peerId)
    val newIdx = this.add(peerId, priority)
    if (currentIdx != newIdx) {
      return newIdx
    }
    return -1
  }

  fun delete(peerId: String) {
    if (this._map.containsKey(peerId)) {
      val idx = this.index(peerId)
      this._map.remove(peerId)
      this._orderedArray.removeAt(idx)
    }
  }

  fun index(peerId: String): Int {
    if (this._map.containsKey(peerId)) {
      return this._orderedArray.indexOf(peerId)
    }
    return -1
  }
}

class DyteSelectedPeers {
  private val _activeSpeakerPeers: PrioritySet = PrioritySet()

  private val _compulsoryPeers: Set<String> = setOf()

  val peers: List<String>
    get() = _peers()

  val compulsoryPeers: List<String>
    get() = _compulsoryPeers.toList()

  val peerPriorities: List<Int>
    get() = _peerPriorities()

  val activeSpeakerPeers: List<String>
    get() = _activeSpeakerPeers()

  fun add(peerId: String, priority: Int): Int {
    if (priority < 0) {
      this._compulsoryPeers.plus(peerId)
      return 0
    }

    // This is for removing peer from selectedpeer list
    // Mostly useful in removing from compulsory peer
    // We also check if we are getting volume events for this peer then
    // we need to remove it from compulsory peer's list.
    // hive mind will make sure if a peer is compulsory peer it's volume
    // diff event doesn't gets emitted.
    if (this._compulsoryPeers.contains(peerId) && (priority > 0 || priority == 246267631)) {
      // println("DyteSelectedPeer::removing_compulsory_peer $peerId")
      this._removeFromCompulsoryPeer(peerId)
      if (priority == 246267631) return -1
    }

    return this._activeSpeakerPeers.add(peerId, priority)
  }

  fun delete(peerId: String) {
    // println("DyteSelectedPeer::deleting_peer_from_selectedPeer $peerId")
    this._removeFromCompulsoryPeer(peerId)
    this._activeSpeakerPeers.delete(peerId)
  }

  fun index(peerId: String): Int {
    return this._activeSpeakerPeers.index(peerId)
  }

  fun _peers(): List<String> {
    val p = this.compulsoryPeers.toList().toMutableList()
    p.addAll(this.activeSpeakerPeers)
    return p.toList()
  }

  fun _compulsoryPeers(): List<String> {
    return this.compulsoryPeers
  }

  fun _peerPriorities(): List<Int> {
    return this._activeSpeakerPeers._orderedArray.mapIndexed { index, _ -> index }
  }

  fun _activeSpeakerPeers(): List<String> {
    return this._activeSpeakerPeers._orderedArray.toList()
  }

  private fun _removeFromCompulsoryPeer(peerId: String) {
    this._compulsoryPeers.minus(peerId)
  }
}
