package io.dyte.core.media

import io.dyte.callstats.CallStats
import io.dyte.callstats.media.ConsumerFacade
import io.dyte.callstats.media.ProducerFacade
import io.dyte.callstats.media.TransportFacade
import io.dyte.callstats.models.AuthPayload
import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.controllers.PermissionType
import io.dyte.core.hive.HiveSFUSocketHandler
import io.dyte.core.models.DyteJoinedMeetingParticipant
import io.dyte.core.models.DyteParticipants
import io.dyte.core.network.models.IceServerData
import io.dyte.core.observability.DyteLogger
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerClosedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerResumedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerMuteModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebRtcCreateTransportModel
import io.dyte.media.hive.*
import io.dyte.media.hive.handlers.HiveCodecOptions
import io.dyte.media.utils.IMediaClientLogger
import io.dyte.media.utils.sdp.SDPUtils
import io.dyte.webrtc.*
import io.ktor.utils.io.core.*
import kotlin.Pair
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonPrimitive
import media.CreateTransportRequest
import media.ProducerKind
import media.RenegotiateRequest
import media.SessionDescription
import media.Target
import media.edge.*

class DyteHive
internal constructor(
  private val data: DyteParticipants,
  controllerContainer: IControllerContainer,
  private val socketHandler: HiveSFUSocketHandler,
  private val callStatsClient: CallStats
) : IDyteHive {
  @OptIn(DelicateCoroutinesApi::class, ExperimentalCoroutinesApi::class)
  private val coroutineScope = CoroutineScope(newSingleThreadContext("DyteHive"))
  private val defScope = CoroutineScope(Dispatchers.Default)

  private var _device: HiveDevice? = null

  private var _sendTransport: HiveTransport? = null

  private var _recvTransport: HiveTransport? = null

  private val _consumers = mutableMapOf<String, HiveConsumer>()

  private val _producers = mutableMapOf<String, HiveProducer>()

  override val producerIdToConsumerIdMap = mutableMapOf<String, String>()

  private val consumersToParticipants = HashMap<String, DyteJoinedMeetingParticipant>()

  private var _controllerContainer = controllerContainer

  private var localAudioTrack: AudioStreamTrack? = null

  private var localVideoTrack: VideoStreamTrack? = null

  private val observer = MutableSharedFlow<HiveEmitData>()

  override fun init() {
    val hiveLogger = HiveLogger()
    _controllerContainer.platformUtilsProvider
      .getMediaUtils()
      .setPlatform(_controllerContainer.platformUtilsProvider.getPlatformUtils().getOsName())

    _device = HiveDevice(HiveDeviceOptions(), coroutineScope, hiveLogger)

    startCallStats()

    val meetingConfig = _controllerContainer.metaController.getMeetingConfig()

    if (
      _controllerContainer.permissionController.isPermissionGrated(PermissionType.CAMERA) &&
        _controllerContainer.selfController.canPublishVideo() &&
        meetingConfig.enableVideo
    ) {
      DyteLogger.info("DyteHive: init: creating video track")
      localVideoTrack = _controllerContainer.platformUtilsProvider.getMediaUtils().getVideoTrack()
      _controllerContainer.selfController.getSelf()._videoTrack = localVideoTrack
      _controllerContainer.selfController.getSelf()._videoEnabled = true
    } else {
      DyteLogger.info("DyteHive: init with no camera")
    }

    if (
      _controllerContainer.permissionController.isPermissionGrated(PermissionType.MICROPHONE) &&
        _controllerContainer.selfController.canPublishAudio() &&
        meetingConfig.enableAudio
    ) {
      DyteLogger.info("DyteHive: init: creating audio track")
      localAudioTrack =
        _controllerContainer.platformUtilsProvider.getMediaUtils().createAudioTrack()
      _controllerContainer.selfController.getSelf()._audioEnabled = true
    } else {
      DyteLogger.info("DyteHive: init with no microphone")
    }
  }

  private fun startCallStats() {
    callStatsClient.authenticate(AuthPayload(_controllerContainer.metaController.getPeerId()))

    callStatsClient.startPingStats(interval = 7000)
  }

  override fun getObserverFlow() = observer

  private fun getProducers() = this._producers

  override fun getConsumers() = this._consumers

  private fun stringToSessionDescriptionType(str: String): SessionDescriptionType? {
    when (str.lowercase()) {
      "offer" -> return SessionDescriptionType.Offer
      "pranswer" -> return SessionDescriptionType.Pranswer
      "answer" -> return SessionDescriptionType.Answer
      "rollback" -> return SessionDescriptionType.Rollback
    }

    return null
  }

  override suspend fun setupTransports(iceServers: List<IceServerData>?) {
    val iceServersData = iceServers ?: _controllerContainer.apiClient.getICEServers().iceServers

    this.createWebRtcTransportProd(model = null, iceServers = iceServersData)
    this.createWebRtcTransportRecv(model = null, iceServers = iceServersData)
  }

  private suspend fun negotiate(
    transport: HiveTransport,
    offer: io.dyte.webrtc.SessionDescription
  ): io.dyte.webrtc.SessionDescription {
    val answer = transport.setRemoteOffer(offer)

    val regReq =
      RenegotiateRequest(
        transport_id = transport.getId()!!,
        description =
          SessionDescription(
            sdp = answer.sdp,
            type = answer.type.toString().lowercase(),
            target = Target.SUBSCRIBER
          )
      )

    socketHandler.renegotitateSessionDescription(regReq)

    return answer
  }

  private suspend fun handleTransport(transport: HiveTransport, consuming: Boolean) {
    transport.observer
      .onEach {
        when (it.eventName) {
          "connect" -> {
            val offer = it.data as io.dyte.webrtc.SessionDescription

            val req =
              CreateTransportRequest(
                consuming = consuming,
                description =
                  SessionDescription(
                    sdp = offer.sdp,
                    type = offer.type.toString().lowercase(),
                    target = if (consuming) Target.SUBSCRIBER else Target.PUBLISHER
                  )
              )

            try {
              val responseDecoded = socketHandler.createTransport(req)!!

              transport.setServerId(responseDecoded.transport_id)

              val description = responseDecoded.description
              val answer =
                SessionDescription(
                  type = stringToSessionDescriptionType(description!!.type)!!,
                  sdp = description.sdp
                )

              transport.externalObserver.emit(
                HiveEmitData(eventName = "returnConnect", data = answer)
              )
            } catch (e: Exception) {
              DyteLogger.error("DyteHive: ${transport.getDirection()} connection error")
            }
          }
          "connectionstatechange" -> {
            when (it.data as HiveConnectionState) {
              HiveConnectionState.Failed -> {
                if (transport.getClosed()) return@onEach
                DyteLogger.warn("DyteHive: ${transport.getDirection()} transport failed")

                // Note(anunaym14): Temporarily disabling this as it is causing a race when self
                // is kicked out of the room.
                //                if (
                //                  _controllerContainer.featureFlagService.isFeatureEnabled(
                //
                // DyteFeatureFlags.ENABLE_HIVE_TRANSPORT_RECONNECTION_ON_ICE_FAILED
                //                  )
                //                ) {
                //                  observer.emit(HiveEmitData("reconnect_transport", transport))
                //                }
              }
              HiveConnectionState.Disconnected -> {
                if (transport.getClosed()) return@onEach
                DyteLogger.warn("DyteHive: ${transport.getDirection()} transport disconnected")

                //                if (
                //                  _controllerContainer.featureFlagService.isFeatureEnabled(
                //
                // DyteFeatureFlags.ENABLE_HIVE_TRANSPORT_RECONNECTION_ON_ICE_FAILED
                //                  )
                //                ) {
                //                  observer.emit(HiveEmitData("reconnect_transport", transport))
                //                }
              }
              else -> {}
            }
          }
          "icecandidate" -> {}
          "datachannel" -> {
            val data = it.data as Map<*, *>

            val channel = data["channel"] as DataChannel
            val msg = data["parsedMessage"] as DCMessage

            if (channel.label == "negotiation") {
              if (!consuming) {
                return@onEach
              }

              // TODO: Check what different thing is to be done
            }

            try {
              when (msg.type) {
                "offer" -> {
                  val offer =
                    SessionDescription(
                      type = SessionDescriptionType.Offer,
                      sdp = msg.payload["sdp"]?.content!!.replace("\\r\\n", "\n")
                    )

                  lateinit var resPayload: DCMessage

                  try {
                    val ans = transport.setRemoteOffer(offer)
                    resPayload =
                      DCMessage(
                        type = "answer",
                        payload =
                          mapOf(
                            "type" to JsonPrimitive(ans.type.toString().lowercase()),
                            "sdp" to JsonPrimitive(ans.sdp)
                          )
                      )
                  } catch (e: Exception) {
                    DyteLogger.warn("DyteHive: datachannel:events error: $e")
                    resPayload =
                      DCMessage(
                        type = "error",
                        payload = mapOf("error" to JsonPrimitive(e.message!!))
                      )
                  }

                  val resPayloadByteArray =
                    (Json.encodeToString(DCMessage.serializer(), resPayload)).toByteArray()
                  channel.send(resPayloadByteArray)
                }
                "consumer_toggle" -> {
                  val mute = msg.payload["mute"]?.content == "true"
                  val trackId = msg.payload["trackId"]?.content

                  DyteLogger.info("DyteHive: consumer_toggle: Id = $trackId, muted = $mute")

                  val consumer = this.getConsumers()[trackId]

                  if (consumer == null) {
                    DyteLogger.info("DyteHive: consumer_toggle: Id = $trackId not found")
                    return@onEach
                  }

                  if (consumer.getPaused() != mute) {
                    DyteLogger.info("DyteHive: consumer_toggle: state is not same")

                    if (mute) {
                      handlePauseConsumer(consumer.getId())
                    } else {
                      handleResumeConsumer(consumer.getId())
                    }
                  }
                }
                else ->
                  DyteLogger.warn(
                    "DyteHive: Unknown datachannel event received from hive node:" + " ${msg.type}"
                  )
              }
            } catch (e: Exception) {
              DyteLogger.warn(
                "DyteHive: Unable to handle the incoming datachannel message on " + channel.label
              )
            }
          }
          "negotiate" -> {
            val offer = it.data as io.dyte.webrtc.SessionDescription

            val answer = negotiate(transport, offer)

            transport.externalObserver.emit(
              HiveEmitData(eventName = "returnNegotiate", data = answer)
            )
          }
          "close" -> {
            if (consuming) {
              this._consumers.values.forEach { consumer ->
                callStatsClient.registerConsumerListenerOnTransportClose(
                  ConsumerFacade(isHive = true, hiveConsumer = consumer)
                )
              }

              this._recvTransport = null
              return@onEach
            }

            this._producers.values.forEach { producer ->
              callStatsClient.registerProducerListenerOnTransportClose(
                ProducerFacade(isHive = true, hiveProducer = producer)
              )
            }
            this._sendTransport = null

            DyteLogger.info(
              "DyteHive: ${transport.getDirection()} with id = " + "${transport.getId()} closed"
            )
          }
          "produce" -> {
            if (consuming) {
              return@onEach
            }

            val data = it.data as Map<*, *>

            val offer = data["offer"] as io.dyte.webrtc.SessionDescription
            val kind = data["kind"] as MediaStreamTrackKind
            val paused = data["paused"] as Boolean
            val appData = data["appData"] as Map<*, *>?

            val parsedSdp = SDPUtils.parse(offer.sdp)
            val msid =
              parsedSdp.media
                .last { media ->
                  if (kind == MediaStreamTrackKind.Video) {
                    media.type == "video"
                  } else {
                    media.type == "audio"
                  }
                }
                .msid

            val req =
              ProducerCreateRequest(
                description =
                  SessionDescription(
                    sdp = offer.sdp,
                    type = offer.type.toString().lowercase(),
                    target = Target.PUBLISHER
                  ),
                paused = paused,
                kind = kind.toString().lowercase(),
                msid = msid!!,
                screen_share = appData?.get("screenShare") as? Boolean ?: false
              )

            try {
              val responseData = socketHandler.produce(req)!!

              val description = responseData.description
              val answer =
                SessionDescription(
                  type = stringToSessionDescriptionType(description!!.type)!!,
                  sdp = description.sdp
                )

              transport.externalObserver.emit(
                HiveEmitData(
                  eventName = "returnProduce",
                  data = mapOf("answer" to answer, "producerId" to responseData.producer_id)
                )
              )
            } catch (e: Exception) {
              DyteLogger.error("DyteHive: Create Producer Error: $e")
            }
          }
          "consumePeer" -> {
            val producingPeerId = it.data as String

            val req = ConsumePeerRequest(producing_peer_id = producingPeerId)

            try {
              val consumerStateMap = socketHandler.consume(req)!!.consumer_ids_map!!.map

              val consumersMap = mutableMapOf<String, HiveConsumerStateObject>()

              consumerStateMap.forEach { entry ->
                consumersMap[entry.key] =
                  HiveConsumerStateObject(
                    consumerId = entry.value.consumer_id,
                    trackId = entry.value.producer_track!!.track_id,
                    streamId = entry.value.producer_track!!.stream_id,
                    kind =
                      if (entry.value.producer_state!!.kind == ProducerKind.VIDEO)
                        MediaStreamTrackKind.Video
                      else MediaStreamTrackKind.Audio,
                    screenShare = entry.value.producer_state!!.screen_share,
                    paused = entry.value.producer_state!!.pause
                  )
              }

              transport.externalObserver.emit(
                HiveEmitData(eventName = "returnConsumePeer", data = consumersMap)
              )
            } catch (e: Exception) {
              DyteLogger.error("DyteHive: consumePeer error: $e")
            }
          }
          "consume" -> {
            val data = it.data as Map<*, *>

            val producerId = data["producerId"] as String
            val producingPeerId = data["producingPeerId"] as String

            val req =
              ConsumePeerRequest(producing_peer_id = producingPeerId, producer_id = producerId)

            try {
              val consumerState = socketHandler.consume(req)!!.consumer_ids_map!!.map[producerId]

              val producerState = consumerState?.producer_state
              val producerTrack = consumerState?.producer_track

              transport.externalObserver.emit(
                HiveEmitData(
                  eventName = "returnConsume",
                  data =
                    HiveConsumerStateObject(
                      consumerId = consumerState!!.consumer_id,
                      screenShare = producerState!!.screen_share,
                      trackId = producerTrack!!.track_id,
                      streamId = producerTrack.stream_id,
                      kind =
                        if (producerState.kind == ProducerKind.VIDEO) MediaStreamTrackKind.Video
                        else MediaStreamTrackKind.Audio,
                      paused = producerState.pause
                    )
                )
              )
            } catch (e: Exception) {
              DyteLogger.error("DyteHive: Error during consuming on server", e)
            }
          }
        }
      }
      .launchIn(defScope)
  }

  override suspend fun createWebRtcTransportRecv(
    model: WebRtcCreateTransportModel?,
    iceServers: List<IceServerData>,
  ) {
    if (this._recvTransport != null && this._recvTransport!!.getConnected()) return

    val hiveIceServers = mutableListOf<IceServer>()

    iceServers
      .filter { it.username?.isNotBlank() ?: false }
      .forEach {
        hiveIceServers.add(
          IceServer(urls = listOf(it.url), username = it.username!!, password = it.credential!!)
        )
      }

    try {
      val transport =
        _device!!.createRecvTransport(HiveTransportOptions(iceServers = hiveIceServers))

      transport.init()

      handleTransport(transport, true)

      transport.connect()

      callStatsClient.registerConsumingTransport(
        TransportFacade(isHive = true, hiveTransport = transport)
      )
      _recvTransport = transport

      this.producerIdToConsumerIdMap.clear()
    } catch (e: Exception) {
      DyteLogger.error("DyteHive: Error on creating recv transport: $e")
    }

    DyteLogger.info("DyteHive: Recv Transport created with ID = ${_recvTransport?.getId()}")
  }

  override suspend fun createWebRtcTransportProd(
    model: WebRtcCreateTransportModel?,
    iceServers: List<IceServerData>,
  ) {
    if (this._sendTransport != null && this._sendTransport!!.getConnected()) return

    val hiveIceServers = mutableListOf<IceServer>()

    iceServers
      .filter { it.username?.isNotBlank() ?: false }
      .forEach {
        hiveIceServers.add(
          IceServer(urls = listOf(it.url), username = it.username!!, password = it.credential!!)
        )
      }

    try {
      val transport =
        _device!!.createSendTransport(HiveTransportOptions(iceServers = hiveIceServers))

      transport.init()

      handleTransport(transport, false)

      transport.connect()

      callStatsClient.registerProducingTransport(
        TransportFacade(isHive = true, hiveTransport = transport)
      )

      _sendTransport = transport
    } catch (e: Exception) {
      DyteLogger.error("DyteHive: Error on creating send transport: $e")
    }

    DyteLogger.info("DyteHive: Send transport created with ID = ${_sendTransport?.getId()}")
  }

  override fun handlePauseConsumer(id: String) {
    _controllerContainer.platformUtilsProvider.getPlatformUtils().runOnMainThread {
      val consumer =
        _consumers[id]
          ?: kotlin.run {
            DyteLogger.warn("DyteHive::handlePauseConsumer::No consumer found for $id")
            return@runOnMainThread
          }

      val participant =
        consumersToParticipants[id]
          ?: kotlin.run {
            DyteLogger.warn(
              "DyteHive::handlePauseConsumer::No participant found for consumer id $id"
            )
            return@runOnMainThread
          }

      val isAudioTrack = consumer.getTrack() is AudioStreamTrack

      try {
        val req = ConsumerToggleRequest(consumer_id = id, pause = true)

        coroutineScope.launch { socketHandler.toggleConsumer(req) }
      } catch (e: Exception) {
        DyteLogger.error("DyteHive: Error on pausing consumer $id: $e")
      }

      coroutineScope.launch { consumer.pause() }
      if (isAudioTrack) {
        // Audio consumers may be paused even if the audio is being produced at the other end
        // so do not set the state as audioEnabled = false
      } else {
        val appData = consumer.getAppData()
        if (appData?.get("screenShare") == true) {
          val screenShareParticipant =
            data.joined.find { ss -> ss.id == participant.id } ?: return@runOnMainThread
          _controllerContainer.participantController.onPeerScreenSharedEnded(screenShareParticipant)
        } else {
          _controllerContainer.participantController.onParticipantVideoMuted(participant)
        }
      }
    }
  }

  override suspend fun cleanupTransport() {
    this._sendTransport?.close()
    this._recvTransport?.close()

    this._sendTransport = null
    this._recvTransport = null

    DyteLogger.info("DyteHive: Transports are closed")
  }

  override fun handleResumeConsumer(id: String) {
    _controllerContainer.platformUtilsProvider.getPlatformUtils().runOnMainThread {
      val consumer =
        _consumers[id]
          ?: kotlin.run {
            DyteLogger.warn("DyteHive::handleResumeConsumer::No consumer found for $id")
            return@runOnMainThread
          }

      val participant =
        consumersToParticipants[id]
          ?: kotlin.run {
            DyteLogger.warn(
              "DyteHive::handleResumeConsumer::No participant found for consumer id $id"
            )
            return@runOnMainThread
          }

      val isVideoTrack = consumer.getTrack() is VideoStreamTrack

      try {
        val req = ConsumerToggleRequest(consumer_id = id, pause = false)

        coroutineScope.launch { socketHandler.toggleConsumer(req) }
      } catch (e: Exception) {
        DyteLogger.error("DyteHive: Error on resuming consumer $id: $e")
      }

      coroutineScope.launch { consumer.resume() }

      if (isVideoTrack) {
        val appData = consumer.getAppData()
        if (appData?.get("screenShare") == true) {
          val screenShareParticipant =
            data.joined.find { ss -> ss.id == participant.id } ?: return@runOnMainThread
          screenShareParticipant._screenShareTrack = consumer.getTrack() as VideoStreamTrack
          _controllerContainer.participantController.onPeerScreenShareStarted(
            screenShareParticipant
          )
        } else {
          _controllerContainer.participantController.onParticipantVideoUnmuted(
            participant,
            consumer.getTrack() as VideoStreamTrack
          )
        }
      }
    }
  }

  private suspend fun handleProducer(
    producerType: ProducerType,
    producer: HiveProducer,
    onDisconnect: () -> Unit,
  ) {
    producer.observer.collect {
      when (it.eventName) {
        "close" -> {
          if (it.data != null) {
            val data = it.data as Map<*, *>

            val offer = data["offerSdp"] as io.dyte.webrtc.SessionDescription
            @Suppress("UNCHECKED_CAST")
            val cb = data["callback"] as suspend (answer: io.dyte.webrtc.SessionDescription) -> Any
            val reason = data["reason"] as? String ?: "Reason not provided peepoSad"

            DyteLogger.info(
              "DyteHive: Producer Id = ${producer.getId()} closed with " + "reason: $reason"
            )

            val req =
              ProducerCloseRequest(
                producer_id = producer.getId(),
                description =
                  SessionDescription(
                    sdp = offer.sdp,
                    type = offer.type.toString().lowercase(),
                    target = Target.PUBLISHER
                  )
              )

            try {
              val description = socketHandler.closeProducer(req)!!.description

              val answer =
                SessionDescription(
                  type = stringToSessionDescriptionType(description!!.type)!!,
                  sdp = description.sdp
                )

              cb.invoke(answer)
            } catch (e: Exception) {
              DyteLogger.error("DyteHive: Producer close error: $e")
            }
          }

          this._producers.remove(producerType.toString())
          onDisconnect.invoke()

          // TODO: Fix setMaxSpatialLayer
          //          if (producerType == ProducerType.SCREENSHARE_VIDEO) {
          //            val camProducer = this.getProducers()[ProducerType.CAM.toString()]
          //
          //            camProducer?.let {
          //              try {
          //                it.setMaxSpatialLayer(3)
          //              } catch (e: Error) {
          //                DyteLogger.info("HiveDebug: Failed to switch spatial layer: $e")
          //              }
          //            }
          //          }
        }
        "trackended" -> {
          DyteLogger.info("DyteHive: Producer Trackended for $producerType")

          if (!(producerType == ProducerType.CAM || producerType == ProducerType.MIC)) {
            onDisconnect.invoke()
          }
        }
      }
    }
  }

  private suspend fun createProducer(
    producerType: ProducerType,
    producerOptions: HiveProducerOptions,
    onDisconnect: () -> Unit,
  ) {
    if (this._sendTransport == null || this._sendTransport!!.getClosed()) {
      return
    }

    try {
      val producer = this._sendTransport!!.produce(producerOptions)

      coroutineScope.launch { handleProducer(producerType, producer, onDisconnect) }

      this._producers[producerType.toString()] = producer

      callStatsClient.registerProducer(ProducerFacade(isHive = true, hiveProducer = producer))
    } catch (e: Exception) {
      DyteLogger.error("DyteHive: Create Producer Error for $producerType: $e")
    }

    // TODO: Fix setMaxSpatialLayer
    //    if (this._producers.keys.contains(ProducerType.SCREENSHARE_VIDEO.toString())) {
    //      val camProducer = this.getProducers()[ProducerType.CAM.toString()]
    //
    //      camProducer?.let {
    //        try {
    //          it.setMaxSpatialLayer(0)
    //        } catch (e: Error) {
    //          DyteLogger.info("HiveDebug: Failed to switch spatial layer: $e")
    //        }
    //      }
    //    }
  }

  override suspend fun pauseProducer(type: String) {
    val producer = this._producers[type]

    if (producer == null) {
      DyteLogger.info("DyteHive: PauseProducer: Producer type = $type not found")
      return
    }

    producer.pause()
    DyteLogger.info("DyteHive: $type producer id = ${producer.getId()} paused")
  }

  override suspend fun resumeProducer(type: String) {
    val producer = this._producers[type]

    if (producer == null) {
      DyteLogger.info("DyteHive: ResumeProducer: Producer type = $type not found")
      return
    }

    producer.resume()
    DyteLogger.info("DyteHive: $type producer id = ${producer.getId()} resumed")
  }

  override fun removeProducer(type: String, stopTracks: Boolean?) {
    val producer = this._producers[type]

    if (producer == null) {
      DyteLogger.info("DyteHive: Producer type = $type not found")
      return
    }

    if (stopTracks != null) {
      if (stopTracks) {
        producer.getTrack().stop()
      }
    }

    coroutineScope.launch { producer.close(null) }
  }

  override suspend fun stopAllProducers() {
    this._producers.forEach {
      DyteLogger.info("DyteHive: StopAllProducers: Closing producer type = ${it.key}")
      it.value.close(null)
    }
  }

  override suspend fun resetVideoProducers(
    videoTrack: MediaStreamTrack?,
    screenShareTrack: MediaStreamTrack?,
  ) {
    if (videoTrack != null) {
      this.removeProducer(ProducerType.CAM.toString(), false)
      this.shareCam(videoTrack)
    }

    if (screenShareTrack != null) {
      this.removeProducer(ProducerType.SCREENSHARE_VIDEO.toString(), false)
      this.shareScreen(screenShareTrack, null)
    }
  }

  override suspend fun pauseCam() {
    val camProducer = this.getProducers()[ProducerType.CAM.toString()]

    if (camProducer == null) {
      DyteLogger.info("DyteHive: PauseCam: could not find cam producer")
      return
    }

    camProducer.pause()

    val req = ProducerToggleRequest(producer_id = camProducer.getId(), pause = true)

    socketHandler.toggleProducer(req)
  }

  override suspend fun resumeCam() {
    val camProducer = this.getProducers()[ProducerType.CAM.toString()]

    if (camProducer == null) {
      DyteLogger.info("DyteHive: ResumeCam: could not find cam producer")
      return
    }

    camProducer.resume()

    val req = ProducerToggleRequest(producer_id = camProducer.getId(), pause = false)

    socketHandler.toggleProducer(req)
  }

  override suspend fun pauseMic() {
    val micProducer = this.getProducers()[ProducerType.MIC.toString()]

    if (micProducer == null) {
      DyteLogger.info("DyteHive: PauseMic: could not find mic producer")
      return
    }

    if (micProducer.getPaused()) {
      DyteLogger.info("DyteHive: PauseMic: mic producer already paused")
      return
    }

    micProducer.pause()

    val req = ProducerToggleRequest(producer_id = micProducer.getId(), pause = true)

    socketHandler.toggleProducer(req)
  }

  override suspend fun resumeMic() {
    val micProducer = this.getProducers()[ProducerType.MIC.toString()]

    if (micProducer == null) {
      DyteLogger.info("DyteHive: ResumeMic: could not find mic producer")
      return
    }

    if (!micProducer.getPaused()) {
      DyteLogger.info("DyteHive: ResumeMic: mic producer already resumed")
      return
    }

    micProducer.resume()

    val req = ProducerToggleRequest(producer_id = micProducer.getId(), pause = false)

    socketHandler.toggleProducer(req)
  }

  override fun disableCam() {
    this.removeProducer(ProducerType.CAM.toString(), null)
  }

  override fun disableMic() {
    this.removeProducer(ProducerType.MIC.toString(), null)
  }

  override fun disableScreenShare() {
    this.removeProducer(ProducerType.SCREENSHARE_VIDEO.toString(), null)
    this.removeProducer(ProducerType.SCREENSHARE_AUDIO.toString(), null)
  }

  override suspend fun replaceTrack(type: String, track: MediaStreamTrack) {
    val producer = this._producers[type]

    if (producer == null) {
      DyteLogger.info("DyteHive: ReplaceTrack: Producer type = $type not found")
      return
    }

    producer.replaceTrack(track)
    DyteLogger.info("DyteHive: Track replaced with id = ${track.id} on producer $type")
  }

  private fun handleConsumer(consumer: HiveConsumer) {
    consumer.observer
      .onEach {
        when (it.eventName) {
          "close" -> {
            handleCloseConsumer(consumer.getId())
          }
        }
      }
      .launchIn(coroutineScope)
  }

  private fun handleCloseConsumer(consumerId: String) {
    val consumer =
      _consumers[consumerId]
        ?: kotlin.run {
          DyteLogger.warn("DyteHive::handleCloseConsumer::No consumer found for $consumerId")
          return
        }

    val appData = consumer.getAppData()
    val participant =
      consumersToParticipants[consumerId]
        ?: kotlin.run {
          DyteLogger.warn(
            "DyteHive::handleCloseConsumer::No participant found for consumer id $consumerId"
          )
          return
        }
    if (appData?.get("screenShare") == true) {
      if (consumer.getKind() == MediaStreamTrackKind.Video) {
        val screenShareParticipant =
          _controllerContainer.participants.joined.find { ss -> ss.id == participant.id } ?: return
        screenShareParticipant._screenShareTrack = null
        _controllerContainer.participantController.onPeerScreenSharedEnded(screenShareParticipant)
      }
    } else if (consumer.getKind() == MediaStreamTrackKind.Video) {
      _controllerContainer.participantController.onParticipantVideoMuted(participant)
    } else {
      _controllerContainer.participantController.onPeerAudioMuted(
        WebSocketPeerMuteModel(participant.id)
      )
    }
    _consumers.remove(consumer.getId())
    consumersToParticipants.remove(consumerId)
  }

  private fun initConsumer(consumer: HiveConsumer?) {
    if (consumer == null) return

    callStatsClient.registerConsumer(ConsumerFacade(isHive = true, hiveConsumer = consumer))

    handleConsumer(consumer)

    this._consumers[consumer.getId()] = consumer
    this.producerIdToConsumerIdMap[consumer.getProducerId()] = consumer.getId()
  }

  override suspend fun consumePeer(producingPeerId: String) {
    if (this._recvTransport == null || this._recvTransport!!.getClosed())
      throw Error("Receiving transport not connected")

    val results = mutableListOf<CompletableDeferred<HiveConsumer>>()

    try {
      results.addAll(this._recvTransport!!.consumePeer(producingPeerId, emptyMap()))
    } catch (e: Exception) {
      DyteLogger.error("DyteHive: Error on consuming peer", e)
    }

    val failedTasks = mutableListOf<HiveConsumerCreationTaskException>()

    results.forEach { deferredConsumer ->
      try {
        val consumer = deferredConsumer.await()

        val joinedParticipant =
          data.joined.find { participant -> participant.id == consumer.getPeerId() }

        joinedParticipant?.let { participant ->
          if (consumer.getTrack().kind == MediaStreamTrackKind.Video) {
            if (consumer.getAppData()?.get("screenShare") as? Boolean == true) {
              val screenShareParticipant = data.joined.find { ss -> ss.id == participant.id }

              screenShareParticipant?.let {
                it._screenShareTrack = consumer.getTrack() as VideoStreamTrack
                _controllerContainer.participantController.onPeerScreenShareStarted(
                  screenShareParticipant
                )
              }
            } else {
              participant._videoEnabled = !consumer.getPaused()
              participant._videoTrack = consumer.getTrack() as VideoStreamTrack
              if (participant._videoEnabled)
                _controllerContainer.participantController.onPeerVideoUnmuted(
                  WebSocketConsumerResumedModel(id = participant.id)
                )
              else
                _controllerContainer.participantController.onPeerVideoMuted(
                  WebSocketConsumerClosedModel(id = participant.id)
                )
            }
          } else {
            if (consumer.getAppData()?.get("screenShare") as? Boolean != true) {
              participant._audioEnabled = !consumer.getPaused()
              if (participant._audioEnabled)
                _controllerContainer.participantController.onPeerAudioUnmuted(
                  WebSocketPeerMuteModel(peerId = participant.id)
                )
              else
                _controllerContainer.participantController.onPeerAudioMuted(
                  WebSocketPeerMuteModel(peerId = participant.id)
                )
            }
          }

          consumersToParticipants.put(consumer.getId(), participant)
        }

        this.initConsumer(consumer)
      } catch (e: Exception) {
        if (e is HiveConsumerCreationTaskException) failedTasks.add(e)
        else DyteLogger.error("DyteHive: Unknown error on consume peer: $e")
      }
    }

    val retryConsumers = this._recvTransport!!.retryFailedConsumerCreationTasks(failedTasks)

    retryConsumers.forEach {
      try {
        val consumer = it.await()
        this.initConsumer(consumer)
      } catch (e: Exception) {
        if (e is HiveConsumerCreationTaskException)
          DyteLogger.error("DyteHive: Consumer creation failed second time")
        else DyteLogger.error("DyteHive: Unknown error on consume peer: $e")
      }
    }
  }

  override suspend fun handleNewConsumer(hiveConsumerOptions: HiveConsumerOptions) {
    if (this._recvTransport == null || this._recvTransport!!.getClosed()) return

    //        val producer  = this.getProducer(consumerOpts.producerId)
    //
    //        if (producer != null) {
    //            DyteLogger.info("HiveUtils: Why are you creating a consumer for local producer?")
    //            return
    //        }
    //
    //        DyteLogger.info("HiveDebug: producer = producer is null")

    try {
      val consumerDeferred = this._recvTransport!!.consume(hiveConsumerOptions)

      val consumer = consumerDeferred.await()

      val joinedParticipant = data.joined.find { it.id == consumer.getPeerId() }

      joinedParticipant?.let { participant ->
        if (consumer.getTrack().kind == MediaStreamTrackKind.Video) {
          if (hiveConsumerOptions.appData?.get("screenShare") as? Boolean == true) {
            val screenShareParticipant = data.joined.find { ss -> ss.id == participant.id }

            screenShareParticipant?.let {
              it._screenShareTrack = consumer.getTrack() as VideoStreamTrack
              _controllerContainer.participantController.onPeerScreenShareStarted(
                screenShareParticipant
              )
            }
          } else {
            participant._videoEnabled = hiveConsumerOptions.paused?.not() ?: true
            participant._videoTrack = consumer.getTrack() as VideoStreamTrack
            if (participant._videoEnabled)
              _controllerContainer.participantController.onPeerVideoUnmuted(
                WebSocketConsumerResumedModel(id = participant.id)
              )
            else
              _controllerContainer.participantController.onPeerVideoMuted(
                WebSocketConsumerClosedModel(id = participant.id)
              )
          }
        } else {
          if (hiveConsumerOptions.appData?.get("screenShare") as? Boolean != true) {
            participant._audioEnabled = hiveConsumerOptions.paused?.not() ?: true
            if (participant._audioEnabled)
              _controllerContainer.participantController.onPeerAudioUnmuted(
                WebSocketPeerMuteModel(peerId = participant.id)
              )
            else
              _controllerContainer.participantController.onPeerAudioMuted(
                WebSocketPeerMuteModel(peerId = participant.id)
              )
          }
        }

        consumersToParticipants.put(consumer.getId(), participant)
      }

      this.initConsumer(consumer)
      //      DyteLogger.info("Hivedebug: here")
      //
      //      if (consumer.getCompletionExceptionOrNull() == null) {
      //        DyteLogger.info("HiveDebug: consumer created")
      //        this._initConsumer(consumer.getCompleted())
      //      } else {
      //        DyteLogger.info("HiveDebug: consumer creation failed")
      //        throw consumer.getCompletionExceptionOrNull()!!
      //      }
    } catch (e: Exception) {
      if (e is HiveConsumerCreationTaskException) {
        val retryConsumer =
          this._recvTransport!!.retryFailedConsumerCreationTasks(listOf(e)).first()

        try {
          val consumer = retryConsumer.await()
          this.initConsumer(consumer)
        } catch (e: Exception) {
          if (e is HiveConsumerCreationTaskException)
            DyteLogger.error("DyteHive: Failed to complete consumer creation task again")
          else DyteLogger.error("DyteHive: Unknown error on creating consumer: $e")
        }
      } else {
        DyteLogger.error("DyteHive: Unknown error on creating consumer: $e")
      }
    }
  }

  override suspend fun closeConsumer(consumerId: String, force: Boolean?) {
    this.closeConsumers(listOf(consumerId), force)
  }

  override suspend fun closeConsumers(consumerIds: List<String>, force: Boolean?) {
    var closeProvidedConsumers = true
    val availableConsumerIds =
      consumerIds.filter { id ->
        val consumer = this._consumers[id]

        if (consumer == null) {
          DyteLogger.info("DyteHive: CloseConsumer: Consumer Id = $id not found")
          return@filter false
        }

        return@filter true
      }

    val closeConsumerMutex = Mutex()

    try {
      val req = ConsumerCloseRequest(consumer_ids = availableConsumerIds)

      closeConsumerMutex.withLock {
        // TODO(anunaym14): Revisit this
        socketHandler.closeConsumer(req)

        //        val response = _socket.requestResponse(
        //          event = SocketServiceUtils.MediaEvent.CLOSE_CONSUMER.id,
        //          payload = ConsumerCloseRequest.ADAPTER.encode(req)
        //        )
        //
        //        response?.let {
        //          val resp = ConsumerClosingResponse.ADAPTER.decode(it)
        //          DyteLogger.info("HiveUtils: Consumer close response for $resp")
        //        }
      }
      // FIXME(itzmanish): here we should get a list of closed consumers from server
      // because there is possibility that if the request somehow fails or server
      // is only able to close a subset of consumers then we should not close all the
      // consumers in the client side, Closing all the consumers in the client side
      // without confirming how many consumers actually closed will produce leak of media.
    } catch (e: Exception) {
      DyteLogger.info("DyteHive: Error on Closing Consumer: $e")
      closeProvidedConsumers = force ?: false
    }

    if (closeProvidedConsumers) {
      availableConsumerIds.forEach { id ->
        val consumer = this.getConsumers()[id]

        this.producerIdToConsumerIdMap.remove(consumer?.getProducerId())
        consumer?.close("Manually closed")
      }
    }
  }

  override suspend fun cleanupConsumers(peerId: String?) {
    val consumers = arrayListOf<String>()

    this._consumers.values.forEach { consumer ->
      if (peerId != null) {
        if (consumer.getPeerId() == peerId) consumers.add(consumer.getId())
      } else {
        consumers.add(consumer.getId())
      }
    }

    this.closeConsumers(consumers, true)
  }

  override suspend fun cleanupConsumers() {
    val localConsumers = ArrayList(_consumers.values)
    localConsumers.forEach { consumer ->
      if (!consumer.getClosed()) {
        consumer.close(REASON_DISCONNECTION_CLEANUP)
      }
    }
    _consumers.clear()
    consumersToParticipants.clear()
    producerIdToConsumerIdMap.clear()
    DyteLogger.info("DyteHive::cleanupConsumers")
  }

  override suspend fun cleanupProducers() {
    val localProducers = ArrayList(_producers.values)
    localProducers.forEach { producer ->
      if (!producer.getClosed()) {
        producer.close(REASON_DISCONNECTION_CLEANUP)
      }
    }
    _producers.clear()
    DyteLogger.info("DyteHive::cleanupProducers")
  }

  override suspend fun refreshTracks() {
    localAudioTrack = _controllerContainer.platformUtilsProvider.getMediaUtils().createAudioTrack()
    localVideoTrack = _controllerContainer.platformUtilsProvider.getMediaUtils().createVideoTrack()
  }

  override suspend fun shareCam(videoTrack: MediaStreamTrack?) {
    if (videoTrack == null) return

    if (this.getProducers().contains(ProducerType.CAM.toString())) {
      val camProducer = this.getProducers()[ProducerType.CAM.toString()]

      if (!camProducer?.getClosed()!!) {
        camProducer.replaceTrack(videoTrack)
        this.resumeCam()
        return
      }

      this.removeProducer(camProducer.getId(), null)
    }

    // TODO: Feature flag for Simulcast
    val simulcastEnbaled = false
    val simulcastEncodings = mutableListOf<CommonRtpEncodingParameters>()

    if (simulcastEnbaled) {
      // TODO: Fix hardcoded value for video track width
      var encoding = CommonRtpEncodingParameters()
      when (480) {
        320 -> {
          encoding.rid = "q"
          encoding.maxBitrateBps = 250_000
          encoding.maxFramerate = 30
          // Scalability mode

          simulcastEncodings.add(encoding)
        }
        640 -> {
          encoding.rid = "h"
          encoding.maxBitrateBps = 750_000
          encoding.maxFramerate = 30
          // Scalability mode

          simulcastEncodings.add(encoding)

          encoding = CommonRtpEncodingParameters()

          encoding.rid = "q"
          encoding.scaleResolutionDownBy = 2.0
          encoding.maxBitrateBps = 250_000
          encoding.maxFramerate = 30
          // Scalability mode

          simulcastEncodings.add(encoding)
        }
        1280 -> {
          encoding.rid = "f"
          encoding.maxBitrateBps = 1_500_000
          encoding.maxFramerate = 30
          // Scalability mode

          simulcastEncodings.add(encoding)

          encoding = CommonRtpEncodingParameters()

          encoding.rid = "h"
          encoding.scaleResolutionDownBy = 2.0
          encoding.maxBitrateBps = 750_000
          encoding.maxFramerate = 30
          // Scalability mode

          simulcastEncodings.add(encoding)

          encoding = CommonRtpEncodingParameters()

          encoding.rid = "q"
          encoding.scaleResolutionDownBy = 4.0
          encoding.maxBitrateBps = 250_000
          encoding.maxFramerate = 30
          // Scalability mode

          simulcastEncodings.add(encoding)
        }
        else -> {
          encoding.rid = "q"
          encoding.maxBitrateBps = 250_000
          encoding.maxFramerate = 30
          // Scalability mode

          simulcastEncodings.add(encoding)
        }
      }
    }

    val nonSimulcastEncodings = CommonRtpEncodingParameters()
    nonSimulcastEncodings.active = true
    nonSimulcastEncodings.scaleResolutionDownBy = 1.0

    val producerOptions =
      HiveProducerOptions(
        track = videoTrack,
        codecOptions = HiveCodecOptions(name = "VP8", parameters = null),
        appData = mapOf("screenShare" to false),
        stopTracks = false,
        disableTrackOnPause = null,
        encodings = if (simulcastEnbaled) simulcastEncodings else listOf(nonSimulcastEncodings),
        zeroRtpOnPause = null,
        stream = _controllerContainer.platformUtilsProvider.getMediaUtils().getVideoStream()!!
      )

    val onDisconnect: () -> Unit = { runBlocking { disableCam() } }

    this.createProducer(ProducerType.CAM, producerOptions, onDisconnect)

    _controllerContainer.selfController.getSelf()._videoEnabled = true
    _controllerContainer.selfController.getSelf()._videoTrack = localVideoTrack
    _controllerContainer.selfController.emitEvent { it.onVideoUpdate(true) }
  }

  override suspend fun shareScreen(videoTrack: MediaStreamTrack?, audioTrack: MediaStreamTrack?) {
    if (videoTrack == null) return

    val videoProducerOptions =
      HiveProducerOptions(
        track = videoTrack,
        codecOptions = HiveCodecOptions(name = "VP8", parameters = null),
        appData =
          mapOf(
            "screenShare" to true,
            // "supportsRemoteControl"
          ),
        stopTracks = false,
        disableTrackOnPause = null,
        encodings = null,
        zeroRtpOnPause = null,
        stream = _controllerContainer.platformUtilsProvider.getMediaUtils().getVideoStream()!!
      )

    val videoOnDisconnect: () -> Unit = { runBlocking { disableScreenShare() } }

    this.createProducer(ProducerType.SCREENSHARE_VIDEO, videoProducerOptions, videoOnDisconnect)

    if (audioTrack != null) {
      val encoding = CommonRtpEncodingParameters()
      encoding.bitratePriority = 4.0

      val audioProducerOptions =
        HiveProducerOptions(
          track = audioTrack,
          codecOptions = HiveCodecOptions(name = "opus", parameters = null),
          appData =
            mapOf(
              "screenShare" to true,
              // "supportsRemoteControl"
            ),
          stopTracks = false,
          disableTrackOnPause = null,
          encodings = listOf(encoding),
          zeroRtpOnPause = null,
          stream = _controllerContainer.platformUtilsProvider.getMediaUtils().getAudioStream()!!
        )

      val audioOnDisconnect: () -> Unit =
        {
          // Do nothing as video will already stop the track
        }

      this.createProducer(ProducerType.SCREENSHARE_AUDIO, audioProducerOptions, audioOnDisconnect)
    }
  }

  override suspend fun shareMic(audioTrack: MediaStreamTrack?) {
    if (audioTrack == null) return

    if (this.getProducers().contains(ProducerType.MIC.toString())) {
      DyteLogger.info("DyteHive: Mic producer already exists")
      val micProducer = this.getProducers()[ProducerType.MIC.toString()]

      if (!micProducer?.getClosed()!!) {
        micProducer.replaceTrack(audioTrack)
        this.resumeMic()
        return
      }
      this.removeProducer(ProducerType.MIC.toString(), false)
    }

    val producerOptions =
      HiveProducerOptions(
        track = audioTrack,
        codecOptions = HiveCodecOptions(name = "opus", parameters = null),
        appData = null,
        stopTracks = false,
        disableTrackOnPause = null,
        encodings = null,
        zeroRtpOnPause = null,
        stream = _controllerContainer.platformUtilsProvider.getMediaUtils().getAudioStream()!!
      )

    val onDisconnect: () -> Unit = { runBlocking { disableMic() } }

    this.createProducer(ProducerType.MIC, producerOptions, onDisconnect)
    DyteLogger.info("DyteHive: Successfully created mic producer")
  }

  override fun getSelfTrack(): Pair<AudioStreamTrack?, VideoStreamTrack?> {
    return Pair(localAudioTrack, localVideoTrack)
  }

  override suspend fun muteSelfAudio(): Boolean {
    try {
      if (_controllerContainer.selfController._roomJoined) {
        pauseMic()
        callStatsClient.sendAudioToggleEvent(false)
      } else {
        DyteLogger.info("DyteHive: Room not joined, not pausing mic producer")
      }

      _controllerContainer.selfController.getSelf()._audioEnabled = false
      _controllerContainer.selfController.emitEvent { it.onAudioUpdate(false) }
      _controllerContainer.participantController.onAudioUpdate(
        _controllerContainer.selfController.getSelf()
      )
    } catch (e: Exception) {
      DyteLogger.error("DyteHive: Error on muting self audio", e)
      return false
    }

    return true
  }

  override suspend fun unmuteSelfAudio(): Boolean {
    try {
      if (localAudioTrack == null)
        localAudioTrack =
          _controllerContainer.platformUtilsProvider.getMediaUtils().createAudioTrack()

      if (_controllerContainer.selfController._roomJoined) {
        if (!_producers.contains(ProducerType.MIC.toString())) {
          coroutineScope.launch { shareMic(localAudioTrack) }
        } else {
          resumeMic()
        }

        callStatsClient.sendAudioToggleEvent(true)
      } else {
        DyteLogger.info("DyteHive: Room not joined, not creating/resuming mic producer")
      }

      _controllerContainer.selfController.getSelf()._audioEnabled = true
      _controllerContainer.selfController.emitEvent { it.onAudioUpdate(true) }
      return true
    } catch (e: Exception) {
      DyteLogger.error("DyteHive: Error on unmuting self audio", e)
      return false
    }
  }

  override suspend fun muteSelfVideo(): Boolean {
    try {
      if (_controllerContainer.selfController._roomJoined) {
        pauseCam()
        callStatsClient.sendVideoToggleEvent(false)
      } else {
        DyteLogger.info("DyteHive: Room not joined, not pausing cam producer")
      }

      _controllerContainer.selfController.getSelf()._videoEnabled = false
      _controllerContainer.selfController.emitEvent { it.onVideoUpdate(false) }
    } catch (e: Exception) {
      DyteLogger.error("DyteHive: Error on muting self video", e)
      return false
    }

    return true
  }

  override suspend fun unmuteSelfVideo(): Boolean {
    try {
      if (localVideoTrack == null)
        localVideoTrack =
          _controllerContainer.platformUtilsProvider.getMediaUtils().createVideoTrack()

      if (_controllerContainer.selfController._roomJoined) {
        if (!_producers.contains(ProducerType.CAM.toString())) {
          coroutineScope.launch { shareCam(localVideoTrack) }
        } else {
          resumeCam()
        }

        callStatsClient.sendVideoToggleEvent(true)
      } else {
        DyteLogger.info("DyteHive: Room not joined, not creating/resuming cam producer")
      }

      _controllerContainer.selfController.getSelf()._videoEnabled = true
      _controllerContainer.selfController.getSelf()._videoTrack = localVideoTrack
      _controllerContainer.selfController.emitEvent { it.onVideoUpdate(true) }
      return true
    } catch (e: Exception) {
      DyteLogger.error("DyteHive: Error on unmuting self video", e)
      return false
    }
  }

  override suspend fun leaveCall() {
    muteSelfAudio()
    muteSelfVideo()

    cleanupTransport()
    callStatsClient.stopPingStats()
    callStatsClient.sendDisconnectEvent()

    val req = PeerLeaveRequest(close_room = false)

    try {
      val closed = socketHandler.leaveRoom(req)?.closed

      if (closed != true) DyteLogger.warn("DyteHive: Weird state on peer closed, should not happen")
    } catch (e: Exception) {
      DyteLogger.error("DyteHive: Error on sending leave room request: $e")
    }
  }

  override fun switchCamera() {
    val camProducer = this.getProducers()[ProducerType.CAM.toString()]

    if (camProducer == null) {
      DyteLogger.info("DyteHive: SwitchCam: could not find cam producer")
      return
    }

    runBlocking { (camProducer.getTrack() as VideoStreamTrack).switchCamera() }
  }

  override suspend fun produceMedia() {
    TODO("Not yet implemented")
  }
}

interface IDyteHive : IDyteSFUUtils {

  val producerIdToConsumerIdMap: MutableMap<String, String>

  fun init()

  suspend fun setupTransports(iceServers: List<IceServerData>? = null)

  suspend fun createWebRtcTransportRecv(
    model: WebRtcCreateTransportModel?,
    iceServers: List<IceServerData>,
  )

  suspend fun createWebRtcTransportProd(
    model: WebRtcCreateTransportModel?,
    iceServers: List<IceServerData>,
  )

  suspend fun pauseProducer(type: String)

  suspend fun resumeProducer(type: String)

  fun removeProducer(type: String, stopTracks: Boolean?)

  suspend fun stopAllProducers()

  suspend fun resetVideoProducers(
    videoTrack: MediaStreamTrack?,
    screenShareTrack: MediaStreamTrack?,
  )

  suspend fun pauseCam()

  suspend fun resumeCam()

  suspend fun pauseMic()

  suspend fun resumeMic()

  fun disableCam()

  fun disableMic()

  fun disableScreenShare()

  suspend fun replaceTrack(type: String, track: MediaStreamTrack)

  suspend fun closeConsumers(consumerIds: List<String>, force: Boolean? = null)

  suspend fun cleanupConsumers(peerId: String?)

  suspend fun shareCam(videoTrack: MediaStreamTrack?)

  suspend fun shareScreen(videoTrack: MediaStreamTrack?, audioTrack: MediaStreamTrack?)

  suspend fun shareMic(audioTrack: MediaStreamTrack?)

  fun getSelfTrack(): Pair<AudioStreamTrack?, VideoStreamTrack?>

  suspend fun handleNewConsumer(hiveConsumerOptions: HiveConsumerOptions)

  fun getConsumers(): Map<String, HiveConsumer>

  suspend fun closeConsumer(consumerId: String, force: Boolean? = null)

  suspend fun consumePeer(producingPeerId: String)

  fun getObserverFlow(): MutableSharedFlow<HiveEmitData>
}

enum class ProducerType {
  CAM,
  MIC,
  SCREENSHARE_VIDEO,
  SCREENSHARE_AUDIO
}

class HiveLogger : IMediaClientLogger {
  override fun traceError(message: String) {
    DyteLogger.error(message)
  }

  override fun traceLog(message: String) {
    DyteLogger.info(message)
  }

  override fun traceWarning(message: String) {
    DyteLogger.warn(message)
  }
}
