package io.dyte.core.feat

import io.dyte.core.Result
import io.dyte.core.ThreadSafeMap
import io.dyte.core.network.models.PluginConfigResponse
import io.dyte.core.platform.DytePluginFile
import io.dyte.core.platform.IDytePlatformUtilsProvider
import io.dyte.core.platform.IDyteWebView
import io.dyte.core.platform.IDyteWebViewListener
import io.dyte.core.platform.WebView
import io.dyte.core.plugins.PluginFileProvider
import io.dyte.core.plugins.PluginHostMeeting
import io.dyte.core.plugins.PluginMessage
import io.dyte.core.plugins.PluginModel
import kotlinx.coroutines.*
import kotlinx.serialization.json.JsonObject

/**
 * Dyte Plugin
 *
 * @constructor Create un-activated Dyte Plugin
 * @property controllerContainer
 */
class DytePlugin
internal constructor(
  val id: String,
  val name: String,
  val description: String,
  val picture: String,
  val isPrivate: Boolean,
  val staggered: Boolean,
  val baseURL: String,
  private val scope: CoroutineScope,
  private val webViewProvider: IDytePlatformUtilsProvider,
  private val pluginModel: PluginModel
) {
  private var hostMeeting: PluginHostMeeting? = null
  private var fileProvider: PluginFileProvider? = null

  private var config: PluginConfigResponse? = null

  private val webViews: ThreadSafeMap<String, IDyteWebView> = ThreadSafeMap()

  val isActive: Boolean
    get() = runBlocking { withContext(scope.coroutineContext) { pluginModel.active } }

  val enabledBy: String?
    get() = runBlocking { withContext(scope.coroutineContext) { pluginModel.enabledBy } }

  private val webViewListener: IDyteWebViewListener =
    object : IDyteWebViewListener {
      override fun onWebViewMessage(message: JsonObject) {
        scope.launch { hostMeeting?.onPluginMessage(id, PluginMessage(message)) }
      }

      override fun onFileRequest() {
        scope.launch { fileProvider?.onFileRequest(this@DytePlugin) }
      }
    }

  /**
   * Submits the message to the Plugin SDK.
   *
   * @param message [PluginMessage] instance containing JSON payload
   */
  internal fun submitMessage(message: PluginMessage) {
    scope.launch(Dispatchers.Main) { webViews[KEY_DEFAULT_WEBVIEW]?.submitEvent(message.message) }
  }

  /** Activate this plugin for all participants. */
  fun activate() {
    scope.launch { hostMeeting?.activatePlugin(id, staggered) }
  }

  /** Deactivate this plugin for all participants. */
  fun deactivate() {
    scope.launch { hostMeeting?.deactivatePlugin(id) }
  }

  fun getPluginView(): WebView {
    return webViews
      .getOrElse(KEY_DEFAULT_WEBVIEW) {
        val webView = webViewProvider.createWebView()
        webView.initializeWebView()
        webView.setListener(webViewListener)
        pluginModel.urlToLoad?.let { webView.loadURL(it) }
        webViews[KEY_DEFAULT_WEBVIEW] = webView
        webViews.getValue(KEY_DEFAULT_WEBVIEW)
      }
      .getWebView()
  }

  private fun removePluginView() {
    webViews.remove(KEY_DEFAULT_WEBVIEW)
  }

  /** Uploads selected file to Plugin. */
  fun uploadFile(file: DytePluginFile) {
    scope.launch(Dispatchers.Main) { webViews[KEY_DEFAULT_WEBVIEW]?.uploadFile(file) }
  }

  /** Enables this plugin for the local user. */
  internal suspend fun enableLocal(enabledBy: String): Result<Unit, Exception> {
    return withContext(scope.coroutineContext) {
      val result = pluginModel.enableLocal(enabledBy)
      result
    }
  }

  /** Disables this plugin for the local user. */
  internal suspend fun disableLocal(): Result<Unit, Exception> {
    return withContext(scope.coroutineContext) {
      val result = pluginModel.disableLocal()
      withContext(Dispatchers.Main) { webViews[KEY_DEFAULT_WEBVIEW]?.loadURL(EMPTY_PAGE_URL) }
      webViews[KEY_DEFAULT_WEBVIEW]?.removeListener()
      removePluginView()
      result
    }
  }

  internal fun setHostMeeting(hostMeeting: PluginHostMeeting) {
    runBlocking(scope.coroutineContext) { this@DytePlugin.hostMeeting = hostMeeting }
  }

  internal fun setFileProvider(fileProvider: PluginFileProvider) {
    runBlocking(scope.coroutineContext) { this@DytePlugin.fileProvider = fileProvider }
  }

  /**
   * Clears the listeners to prevent a leak. Should be called when this DytePlugin is no longer used
   * and will be destroyed.
   *
   * For e.g. while leaving room
   */
  internal suspend fun clear() {
    withContext(scope.coroutineContext) {
      this@DytePlugin.hostMeeting = null
      this@DytePlugin.fileProvider = null
    }
  }

  companion object {
    private const val KEY_DEFAULT_WEBVIEW = "default"
    private const val EMPTY_PAGE_URL = "about:blank"
  }
}
