package io.dyte.core.controllers.stage

import io.dyte.core.controllers.DyteStageStatus
import io.dyte.core.controllers.ParticipantController
import io.dyte.core.controllers.SelfController
import io.dyte.core.events.EventEmitter
import io.dyte.core.events.InternalEvents
import io.dyte.core.host.IHostController
import io.dyte.core.models.DyteJoinedMeetingParticipant
import io.dyte.core.models.DyteMediaPermission
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.observability.DyteLogger
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketJoinRoomModel
import io.dyte.core.socket.socketservice.ISockratesSocketService
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import io.dyte.core.socket.socketservice.SocketServiceUtils
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import socket.livestreaming.DenyStageAccessRequest
import socket.livestreaming.GetStagePeersResponse
import socket.livestreaming.GetStageRequestsResponse
import socket.livestreaming.GrantStageAccessRequest
import socket.livestreaming.LeaveStageRequest
import socket.livestreaming.StageRequest
import socket.room.PeerStatusUpdate
import socket.room.StageType

internal class StageSocketServiceController(
  val self: DyteSelfParticipant,
  private val socketService: ISockratesSocketService,
  scope: CoroutineScope,
  private val participantController: ParticipantController,
  private val selfController: SelfController,
  private val hostController: IHostController,
  private val internalEventsEmitter: EventEmitter<InternalEvents>
) : BaseStageController(self, scope) {
  private val logger = DyteLogger

  // we dont show any viewers in case of livestream stage. this is only to support webinar
  override val viewers: List<DyteJoinedMeetingParticipant>
    get() = emptyList()

  private val socketEventListener =
    object : SocketServiceEventListener {
      override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
        scope.launch { handleStageSocketEvent(event, payload) }
      }
    }

  override fun setupEvents() {
    socketService.subscribe(
      SocketServiceUtils.RoomEvent.REQUEST_STAGE_ACCESS.id,
      socketEventListener
    )
    socketService.subscribe(
      SocketServiceUtils.RoomEvent.PEER_STAGE_STATUS_UPDATE.id,
      socketEventListener
    )

    if (self.permissions.host.canAcceptRequests) {
      socketService.subscribe(
        SocketServiceUtils.RoomEvent.GET_STAGE_REQUESTS.id,
        socketEventListener
      )
    }

    // When a participant joins or leave stage
    socketService.subscribe(SocketServiceUtils.RoomEvent.GET_STAGE_PEERS.id, socketEventListener)

    socketService.subscribe(
      SocketServiceUtils.RoomEvent.GRANT_STAGE_REQUEST.id,
      socketEventListener
    )

    socketService.subscribe(SocketServiceUtils.RoomEvent.DENY_STAGE_REQUEST.id, socketEventListener)
  }

  override fun onRoomJoined(webSocketJoinRoomModel: WebSocketJoinRoomModel) {
    self._stageStatus = DyteStageStatus.fromWebsocketStageStatus(webSocketJoinRoomModel.stageStatus)

    if (self.permissions.host.canAcceptRequests) {
      runBlocking(scope.coroutineContext) {
        val socketResponse =
          try {
            socketService.requestResponse(
              event = SocketServiceUtils.RoomEvent.GET_STAGE_REQUESTS.id,
              payload = null
            )
          } catch (e: Exception) {
            logger.warn(
              "StageController::onRoomJoined::failed to get stage requests -> ${e.message}"
            )
            return@runBlocking
          }

        val getStageRequestsResponse =
          if (socketResponse != null) {
            try {
              GetStageRequestsResponse.ADAPTER.decode(socketResponse)
            } catch (e: Exception) {
              DyteLogger.warn(
                "StageController::onRoomJoined::failed to decode GetStageRequestsResponse"
              )
              GetStageRequestsResponse(emptyList())
            }
          } else {
            // currently null response means empty request list
            GetStageRequestsResponse(emptyList())
          }

        refreshAccessRequests(getStageRequestsResponse.stage_requests)

        if (_accessRequests.size > 0) {
          emitEvent { it.onStageRequestsUpdated(_accessRequests) }
        }
      }
    }
  }

  override suspend fun requestAccess() {
    if (!self.permissions.miscellaneous.stageEnabled) {
      logger.error("StageController::requestAccess::stage is disabled")
      return
    }

    val selfStageStatus = self.stageStatus // for thread-safety
    if (selfStageStatus !== DyteStageStatus.OFF_STAGE) {
      logger.warn("StageController::requestAccess::stage status is already $selfStageStatus")
      return
    }

    // beyond this Self is OFF_STAGE
    if (self.permissions.miscellaneous.stageAccess == DyteMediaPermission.NOT_ALLOWED) {
      logger.warn("StageController::requestAccess::self not allowed to send stage request")
      return
    }

    if (self.permissions.miscellaneous.stageAccess === DyteMediaPermission.ALLOWED) {
      updateSelfStageStatusAndEmit(DyteStageStatus.ACCEPTED_TO_JOIN_STAGE)
      return
    }

    try {
      socketService.send(SocketServiceUtils.RoomEvent.REQUEST_STAGE_ACCESS.id, null)
      updateSelfStageStatusAndEmit(DyteStageStatus.REQUESTED_TO_JOIN_STAGE)
    } catch (e: Exception) {
      logger.error("StageController::requestAccess::socket request failed ${e.message}")
    }
  }

  // TODO: add permission checks
  override suspend fun denyAccess(id: String) {
    val participantToDenyAccess =
      _accessRequests.find { it.id == id }
        ?: kotlin.run {
          logger.error("StageController::denyAccess::stage request for peerId $id not found")
          return
        }
    val userIdToDenyAccess = participantToDenyAccess.userId
    try {
      socketService.send(
        SocketServiceUtils.RoomEvent.DENY_STAGE_REQUEST.id,
        DenyStageAccessRequest(listOf(userIdToDenyAccess)).encode()
      )
    } catch (e: Exception) {
      logger.error("StageController::denyAccess::socket request failed ${e.message}")
    }
  }

  // TODO: add permission checks
  override suspend fun denyAccessAll() {
    val userIds = _accessRequests.map { it.userId }
    try {
      socketService.send(
        SocketServiceUtils.RoomEvent.DENY_STAGE_REQUEST.id,
        DenyStageAccessRequest(userIds).encode()
      )
    } catch (e: Exception) {
      logger.error("StageController::denyAccessAll::socket request failed ${e.message}")
    }
  }

  // TODO: add permission checks
  override suspend fun grantAccess(id: String) {
    val participantToGrantAccess =
      _accessRequests.find { it.id == id }
        ?: kotlin.run {
          logger.error("StageController::grantAccess::stage request for peerId $id not found")
          return
        }
    val userIdToGrantAccess = participantToGrantAccess.userId
    try {
      socketService.send(
        SocketServiceUtils.RoomEvent.GRANT_STAGE_REQUEST.id,
        GrantStageAccessRequest(listOf(userIdToGrantAccess)).encode()
      )
    } catch (e: Exception) {
      logger.error("StageController::grantAccess::socket request failed ${e.message}")
    }
  }

  // TODO: add permission checks
  override suspend fun grantAccessAll() {
    val userIds = _accessRequests.map { it.userId }
    try {
      socketService.send(
        SocketServiceUtils.RoomEvent.GRANT_STAGE_REQUEST.id,
        GrantStageAccessRequest(userIds).encode()
      )
    } catch (e: Exception) {
      logger.error("StageController::grantAccessAll::socket request failed ${e.message}")
    }
  }

  override suspend fun join() {
    if (self.stageStatus == DyteStageStatus.ON_STAGE) {
      logger.warn("StageController::join::self already ON_STAGE")
      return
    }

    if (
      self.permissions.miscellaneous.stageAccess != DyteMediaPermission.ALLOWED &&
        self.stageStatus != DyteStageStatus.ACCEPTED_TO_JOIN_STAGE
    ) {
      logger.warn("StageController::join::self not allowed to join stage")
      return
    }

    try {
      socketService.send(SocketServiceUtils.RoomEvent.JOIN_STAGE.id, null)
      internalEventsEmitter.emitEvent { it.connectToRoomNode() }
      // internalEventsEmitter.emitEvent { it.connectMedia() }
      updateSelfStageStatusAndEmit(DyteStageStatus.ON_STAGE)
    } catch (e: Exception) {
      logger.error("StageController::join::socket request failed ${e.message}")
    }
  }

  override suspend fun leave() {
    if (
      !(self.stageStatus == DyteStageStatus.ON_STAGE ||
        self.stageStatus == DyteStageStatus.ACCEPTED_TO_JOIN_STAGE)
    ) {
      logger.warn("StageController::leave::self is currently ${self.stageStatus}")
      return
    }

    internalEventsEmitter.emitEvent { it.disconnectFromRoomNode() }
    val request = LeaveStageRequest(listOf(self.userId)).encode()
    try {
      socketService.send(SocketServiceUtils.RoomEvent.LEAVE_STAGE.id, request)
      updateSelfStageStatusAndEmit(DyteStageStatus.OFF_STAGE)
    } catch (e: Exception) {
      logger.error("StageController::leave::socket request failed ${e.message}")
    }
  }

  // TODO: add acceptStageRequests permission check
  override suspend fun kick(id: String) {
    if (!self.permissions.host.canAcceptRequests) {
      logger.warn("StageController::kick::self not allowed to kick a participant")
      return
    }

    val participantToKick =
      participantController.joinedParticipants.find { it.id == id }
        ?: kotlin.run {
          logger.error("StageController::kick::participant $id not found")
          return
        }

    if (
      !(participantToKick.stageStatus == DyteStageStatus.ON_STAGE ||
        participantToKick.stageStatus == DyteStageStatus.ACCEPTED_TO_JOIN_STAGE)
    ) {
      logger.error(
        "StageController::kick::participant $id is currently ${participantToKick.stageStatus}"
      )
      return
    }

    try {
      val leaveStageRequest = LeaveStageRequest(listOf(participantToKick.userId)).encode()
      socketService.send(SocketServiceUtils.RoomEvent.LEAVE_STAGE.id, leaveStageRequest)
    } catch (e: Exception) {
      logger.error("StageController::grantAccess::socket request failed ${e.message}")
    }
  }

  override suspend fun cancelRequestAccess() {
    if (!self.permissions.miscellaneous.stageEnabled) {
      logger.error("StageController::cancelRequestAccess::stage is disabled")
      return
    }

    if (self.permissions.miscellaneous.stageAccess != DyteMediaPermission.CAN_REQUEST) {
      logger.warn("StageController::cancelRequestAccess::self cannot send or cancel stage request")
      return
    }

    // beyond this Self has CAN_REQUEST permission
    val selfStageStatus = self.stageStatus // for thread-safety
    if (selfStageStatus !== DyteStageStatus.REQUESTED_TO_JOIN_STAGE) {
      logger.warn(
        "StageController::cancelRequestAccess::stage status is not REQUESTED_TO_JOIN_STAGE"
      )
      return
    }

    // Self stageStatus is REQUESTED_TO_JOIN_STAGE
    try {
      socketService.send(SocketServiceUtils.RoomEvent.CANCEL_STAGE_REQUEST.id, null)
      updateSelfStageStatusAndEmit(DyteStageStatus.OFF_STAGE)
    } catch (e: Exception) {
      logger.error("StageController::cancelRequestAccess::socket request failed ${e.message}")
    }
  }

  private fun handleStageSocketEvent(event: Int, payload: ByteArray?) {
    when (event) {
      SocketServiceUtils.RoomEvent.PEER_STAGE_STATUS_UPDATE.id -> {
        if (payload == null) {
          return
        }

        val peerStageStatusUpdate =
          try {
            PeerStatusUpdate.ADAPTER.decode(payload)
          } catch (e: Exception) {
            DyteLogger.warn(
              "StageController::handle PEER_STAGE_STATUS_UPDATE socket event::failed to decode PeerStatusUpdate"
            )
            return
          }

        val updatedStageStatus =
          peerStageStatusUpdate.stage_type?.toDyteStageStatus()
            ?: kotlin.run {
              logger.warn(
                "StageController::handle PEER_STAGE_STATUS_UPDATE socket event::stage status received is null"
              )
              return
            }

        /*
         * note: Temporarily handling self removed from stage scenario using PEER_STAGE_STATUS_UPDATE.
         * In coming days, core will receive LEAVE_STAGE for this scenario.
         * */
        if (peerStageStatusUpdate.peer_id == self.id) {
          handleSelfRemovedFromStage(updatedStageStatus)
          return
        }

        val updatePeerStageStatusResult =
          participantController.updatePeerStageStatus(
            peerStageStatusUpdate.peer_id,
            updatedStageStatus
          )
            ?: return

        val oldStageStatus = updatePeerStageStatusResult.oldStageStatus

        // started presenting
        // should we also check for invalid transition like OFF_STAGE to ON_STAGE?
        if (updatedStageStatus == DyteStageStatus.ON_STAGE) {
          emitEvent { it.onParticipantStartedPresenting(updatePeerStageStatusResult.updatedPeer) }
          return
        }

        // stopped presenting
        // should we also check for invalid transition like ON_STAGE to REQUESTED?
        if (oldStageStatus == DyteStageStatus.ON_STAGE) {
          /*
           * TODO: Check if there is any difference in resultant stageStatus is an ACCEPTED participant
           *  leaves the stage vs they are removed from stage.
           * */
          emitEvent { it.onParticipantStoppedPresenting(updatePeerStageStatusResult.updatedPeer) }
          return
        }

        // requested to join stage (emit only if self is host)
        if (
          oldStageStatus == DyteStageStatus.OFF_STAGE &&
            updatedStageStatus == DyteStageStatus.REQUESTED_TO_JOIN_STAGE &&
            self.permissions.host.canAcceptRequests
        ) {
          emitEvent { it.onPresentRequestAdded(updatePeerStageStatusResult.updatedPeer) }
          return
        }

        /*
         * currently no support for emitting following peer events:
         * - accepted to join stage (REQUESTED -> ACCEPTED)
         * - invited to join stage (OFF_STAGE -> ACCEPTED)
         * - rejected to join stage (REQUESTED -> OFF_STAGE)
         * - request withdrawn
         * */

        // should we emit DyteParticipantEventsListener.onUpdate?
      }
      /** triggered whenever there is any change in stage access requests */
      SocketServiceUtils.RoomEvent.GET_STAGE_REQUESTS.id -> {
        if (!self.permissions.host.canAcceptRequests) {
          return
        }

        val getStageRequestsResponse =
          if (payload != null) {
            try {
              GetStageRequestsResponse.ADAPTER.decode(payload)
            } catch (e: Exception) {
              DyteLogger.warn(
                "StageController::handle GET_STAGE_REQUESTS socket event::failed to decode GetStageRequestsResponse"
              )
              GetStageRequestsResponse(emptyList())
            }
          } else {
            // currently null response means empty request list
            GetStageRequestsResponse(emptyList())
          }

        refreshAccessRequests(getStageRequestsResponse.stage_requests)

        emitEvent { it.onStageRequestsUpdated(_accessRequests) }
      }
      SocketServiceUtils.RoomEvent.GET_STAGE_PEERS.id -> {
        val getStagePeersResponse =
          if (payload != null) {
            try {
              GetStagePeersResponse.ADAPTER.decode(payload)
            } catch (e: Exception) {
              DyteLogger.warn(
                "StageController::handle GET_STAGE_PEERS socket event::failed to decode GetStagePeersResponse"
              )
              GetStagePeersResponse()
            }
          } else {
            // currently null response means empty request list
            GetStagePeersResponse()
          }

        println("DyteWeb: GET_STAGE_PEERS -> $getStagePeersResponse")
      }
      SocketServiceUtils.RoomEvent.GRANT_STAGE_REQUEST.id -> {
        // ignore event if self is host (presenter)
        if (self.permissions.miscellaneous.stageAccess == DyteMediaPermission.ALLOWED) {
          return
        }
        updateSelfStageStatusAndEmit(DyteStageStatus.ACCEPTED_TO_JOIN_STAGE)
        emitEvent { it.onPresentRequestReceived() }
      }
      SocketServiceUtils.RoomEvent.DENY_STAGE_REQUEST.id -> {
        // ignore event if self is host (presenter)
        if (self.permissions.miscellaneous.stageAccess == DyteMediaPermission.ALLOWED) {
          return
        }
        updateSelfStageStatusAndEmit(DyteStageStatus.OFF_STAGE)
      }
      else -> {}
    }
  }

  private fun updateSelfStageStatusAndEmit(stageStatus: DyteStageStatus) {
    if (self.stageStatus == stageStatus) {
      return
    }

    self._stageStatus = stageStatus
    emitEvent { it.onStageStatusUpdated(self.stageStatus) }
    selfController.emitEvent { it.onStageStatusUpdated(self.stageStatus) }
  }

  private fun handleSelfRemovedFromStage(updatedStageStatus: DyteStageStatus) {
    val selfCurrentStageStatus = self.stageStatus
    if (updatedStageStatus == selfCurrentStageStatus) {
      return
    }

    if (
      selfCurrentStageStatus == DyteStageStatus.ON_STAGE &&
        updatedStageStatus == DyteStageStatus.OFF_STAGE
    ) {
      internalEventsEmitter.emitEvent { it.disconnectFromRoomNode() }
      // internalEventsEmitter.emitEvent { it.disconnectMedia() }
      updateSelfStageStatusAndEmit(updatedStageStatus)
    }
  }

  private fun refreshAccessRequests(updatedStageRequests: List<StageRequest>) {
    _accessRequests.clear()
    for (stageRequest in updatedStageRequests) {
      val requestingPeer =
        participantController.joinedParticipants.find { it.id == stageRequest.peer_id } ?: continue
      _accessRequests.add(requestingPeer)
    }
  }

  private fun calculateStageRequestsDiff(
    existingStageRequests: List<DyteJoinedMeetingParticipant>,
    socketStageRequests: List<StageRequest>
  ): StageRequestsDiff {
    val existingRequestsMapByPeerId: Map<String, DyteJoinedMeetingParticipant> =
      existingStageRequests.associateBy { it.id }

    val updatedRequestsMapByPeerId: Map<String, StageRequest> =
      socketStageRequests.associateBy { it.peer_id }

    // Determine peerIds of cancelled requests
    val cancelledRequestPeerIds = mutableListOf<String>()
    for (participant in existingRequestsMapByPeerId.values) {
      if (!updatedRequestsMapByPeerId.containsKey(participant.id)) {
        cancelledRequestPeerIds.add(participant.id)
      }
    }

    // Determine peerIds of newly added requests
    val addedRequestPeerIds = mutableListOf<String>()
    for (stageRequest in updatedRequestsMapByPeerId.values) {
      if (!existingRequestsMapByPeerId.containsKey(stageRequest.peer_id)) {
        addedRequestPeerIds.add(stageRequest.peer_id)
      }
    }

    return StageRequestsDiff(addedRequestPeerIds, cancelledRequestPeerIds)
  }

  companion object {
    private fun StageType.toDyteStageStatus(): DyteStageStatus {
      return when (this) {
        StageType.STAGE_TYPE_ON_STAGE -> DyteStageStatus.ON_STAGE
        StageType.STAGE_TYPE_APPROVED_STAGE -> DyteStageStatus.ACCEPTED_TO_JOIN_STAGE
        StageType.STAGE_TYPE_REQUESTED_STAGE -> DyteStageStatus.REQUESTED_TO_JOIN_STAGE
        else -> DyteStageStatus.OFF_STAGE
      }
    }
  }

  /** Represent difference between local and received stage requests. */
  private data class StageRequestsDiff(
    val addedRequestPeerIds: List<String>,
    val removedRequestPeerIds: List<String>
  )
}
