package io.dyte.core.platform

import android.Manifest
import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.content.pm.PackageManager
import android.os.Build.VERSION
import android.os.Build.VERSION_CODES
import android.os.CountDownTimer
import androidx.annotation.ChecksSdkIntAtLeast
import androidx.annotation.RequiresApi
import androidx.core.content.ContextCompat
import io.dyte.core.controllers.Permission
import io.dyte.core.controllers.PermissionType
import io.dyte.core.controllers.PermissionType.CAMERA
import io.dyte.core.controllers.PermissionType.MICROPHONE
import kotlinx.coroutines.DelicateCoroutinesApi
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch

class DyteAndroidPlatformPermissions(private val activity: Activity) :
  IDytePlatformPermissionUtils {
  @RequiresApi(VERSION_CODES.M)
  override fun arePermissionsGranted(permissions: List<Permission>): Boolean {
    val grantedPermissions = arrayListOf<Permission>()
    val deniedPermissions = arrayListOf<Permission>()

    permissions.forEach { permission ->
      if (isPermissionGrantedInternal(activity, permission)) {
        grantedPermissions.add(permission)
      } else {
        deniedPermissions.add(permission)
      }
    }
    return deniedPermissions.isEmpty()
  }

  @OptIn(DelicateCoroutinesApi::class)
  @RequiresApi(VERSION_CODES.M)
  override fun askPermissions(
    permissions: List<Permission>,
    onPermissionsGranted: () -> Unit,
    onPermissionsDenied: () -> Unit
  ): Boolean {
    if (VERSION.SDK_INT < VERSION_CODES.M) {
      val deniedPermissions = arrayListOf<Permission>()
      permissions.forEach { permission ->
        if (!isPermissionGrantedInternal(activity, permission)) {
          deniedPermissions.add(permission)
        }
      }
      if (deniedPermissions.isNotEmpty()) {
        onPermissionsDenied()
      } else {
        onPermissionsGranted()
      }
    } else {
      val platformPermissions = permissions.map { getPlatformPermission(it) }
      activity.requestPermissions(platformPermissions.toTypedArray(), 111)
      GlobalScope.launch(Dispatchers.Main) {
        pollForPermissions(permissions, onPermissionsGranted, onPermissionsDenied)
      }
    }
    return true
  }

  @SuppressLint("NewApi")
  private fun pollForPermissions(
    permissions: List<Permission>,
    onPermissionGranted: () -> Unit,
    onPermissionsDenied: () -> Unit
  ) {
    // timer for 30 seconds, ticked every 1 second
    val timer =
      object : CountDownTimer(30000, 1000) {
        override fun onTick(millisUntilFinished: Long) {
          val deniedPermissions = arrayListOf<Permission>()
          val grantedPermissions = arrayListOf<Permission>()

          permissions.forEach {
            if (isPermissionGranted(it)) {
              grantedPermissions.add(it)
            } else {
              deniedPermissions.add(it)
            }
          }

          if (deniedPermissions.isEmpty()) {
            onPermissionGranted.invoke()
            cancel()
          }
        }

        override fun onFinish() {
          val deniedPermissions = arrayListOf<Permission>()
          val grantedPermissions = arrayListOf<Permission>()

          permissions.forEach {
            if (isPermissionGranted(it)) {
              grantedPermissions.add(it)
            } else {
              deniedPermissions.add(it)
            }
          }

          if (deniedPermissions.isEmpty()) {
            onPermissionGranted.invoke()
          } else {
            onPermissionsDenied.invoke()
          }
        }
      }
    timer.start()
  }

  @RequiresApi(VERSION_CODES.M)
  override fun isPermissionGranted(permission: Permission): Boolean {
    return isPermissionGrantedInternal(activity, permission)
    // return activity.checkSelfPermission(getPlatformPermission(permission)) ==
    // PackageManager.PERMISSION_GRANTED
  }

  @ChecksSdkIntAtLeast(api = VERSION_CODES.S)
  override fun shouldRequestBluetoothPermission(): Boolean {
    return VERSION.SDK_INT >= VERSION_CODES.S
  }

  private fun getPlatformPermission(permission: Permission): String {
    return when (permission.type) {
      CAMERA -> {
        Manifest.permission.CAMERA
      }
      MICROPHONE -> {
        Manifest.permission.RECORD_AUDIO
      }
      PermissionType.STORAGE -> {
        if (VERSION.SDK_INT >= VERSION_CODES.R) {
          Manifest.permission.MANAGE_EXTERNAL_STORAGE
        } else {
          Manifest.permission.WRITE_EXTERNAL_STORAGE
        }
      }
      PermissionType.BLUETOOTH -> {
        if (VERSION.SDK_INT >= VERSION_CODES.S) {
          Manifest.permission.BLUETOOTH_CONNECT
        } else {
          Manifest.permission.BLUETOOTH
        }
      }
    }
  }

  private fun isPermissionGrantedInternal(context: Context, permission: Permission): Boolean {
    return ContextCompat.checkSelfPermission(context, getPlatformPermission(permission)) ==
      PackageManager.PERMISSION_GRANTED
  }
}
