package io.dyte.core.media

import android.Manifest
import android.content.pm.PackageManager
import android.graphics.Bitmap
import android.graphics.Matrix
import android.opengl.GLES20
import android.opengl.GLUtils
import androidx.core.content.ContextCompat
import io.dyte.webrtc.ApplicationContextHolder
import io.dyte.webrtc.AudioStreamTrack
import io.dyte.webrtc.CameraPermissionException
import io.dyte.webrtc.CameraVideoCaptureController
import io.dyte.webrtc.MediaStream
import io.dyte.webrtc.MediaStreamConstraintsBuilder
import io.dyte.webrtc.RecordAudioPermissionException
import io.dyte.webrtc.VideoStreamTrack
import io.dyte.webrtc.WebRtc
import io.dyte.webrtc.toMandatoryMap
import io.dyte.webrtc.toOptionalMap
import io.dyte.webrtc.yuv.YuvFrame
import io.webrtc.MediaConstraints
import io.webrtc.SurfaceTextureHelper
import io.webrtc.TextureBufferImpl
import io.webrtc.VideoFrame
import io.webrtc.VideoProcessor
import io.webrtc.VideoSink
import io.webrtc.YuvConverter
import java.util.UUID

internal actual val userMedia: DyteUserMedia = DyteUserMediaImpl

private object DyteUserMediaImpl : DyteUserMedia {
  override suspend fun getUserMedia(
    streamConstraints: MediaStreamConstraintsBuilder.() -> Unit
  ): MediaStream {
    val constraints =
      MediaStreamConstraintsBuilder().let {
        streamConstraints(it)
        it.constraints
      }

    var audioTrack: AudioStreamTrack? = null
    if (constraints.audio != null) {
      checkRecordAudioPermission()
      val mediaConstraints =
        MediaConstraints().apply {
          mandatory.addAll(
            constraints.audio!!.toMandatoryMap().map { (k, v) ->
              MediaConstraints.KeyValuePair("$k", "$v")
            }
          )
          optional.addAll(
            constraints.audio!!.toOptionalMap().map { (k, v) ->
              MediaConstraints.KeyValuePair("$k", "$v")
            }
          )
        }
      val audioSource = WebRtc.peerConnectionFactory.createAudioSource(mediaConstraints)
      val androidTrack =
        WebRtc.peerConnectionFactory.createAudioTrack(UUID.randomUUID().toString(), audioSource)
      audioTrack = AudioStreamTrack(androidTrack, audioSource)
    }

    var videoTrack: VideoStreamTrack? = null
    var newFrame: VideoFrame? = null

    if (constraints.video != null) {
      checkCameraPermission()
      val videoSource = WebRtc.peerConnectionFactory.createVideoSource(false)

      val videoCaptureController = CameraVideoCaptureController(constraints.video!!, videoSource)

      videoSource.setVideoProcessor(
        object : VideoProcessor {
          private var _sink: VideoSink? = null

          override fun onCapturerStarted(p0: Boolean) {}

          override fun onCapturerStopped() {}

          override fun onFrameCaptured(p0: VideoFrame?) {
            var skipFrame = false
            val textureHelper = videoCaptureController.textureHelper!!

            if (DyteVideoMiddlewares.getVideoMiddlewares().isNotEmpty()) {
              val bitmap = YuvFrame(p0, YuvFrame.PROCESSING_NONE, p0!!.timestampNs).bitmap!!

              DyteVideoFrame.FRAME_HEIGHT = bitmap.height
              DyteVideoFrame.FRAME_WIDTH = bitmap.width

              var middlewareFrame: DyteVideoFrame? = DyteVideoFrame.fromBitmap(bitmap)

              run breaking@{
                DyteVideoMiddlewares.getVideoMiddlewares().forEach { mw ->
                  middlewareFrame = mw.onVideoFrame(middlewareFrame!!)
                  if (middlewareFrame == null) {
                    return@breaking
                  }
                }
              }

              textureHelper.handler.post {
                if (middlewareFrame == null) {
                  skipFrame = true
                } else {
                  newFrame =
                    bitmapToVideoFrame(middlewareFrame!!.toBitmap(), p0.timestampNs, textureHelper)
                }
                if (!skipFrame) _sink?.onFrame(newFrame)
              }
            } else {
              newFrame = p0
              if (!skipFrame) _sink?.onFrame(newFrame)
            }
          }

          override fun setSink(p0: VideoSink?) {
            _sink = p0
          }
        }
      )

      val androidTrack =
        WebRtc.peerConnectionFactory.createVideoTrack(UUID.randomUUID().toString(), videoSource)
      videoTrack = VideoStreamTrack(androidTrack, videoCaptureController)
    }

    val localMediaStream =
      WebRtc.peerConnectionFactory.createLocalMediaStream(UUID.randomUUID().toString())
    return MediaStream(localMediaStream).apply {
      if (audioTrack != null) addTrack(audioTrack)
      if (videoTrack != null) addTrack(videoTrack)
    }
  }

  private fun checkCameraPermission() {
    val result =
      ContextCompat.checkSelfPermission(
        ApplicationContextHolder.context,
        Manifest.permission.CAMERA
      )
    if (result == PackageManager.PERMISSION_DENIED) throw CameraPermissionException()
  }

  private fun checkRecordAudioPermission() {
    val result =
      ContextCompat.checkSelfPermission(
        ApplicationContextHolder.context,
        Manifest.permission.RECORD_AUDIO
      )
    if (result == PackageManager.PERMISSION_DENIED) throw RecordAudioPermissionException()
  }

  private fun bitmapToVideoFrame(
    bitmap: Bitmap,
    timestampNs: Long,
    textureHelper: SurfaceTextureHelper
  ): VideoFrame {
    val converter = YuvConverter()

    val textures = IntArray(1)
    GLES20.glGenTextures(0, textures, 0)

    val matrix = Matrix()
    matrix.setScale(-1f, 1f)

    val textureBuffer =
      TextureBufferImpl(
        bitmap.width,
        bitmap.height,
        VideoFrame.TextureBuffer.Type.RGB,
        textures[0],
        matrix,
        textureHelper.handler,
        converter,
        null
      )

    GLES20.glTexParameteri(GLES20.GL_TEXTURE_2D, GLES20.GL_TEXTURE_MIN_FILTER, GLES20.GL_NEAREST)
    GLES20.glTexParameteri(GLES20.GL_TEXTURE_2D, GLES20.GL_TEXTURE_MAG_FILTER, GLES20.GL_NEAREST)
    GLUtils.texImage2D(GLES20.GL_TEXTURE_2D, 0, bitmap, 0)

    val opBuffer = textureBuffer.toI420()

    return VideoFrame(opBuffer, 180, timestampNs)
  }
}
