package io.dyte.core.hive

import io.dyte.core.observability.DyteLogger
import io.dyte.core.socket.socketservice.ISockratesSocketService
import io.dyte.core.socket.socketservice.SocketServiceUtils
import media.CreateTransportRequest
import media.CreateTransportResponse
import media.RenegotiateRequest
import media.edge.ConsumePeerRequest
import media.edge.ConsumePeerResponse
import media.edge.ConsumerCloseRequest
import media.edge.ConsumerToggleRequest
import media.edge.PeerLeaveRequest
import media.edge.PeerLeaveResponse
import media.edge.ProducerCloseRequest
import media.edge.ProducerClosingResponse
import media.edge.ProducerCreateRequest
import media.edge.ProducerCreateResponse
import media.edge.ProducerToggleRequest

internal interface HiveSFUSocketHandler {
  suspend fun renegotitateSessionDescription(renegotiateRequest: RenegotiateRequest)

  suspend fun createTransport(
    createTransportRequest: CreateTransportRequest
  ): CreateTransportResponse?

  suspend fun produce(producerCreateRequest: ProducerCreateRequest): ProducerCreateResponse?

  suspend fun closeProducer(producerCloseRequest: ProducerCloseRequest): ProducerClosingResponse?

  suspend fun toggleProducer(producerToggleRequest: ProducerToggleRequest)

  suspend fun leaveRoom(peerLeaveRequest: PeerLeaveRequest): PeerLeaveResponse?

  suspend fun consume(consumePeerRequest: ConsumePeerRequest): ConsumePeerResponse?

  suspend fun toggleConsumer(consumerToggleRequest: ConsumerToggleRequest)

  suspend fun closeConsumer(consumerCloseRequest: ConsumerCloseRequest)
}

internal class DefaultHiveSFUSocketHandler(private val socketService: ISockratesSocketService) :
  HiveSFUSocketHandler {
  override suspend fun renegotitateSessionDescription(renegotiateRequest: RenegotiateRequest) {
    try {
      socketService.send(
        event = SocketServiceUtils.MediaEvent.RENEGOTIATE_SESSION_DESCRIPTION.id,
        payload = RenegotiateRequest.ADAPTER.encode(renegotiateRequest)
      )
    } catch (e: Exception) {
      DyteLogger.error("HiveSocketHandler: Error on renegotitate session description: ${e.message}")
    }
  }

  override suspend fun createTransport(
    createTransportRequest: CreateTransportRequest
  ): CreateTransportResponse? {
    return try {
      CreateTransportResponse.ADAPTER.decode(
        socketService.requestResponse(
          event = SocketServiceUtils.MediaEvent.CREATE_WEBRTC_TRANSPORT.id,
          payload = CreateTransportRequest.ADAPTER.encode(createTransportRequest)
        )!!
      )
    } catch (e: Exception) {
      DyteLogger.error("HiveSocketHandler: Error on create transport: ${e.message}")
      null
    }
  }

  override suspend fun produce(
    producerCreateRequest: ProducerCreateRequest
  ): ProducerCreateResponse? {
    return try {
      ProducerCreateResponse.ADAPTER.decode(
        socketService.requestResponse(
          event = SocketServiceUtils.MediaEvent.PRODUCE.id,
          payload = ProducerCreateRequest.ADAPTER.encode(producerCreateRequest)
        )!!
      )
    } catch (e: Exception) {
      DyteLogger.error("HiveSocketHandler: Error on produce: ${e.message}")
      null
    }
  }

  override suspend fun closeProducer(
    producerCloseRequest: ProducerCloseRequest
  ): ProducerClosingResponse? {
    return try {
      ProducerClosingResponse.ADAPTER.decode(
        socketService.requestResponse(
          event = SocketServiceUtils.MediaEvent.CLOSE_PRODUCER.id,
          payload = ProducerCloseRequest.ADAPTER.encode(producerCloseRequest)
        )!!
      )
    } catch (e: Exception) {
      DyteLogger.error("HiveSocketHandler: Error on close producer: ${e.message}")
      null
    }
  }

  override suspend fun toggleProducer(producerToggleRequest: ProducerToggleRequest) {
    try {
      socketService.send(
        event = SocketServiceUtils.MediaEvent.TOGGLE_PRODUCER.id,
        payload = ProducerToggleRequest.ADAPTER.encode(producerToggleRequest)
      )
    } catch (e: Exception) {
      DyteLogger.error("HiveSocketHandler: Error on toggle producer: ${e.message}")
    }
  }

  override suspend fun leaveRoom(peerLeaveRequest: PeerLeaveRequest): PeerLeaveResponse? {
    return try {
      PeerLeaveResponse.ADAPTER.decode(
        socketService.requestResponse(
          event = SocketServiceUtils.MediaEvent.LEAVE_ROOM.id,
          payload = PeerLeaveRequest.ADAPTER.encode(peerLeaveRequest)
        )!!
      )
    } catch (e: Exception) {
      DyteLogger.error("HiveSocketHandler: Error on leave room: ${e.message}")
      null
    }
  }

  override suspend fun consume(consumePeerRequest: ConsumePeerRequest): ConsumePeerResponse? {
    return try {
      ConsumePeerResponse.ADAPTER.decode(
        socketService.requestResponse(
          event = SocketServiceUtils.MediaEvent.CONSUME.id,
          payload = ConsumePeerRequest.ADAPTER.encode(consumePeerRequest)
        )!!
      )
    } catch (e: Exception) {
      DyteLogger.error("HiveSocketHandler: Error on consume: ${e.message}")
      null
    }
  }

  override suspend fun toggleConsumer(consumerToggleRequest: ConsumerToggleRequest) {
    try {
      socketService.send(
        event = SocketServiceUtils.MediaEvent.TOGGLE_CONSUMER.id,
        payload = ConsumerToggleRequest.ADAPTER.encode(consumerToggleRequest)
      )
    } catch (e: Exception) {
      DyteLogger.error("HiveSocketHandler: Error on toggle consumer: ${e.message}")
    }
  }

  override suspend fun closeConsumer(consumerCloseRequest: ConsumerCloseRequest) {
    try {
      socketService.send(
        event = SocketServiceUtils.MediaEvent.CLOSE_CONSUMER.id,
        payload = ConsumerCloseRequest.ADAPTER.encode(consumerCloseRequest)
      )
    } catch (e: Exception) {
      DyteLogger.error("HiveSocketHandler: Error on close consumer: ${e.message}")
    }
  }
}
